{*****************************************************************************}
{ [TCDEVENTS]                                                                 }
{   Special Units = DBTMsg                                                    }
{   Version = 1.21                                                            }
{ [SOURCE]                                                                    }
{   Copyright  1998 by Tom Deprez                                            }
{   (Just to prevent to get multiple components, which all have great         }
{    things but would be a GREAT component with a lot (all) of GREAT things   }
{    That way everybody can benefit of such a GREAT component)                }
{ ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ }
{ [COMPONENT]                                                                 }
{ [properties]                                                                }
{   AutoRun : Disable/enable CD autorun feature                               }
{ [events]                                                                    }
{   AfterArrival : Fires when new CD is inserted by user                      }
{   AfterRemove  : Fires when CD is removed by user                           }
{                                                                             }
{ ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ }
{ [AUTHOR]                                                                    }
{   Author name   = ZifNab (Tom Deprez)                                       }
{   Author e-mail = tom.deprez@uz.kuleuven.ac.be                              }
{ ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ }
{ [HISTORY]                                                                   }
{   18/01/1998 : first release                                                }
{   04/02/1998 : changed the order of checking the message (WMDeviceChange)   }
{   21/02/1998 : added disable/enable CD autorun feature                      }
{                added eject/close CD-door                                    }
{   12/03/1998 : fixed a bug (thanks to Jaco Vis)                             }
{                                                                             }
{*****************************************************************************}

{*****************************************************************************}
{  [COMMENTS]                                                                 }
{   I very often visit the Delphi page of the www.Experts-Exchange.com site   }
{   It's a place where you can ask or answer questions. One day somebody      }
{   needed to know when a CD is inserted in or ejected out of the CD-Rom      }
{   drive. I found it an interesting question, especially because I could     }
{   not find some 'delphi' source for this problem.                           }
{   I started to search for a solution.                                       }
{   Well, that was the beginning of this component. Where it ends?            }
{   That I don't know, it really depends on the reaction of you! If I get     }
{   some reaction, I'll futher improve it, otherwise it sticks with this.     }
{   Too bad, because I think it can become a great tool.                      }
{                                                                             }
{  [IMPORTANT]                                                                }
{    Made some improvements? : please let me know of, so I can update it      }
{    Need some improvements? : please, just ask and I'll try to make it       }
{    Have you got some ideas? : please, send them to me                       }
{    You use this component? : please, send me an e-mail, why you use this    }
{                              component. An E-mail isn't asked too much.     }
{                              Isn't it? At least let me know of the fact     }
{                              you're using it. It makes me very HAPPY!       }
{  [THANKS]                                                                   }
{    Matvey who brought the first idea to this component                      }
{    Mark                                                                     }
{    Perhaps YOU?                                                             }
{                                                                             }
{ ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ }
{  [COPYRIGHT]                                                                }
{    This file is distributed as freeware and without warranties of any       }
{    kind. You can use it in your own applications at your own risk.          }
{                                                                             }
{  [NOTE]                                                                     }
{    It's freeware, but don't hesitate to send me some money if you've        }
{    become rich with the help of this component ;-)                          }
{    It's freeware, but don't hesitate to send me the program or the          }
{    component, where you use this component in. I would be ever gratefull    }
{    to you, thanks.                                                          }
{*****************************************************************************}

unit CDEvents;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DBTMsg, DsgnIntf, Registry, MMSystem;

const cVersion = '1.2';

type
 EMCIDeviceError = class(Exception);

 TDeviceChangeEvent = procedure (Sender : TObject; FirstDriveLetter : char)
                         of object;

 TAboutZifNabProperty = class(TPropertyEditor)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    function GetValue: string; override;
  end;

 TCDOptions = class(TPersistent)
  private
    fOldAutoRun : Boolean;
    fAutoReset : Boolean;
  protected
    procedure SetCDAutoRun(vAutoRun : Boolean);
    function GetCDAutoRun : Boolean;
  public
    constructor Create;
    destructor Destroy; override;
    procedure ResetCDOptions;
  published
    property AutoReset: Boolean read fAutoReset write fAutoReset default true;
    property AutoRun: Boolean read GetCDAutoRun write SetCDAutoRun;
 end;

 TCDEvents = class(TComponent)
  private
    { Private declarations }
     FAbout : TAboutZifNabProperty;
     FFlags : Longint;
     FID : Word;
     fErrCode : longint;
     FWindowHandle: HWND;
     fOptions : TCDOptions;
     fAfterArrival: TDeviceChangeEvent;
     fAfterRemove: TDeviceChangeEvent;
     procedure WndProc(var Msg: TMessage);
  protected
    { Protected declarations }
     function GetFirstDriveLetter(unitmask:longint):char;
     procedure WMDeviceChange(var Msg : TWMDeviceChange); dynamic;
     procedure SetOptions(value : TCDOptions);
  public
    { Public declarations }
     constructor Create(AOwner: TComponent); override;
     destructor Destroy; override;
     procedure OpenDevice;
     procedure OpenDoor;
     procedure CloseDoor;
     procedure CloseDevice;
     property ID: Word read fID write fID;     
  published
    { Published declarations }
    property About : TAboutZifNabProperty read FAbout write FAbout;
    property AfterArrival : TDeviceChangeEvent read fAfterArrival write fAfterArrival;
    property AfterRemove: TDeviceChangeEvent read fAfterRemove write fAfterRemove;
    property Options : TCDOptions read fOptions write SetOptions;
  end;

implementation

{*********************** tAboutZifNabproperty component ***********************}

procedure TAboutZifNabProperty.Edit;
begin
  Application.MessageBox ('        tCDEvents v'+cVersion
                         +#13#10'This component is freeware.'
                         +#13#10'      1998 ZifNab         '
                         +#13#10''
                         +#13#10'mailto:Tom.Deprez@uz.kuleuven.ac.be',
                         'About',
                         MB_OK+ MB_ICONINFORMATION);
end;

function TAboutZifNabProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paMultiSelect, paDialog, paReadOnly];
end;

function TAboutZifNabProperty.GetValue: string;
begin
  Result := '(about)';
end;

{**************************** tCDOptions component ****************************}

constructor TCDOptions.Create;
begin
  fAutoReset := True;
  fOldAutoRun := GetCDAutoRun;
end;

destructor TCDOptions.Destroy;
begin
  if fAutoReset then ResetCDOptions;
  inherited Destroy;
end;

procedure TCDOptions.SetCDAutoRun(vAutoRun : Boolean);
 (* this code comes from Delphi Developer Support *)
var
 reg: TRegistry;
 AutoRunSetting : integer;
begin
 reg := TRegistry.Create;
 with reg do begin
  RootKey := HKEY_CURRENT_USER;
  LazyWrite := false;
  OpenKey('Software\Microsoft\Windows\CurrentVersion\Policies\Explorer',
          false);
  ReadBinaryData('NoDriveTypeAutoRun',
                 AutoRunSetting,
                 SizeOf(AutoRunSetting));
  if vAutoRun then
   AutoRunSetting := AutoRunSetting and not (1 shl 5)
  else
   AutoRunSetting := AutoRunSetting or (1 shl 5);
  reg.WriteBinaryData('NoDriveTypeAutoRun',
                      AutoRunSetting,
                      SizeOf(AutoRunSetting));
  CloseKey;
  Free;
 end;
end;

function TCDOptions.GetCDAutoRun : Boolean;
 (* this code comes from Delphi Developer Support *)
var reg : TRegistry;
    AutoRunSetting : integer;
begin
 reg := TRegistry.Create;
 with reg do begin
  Rootkey := HKEY_CURRENT_USER;
  OpenKey('Software\Microsoft\Windows\CurrentVersion\Policies\Explorer',
          false);
  ReadBinaryData('NoDriveTypeAutoRun',
                 AutoRunSetting,
                 SizeOf(AutoRunSetting));
  CloseKey;
  Free;
 end;
 Result := not ((AutoRunSetting and (1 shl 5)) <> 0);
end;

procedure TCDOptions.ResetCDOptions;
begin
 SetCDAutoRun(fOldAutoRun);
end;

{**************************** tCDEvents component *****************************}

constructor TCDEvents.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FWindowHandle := AllocateHWnd(WndProc);
  fOptions := TCDOptions.Create;
  fID := 0;
end;

destructor TCDEvents.Destroy;
begin
  fOptions.Free;
  DeallocateHWnd(FWindowHandle);
  inherited Destroy;
end;

procedure TCDEvents.SetOptions(value : TCDOptions);
begin
 fOptions.Assign(value);
end;

procedure TCDEvents.WndProc(var Msg: TMessage);
begin
     if (Msg.Msg = WM_DEVICECHANGE) then
      try
        WMDeviceChange(TWMDeviceChange(Msg));
      except
        Application.HandleException(Self);
      end
    else
      Msg.Result := DefWindowProc(FWindowHandle, Msg.Msg, Msg.wParam, Msg.lParam);
end;

function TCDEvents.GetFirstDriveLetter(unitmask : longint):char;
var DriveLetter : shortint;
begin
 DriveLetter := Ord('A');
 while (unitmask and 1)=0  do begin
  unitmask := unitmask shr 1;
  inc(DriveLetter);
 end;
 Result := Char(DriveLetter);
end;

procedure TCDEvents.WMDeviceChange(var Msg : TWMDeviceChange);
var lpdb : PDEV_BROADCAST_HDR;
    lpdbv : PDEV_BROADCAST_VOLUME;
    
 function IsCD : Boolean;
  begin
   Result := false;
   if lpdb^.dbch_devicetype = DBT_DEVTYP_VOLUME then begin
    lpdbv := PDEV_BROADCAST_VOLUME(Msg.dwData);
    Result := ((lpdbv^.dbcv_flags and DBTF_MEDIA) = 1);
   end;
  end;

begin
 (* received a wm_devicechange message *)
  lpdb := PDEV_BROADCAST_HDR(Msg.dwData);
  (* look at the event send together with the wm_devicechange message *)
   case Msg.Event of
    DBT_DEVICEARRIVAL : begin
        if IsCD and Assigned(fAfterArrival) then
         fAfterArrival(Self, GetFirstDriveLetter(lpdbv^.dbcv_unitmask));
     end;
    DBT_DEVICEREMOVECOMPLETE : begin
        if IsCD and Assigned(fAfterRemove) then
         fAfterRemove(Self, GetFirstDriveLetter(lpdbv^.dbcv_unitmask));
     end;
   end;
 end;

{~~~ MCI Commands ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~}

procedure TCDEvents.OpenDevice;
var OpenParms : TMCI_Open_Parms;
begin
 FFlags := 0;
 FFlags := mci_notify or mci_open_type or mci_open_shareable;
 OpenParms.lpstrDeviceType := 'CDAudio';
 OpenParms.dwCallback := 0;
 fErrCode := mciSendCommand(fID, mci_open, FFlags, Longint(@OpenParms));
 if FErrCode <> 0 then {problem opening device}
    raise EMCIDeviceError.Create('error')
  else {device successfully opened}
   begin
    fID := OpenParms.wDeviceID;
   end;
end;

procedure TCDEvents.OpenDoor;
var
  SetParms: TMCI_Set_Parms;
begin
 if fID <> 0 then begin
  FFlags := 0;
  FFlags := mci_notify or mci_set_door_open;
  SetParms.dwCallback := 0;
  fErrCode := mciSendCommand(fID, mci_Set, FFlags, Longint(@SetParms));
 end;
end;

procedure TCDEvents.CloseDoor;
var
  SetParm: TMCI_Set_Parms;
begin
 if fID <> 0 then begin
  FFlags := 0;
  FFlags := mci_notify or mci_set_door_closed;
  SetParm.dwCallback := 0;
  fErrCode := mciSendCommand( fID, mci_Set, FFlags, Longint(@SetParm) );
 end;
end;

procedure TCDEvents.CloseDevice;
var
  GenParms: TMCI_Generic_Parms;
begin
 if fID <> 0 then begin
  FFlags := 0;
  FFlags := mci_notify;
  FErrCode := mciSendCommand(FID, mci_Close, FFlags, Longint(@GenParms));
  if FErrCode = 0 then
    begin
      FID := 0;
    end;
 end;
end;

end.
