unit FourWinsControl;
(***********************************************)
(* The Game of Four Wins in a Box,             *)
(* including an intelligent Computer Player    *)
(************************************************)
(* Date:   22.05.2001                           *)
(* Author: Brian Schrder                       *)
(* eMail:  delphi@brian-schroeder.de            *)
(* www:    http://www.brian-schroeder.de        *)
(************************************************)
(* Description                                  *)
(* This Unit Contains a Component, that gives   *)
(* an User-Interface for the Game 4-Wins.       *)
(* The Goal of the Game is to get 4 Pieces in   *)
(* a row.                                       *)
(* The Game-Class includes an implementation of *)
(* the MinMax Algorithm with Alpha Beta Pruning.*)
(* It is strong enough to beat me in this Game. *)
(* For the Usage of the Components see the      *)
(* Example Programm.                            *)
(************************************************)
(* Legal Stuff                                  *)
(* The Code is provided as is, WITHOUT ANY      *)
(* WARRANTY OF ANY KIND.                        *)
(* Don't blame me in any case where the usage of*)
(* this code has done you harm.                 *)
(************************************************)
(* This Code may be used freely in any          *)
(* Application as long as my name and eMail     *)
(* are mentioned.                               *)
(************************************************)

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, Math;

const FRows = 8;
      FCols = 12;

const MinInt = Low(Integer);

type TPlayer       = -1..2; // -1 = Kein Player 0 = Player 0 1 = player 1 2 = Stopper
     TMinMaxResult = record
                       Value, Bestmove : Integer;
                     end;
type TFeld         = Array[1..FRows+1,1..FCols] of TPlayer;

type TInsertEvent = Procedure (Sender : TObject; Slot : Integer) of object;

type TStatus = (Playing, Finished);
type
  TFourWins = class(TObject)
  private
    FPercentDone : Double;
    FStatus : TStatus;
    FFeld   : TFeld;
    FPlayer,
    FMax     : TPlayer;
    FOnReset,
    FOnGameEnds,
    FOnChange,
    FOnBeginThink,
    FOnEndThink,
    FOnProgress : TNotifyEvent;
    FOnInsert  : TInsertEvent;


    function  GetRows   : Integer;
    function  GetCols   : Integer;
    function  GetWinner : TPlayer;

    function  MMInsertInto(Feld : TFeld; slot : Integer; Player : TPlayer)    : TFeld;
    function  MMMayInsertInto(Feld : TFeld; Slot : Integer) : Boolean;
    function  MMMinMax : TMinMaxResult;
    function  MMWhoWon(Feld : TFeld) : TPlayer;
    function  GMax(Feld : TFeld; Alpha, Beta, MyDepth : Integer) : TMinMaxResult;
    function  GMin(Feld : TFeld; Alpha, Beta, MyDepth : Integer) : TMinMaxResult;
  public
    function  MMGetValue(Feld : TFeld)        : Integer;
    function  InsertInto(slot : Integer; Player : TPlayer) : Boolean;
    function  MayInsertInto(slot : Integer) : Boolean;
    procedure Reset;
    property Feld   : TFeld    read FFeld;
    
    constructor Create;
    
    function GetValue : Integer;
    
    function ComputerMakeMove : Integer;
  published
    property Rows   : Integer   read getRows;
    property Cols   : Integer   read getCols;
    property Winner : TPlayer   read GetWinner;
    property Value  : Integer  read GetValue;
    property Player : TPlayer read FPlayer write FPlayer;
    property Status : TStatus read FStatus;
    property Max    : TPlayer read FMax write FMax;
    property PercentDone : double read FPercentDone; 

    property OnReset       : TNotifyEvent read FOnReset    write FOnReset;
    property OnGameEnds    : TNotifyEvent read FOnGameEnds   write FOnGameEnds;
    property OnChange      : TNotifyEvent read FOnChange     write FOnChange;
    property OnBeginThink  : TNotifyEvent read FOnBeginThink write FOnBeginThink;
    property OnEndThink    : TNotifyEvent read FOnEndThink   write FOnEndThink;
    property OnProgress    : TNotifyEvent read FOnProgress   write FOnProgress;
    property OnInsert      : TInsertEvent read FOnInsert     write FOnInsert;
  end;

type
    TActiveSlotType = 0..FCols;

type
  TFourWinsControl = class(TGraphicControl)
  private
    { Private-Deklarationen }
    FGame : TFourWins;
    FActiveSlot : TActiveSlotType;
    FOnGameEnds,
    FOnChange,
    FOnProgress : TNotifyEvent;
    FCPO : Boolean;
    procedure SetActiveSlot(ASlot : TActiveSlotType);
    function  GetWinner : String;
    function  GetRows   : Integer;
    function  GetCols   : Integer;
    function  getPercentDone : Double;

    procedure DoInsert(Sender : TObject; Slot : Integer);
    procedure DoChange(Sender : TObject);
    procedure DoGameEnds(Sender : TObject);
    procedure DoProgress(Sender : TObject);
  protected
    { Protected-Deklarationen }
    procedure Paint; override;
  public
    { Public-Deklarationen }
    procedure Reset;
    constructor Create(AOwner: TComponent); override;
    destructor destroy; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    function GetValue : Integer;

    function ComputerMakeMove : Integer;
  published
    { Published-Deklarationen }
    property Rows : Integer read GetRows;
    property Cols : Integer read GetCols;
    property Winner : String read GetWinner;
    property ActiveSlot : TActiveSlotType read FActiveSlot write SetActiveSlot;
    property ComputerPlayerOn : Boolean read FCPO write FCPO;
    property PercentDone : Double read getPercentDone;

    property Align;

    property OnGameEnds    : TNotifyEvent read FOnGameEnds   write FOnGameEnds;
    property OnProgress    : TNotifyEvent read FOnProgress   write FOnProgress;
    property OnChange      : TNotifyEvent read FOnChange     write FOnChange;
  end;

procedure Register;

implementation

constructor TFourWinsControl.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width  := 100;
  Height := 100;
  FGame  := TFourWins.Create;
  FGame.OnChange := DoChange;
  FGame.OnGameEnds := DoGameEnds;
  FGame.OnProgress := DoProgress;
  FGame.OnInsert := DoInsert;
  ControlStyle := [csCaptureMouse, csClickEvents, csSetCaption, csOpaque, csDoubleClicks];
end;

destructor TFourWinsControl.destroy;
begin
  FGame.Free;
  inherited Destroy;
end;

procedure Register;
begin
  RegisterComponents('BSSoftware', [TFourWinsControl]);
end;

function  TFourWinsControl.getPercentDone : Double;
begin
  result := FGame.PercentDone;
end;

procedure TFourWinsControl.DoInsert(Sender : TObject; Slot : Integer);
var i : Integer;
begin
  Canvas.CopyMode := cmDstInvert;
  for i := 0 to 5 do begin
  Canvas.CopyRect(rect((Slot-1)*(Width div Cols), 0,
                       (Slot)*(Width div Cols),        Height), Canvas, Rect(0,0,0,0));
    Sleep(100);                   
  end;
  Canvas.CopyMode := cmSrcCopy;
end;

function  TFourWinsControl.GetValue : Integer;
begin
  result := FGame.GetValue;
end;

procedure TFourWinsControl.DoChange(Sender : TObject);
begin
  Repaint;
  if Assigned(FOnChange) then FOnChange(Sender);
end;

procedure TFourWinsControl.DoGameEnds(Sender : TObject);
begin
  Repaint;
  if Assigned(FOnGameEnds) then FOnGameEnds(Sender);
end;

procedure TFourWinsControl.DoProgress(Sender : TObject);
begin
  Repaint;
  if Assigned(FOnProgress) then FOnProgress(Sender);
end;

procedure TFourWinsControl.SetActiveSlot(ASlot : TActiveSlotType);
begin
  if ASlot<>FActiveSlot then begin
    FActiveSlot:=ASlot;
    Repaint;
  end;
end;

procedure TFourWinsControl.Reset;
begin
  FGame.Reset;
end;

procedure TFourWinsControl.MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
begin
  if Button = mbLeft then FGame.InsertInto(ActiveSlot, FGame.Player);
  if ComputerPlayerOn then FGame.ComputerMakeMove;
end;

procedure TFourWinsControl.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  ActiveSlot := Trunc(x / (Width / Cols))+1;
end;

procedure TFourWinsControl.MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
begin
  If (0<=x) and (x<width) and (0<=y) and (y<height) and (Shift = [ssLeft]) then FGame.InsertInto(ActiveSlot, 0); 
end;


procedure TFourWinsControl.Paint;
var b : TBitmap;
    x,y:Integer;
begin
  b := TBitmap.Create;
  try
    b.Width := Width;
    b.height := Height;
    b.Canvas.Brush.Style := bsSolid;
    for x := 1 to Cols do begin
      if x = ActiveSlot then begin
         b.Canvas.Brush.Color := clAqua;
         b.Canvas.Pen.Color := clAqua;
         b.Canvas.FillRect(rect((x-1)*(Width div Cols), 0,
                           (x)*(Width div Cols),        Height));
      end else begin
         b.Canvas.Brush.Color := clBlue;
         b.Canvas.Pen.Color := clSilver;
         b.Canvas.FillRect(rect((x-1)*(Width div Cols), 0,
                           (x)*(Width div Cols),        Height));
      end;
      for y := 1 to Rows do
        case FGame.Feld[y,x] of
          0 : begin
                b.Canvas.Brush.Color := clRed;
                b.Canvas.Ellipse((x-1)*(Width div Cols)+2,(y-1)*(Height div Rows)+2, x*(Width div Cols)-2, y*(Height div Rows)-2);
              end;
          1 : begin
                b.Canvas.Brush.Color := clYellow;
                b.Canvas.Ellipse((x-1)*(Width div Cols)+2,(y-1)*(Height div Rows)+2, x*(Width div Cols)-2, y*(Height div Rows)-2);
              end;
          else
              begin
                b.Canvas.Brush.Color := clGray;
                b.Canvas.Ellipse((x-1)*(Width div Cols)+2,(y-1)*(Height div Rows)+2, x*(Width div Cols)-2, y*(Height div Rows)-2);
              end;
        end;
    end;
    Canvas.Draw(0,0,b);
  finally
    b.Free;
  end;
end;

function TFourWinsControl.GetWinner : String;
begin
  case FGame.Winner of
    0 : Result := 'Rot';
    1 : Result := 'Gelb';
    else
    Result := '';
  end;
end;
function TFourWinsControl.GetRows   : Integer; begin result := FGame.Rows; end;
function TFourWinsControl.GetCols   : Integer; begin result := FGame.Cols; end;

function TFourWinsControl.ComputerMakeMove : Integer;
begin
  Result := FGame.ComputerMakeMove;
end;

// ( Four Wins ) private

constructor TFourWins.Create;
begin
  reset;
end;

function TFourWins.GetRows   : Integer; begin result := FRows; end;
function TFourWins.GetCols   : Integer; begin result := FCols; end;

function  TFourWins.GetWinner : TPlayer; // Gewinner ausrechnen
begin
  result := MMWhoWon(FFeld);
end;

function  TFourWins.MMMayInsertInto(Feld : TFeld; Slot : Integer) : Boolean;
begin
  result := (Status=Playing) and (Feld[1, Slot] = -1);
end;

function  TFourWins.MMInsertInto(Feld : TFeld; Slot : Integer; Player : TPlayer) : TFeld;
var Row : Integer;
begin
  if MMMayInsertInto(Feld, Slot) then begin
    Row := 0;
    repeat inc(Row); until Feld[Row, Slot] <> -1;
    dec(Row);
    Feld[Row, Slot] := Player;
  end;
  result := Feld;
end;

function  TFourWins.MMWhoWon(Feld : TFeld) : TPlayer;
var x, y : Integer;
begin
  result := -1;
  for y := 1 to Rows do
    For x := 1 to Cols-3 do
      if (Feld[y,x] = Feld[y,x+1]) and
         (Feld[y,x] = Feld[y,x+2]) and
         (Feld[y,x] = Feld[y,x+3]) and
         (Feld[y,x] <> -1) then result := Feld[y,x];

  for y := 1 to Rows-3 do
    For x := 1 to Cols do
      if (Feld[y,x] = Feld[y+1,x]) and
         (Feld[y,x] = Feld[y+2,x]) and
         (Feld[y,x] = Feld[y+3,x]) and
         (Feld[y,x] <> -1) then result := Feld[y,x];

  for y := 1 to Rows-3 do
    For x := 1 to Cols-3 do begin
      if (Feld[y,x] = Feld[y+1,x+1]) and
         (Feld[y,x] = Feld[y+2,x+2]) and
         (Feld[y,x] = Feld[y+3,x+3]) and
         (Feld[y,x] <> -1) then result := Feld[y,x];
      if (Feld[y,x+3] = Feld[y+1,x+2]) and
         (Feld[y,x+3] = Feld[y+2,x+1]) and
         (Feld[y,x+3] = Feld[y+3,x]) and
         (Feld[y,x+3] <> -1) then result := Feld[y,x+3];
    end;
end;

function  TFourWins.MMGetValue(Feld : TFeld)                  : Integer;
function Bewertung(Vorher, Anzahl : Integer) : Integer;
begin
  Result := Vorher;
  if (Vorher < MaxInt-9) and (Vorher > MinInt+9) then
  case Anzahl of
    1: Inc(Result, 1);
    2: Inc(Result, 5);
    3: Inc(Result, 25);
    4: Result := MaxInt-1;
    -1: Dec(Result, 1);
    -2: Dec(Result, 6);
    -3: Dec(Result, 36);
    -4: Result := MinInt+1;
  end;
end;

function Gleiche(p1,p2,p3,p4: TPlayer) : Integer;
var Min, Max : TPlayer;
begin
  Max := FMax;
  Min := 1-Max;
  result := 0;
  if (p1<>Min) and (p2<>Min) and (p3<>Min) and (p4<>Min) then begin
    if p1 = Max then inc(Result);
    if p2 = Max then inc(Result);
    if p3 = Max then inc(Result);
    if p4 = Max then inc(Result);
  end else
  if (p1<>Max) and (p2<>Max) and (p3<>Max) and (p4<>Max) then begin
    if p1 = Min then dec(Result);
    if p2 = Min then dec(Result);
    if p3 = Min then dec(Result);
    if p4 = Min then dec(Result);
  end;
end;

var x, y : Integer;
begin // Heuristic
  result := 0;

  // Zwickmhlen
  // Horizontale
  for y := 1 to Rows do
    For x := 1 to Cols-4 do
      if (Feld[y,x] = -1) and (Feld[y,x+4] = -1) then
        case Gleiche(Feld[y,x+1], Feld[y,x+2], Feld[y,x+3], -1) of
          2:  Inc(Result, 100);
          3:  Inc(Result, 500);
          -2: Dec(Result, 5000);
          -3: Dec(Result, 10000);
        end;

  // Diagonalen
  for y := 1 to Rows-4 do
    For x := 1 to Cols-4 do begin
      if (Feld[y,x] = -1) and (Feld[y+4,x+4] = -1) then
        case Gleiche(Feld[y+1,x+1], Feld[y+2,x+2], Feld[y+3,x+3], -1) of
          2:  Inc(Result, 100);
          3:  Inc(Result, 500);
          -2: Dec(Result, 5000);
          -3: Dec(Result, 10000);
        end;
      if (Feld[y,x+4] = -1) and (Feld[y+4,x] = -1) then
        case Gleiche(Feld[y+1,x+3], Feld[y+2,x+2], Feld[y+3,x+1], -1) of
          2:  Inc(Result, 100);
          3:  Inc(Result, 500);
          -2: Dec(Result, 5000);
          -3: Dec(Result, 10000);
        end;
    end;

  // Horizontale Vierer
  For y := 1 to Rows do
    For x := 1 to Cols-3 do
      Result := Bewertung(Result, Gleiche(Feld[y,x], Feld[y,x+1], Feld[y,x+2], Feld[y,x+3]));

  // Vertikal Vierer
  for y := 1 to Rows-3 do
    For x := 1 to Cols do
      Result := Bewertung(Result, Gleiche(Feld[y,x], Feld[y+1,x], Feld[y+2,x], Feld[y+3,x]));

  // Diagonal
  for y := 1 to Rows-3 do
    For x := 1 to Cols-3 do begin
      Result := Bewertung(Result, Gleiche(Feld[y,x], Feld[y+1,x+1], Feld[y+2,x+2], Feld[y+3,x+3]));
      Result := Bewertung(Result, Gleiche(Feld[y,x+3], Feld[y+1,x+2], Feld[y+2,x+1], Feld[y+3,x]));
    end;

end;

// *********** MINI-MAX Functions **************++
function TFourWins.GMin(Feld : TFeld; Alpha, Beta, MyDepth : Integer) : TMinMaxResult;
var i, wert, z : Integer;
    p : TPlayer;
    Order : array[1..FCols] of Integer;
begin

  // *** Schlechtestmglicher Beta Wert
  Result.Bestmove := 1;
  Result.Value := MaxInt;

  p := MMWhoWon(Feld);
  if p = Max then begin
          result.Value := MaxInt-1; exit ;
  end else if p = 1-Max then begin
          result.Value := MinInt+1; exit;
  end;

  for i := 1 to FCols do Order[i] := i;
  for i := 1 to FCols do begin
    z := Random(FCols)+1;
    wert := Order[i];
    Order[i] := Order[z];
    Order[z] := wert;
  end;
  // *** Suchtiefenbeschrnkt
  inc(MyDepth);
  if MyDepth>5 then begin
    Result.Value    := MMGetValue(Feld);
    exit;
  end;

  // ** Nachfolgezustnde erzeugen
  for i := 1 to Cols do
    if MMMayInsertInto(Feld,Order[i]) then begin
      wert := GMax(MMInsertInto(Feld, Order[i], 1-Max), Alpha, Beta, MyDepth).Value;
      if Wert < Beta then Beta := Wert;
      if Alpha >= Beta then begin
        Result.Bestmove := Order[i];
        Result.Value    := Alpha;
        exit;
      end;
      if Wert < Result.Value then begin
        Result.Bestmove := Order[i];
        Result.Value    := wert;
      end;
    end;
end;

function TFourWins.GMax(Feld : TFeld; Alpha, Beta, MyDepth : Integer) : TMinMaxResult;
var i, wert, z : Integer;
    p : TPlayer;
    Order : array[1..FCols] of Integer;
begin

  // Reihenfolge randomisieren
  for i := 1 to FCols do Order[i] := i;
  for i := 1 to FCols do begin
    z := Random(FCols)+1;
    wert := Order[i];
    Order[i] := Order[z];
    Order[z] := wert;
  end;

  // *** Schlechtestmglicher Alpha Wert
  Result.Bestmove := 1;
  Result.Value := MinInt;


  p := MMWhoWon(Feld);
  if p = Max then begin
          result.Value := MaxInt-1; exit ;
  end else if p = 1-Max then begin
          result.Value := MinInt+1; exit;
  end;

  // *** Suchtiefenbeschrnkt
  inc(MyDepth);

  // ** Nachfolgezustnde erzeugen
  for i := 1 to Cols do
    if MMMayInsertInto(Feld,Order[i]) then begin
      wert := GMin(MMInsertInto(Feld, Order[i], Max), Alpha, Beta, MyDepth).Value;
      if MyDepth = 1 then begin
        FPercentDone := i / FCols;
        If Assigned(FOnProgress) then FOnProgress(Self);
      end;
      if Wert > Alpha then Alpha := Wert;
      if Alpha >= Beta then begin
        Result.Bestmove := Order[i];
        Result.Value    := Beta;
        exit;
      end;
      if Wert > Result.Value then begin
        Result.Bestmove := Order[i];
        Result.Value    := wert;
      end;
    end;
end;

function  TFourWins.MMMinMax : TMinMaxResult;
begin // MiniMax
  Randomize;
  FPercentDone := 0;
  If Assigned(FOnProgress) then FOnProgress(Self);
  Result := GMax(Feld, MinInt, MaxInt, 0);
  FPercentDone := 1;
  If Assigned(FOnProgress) then FOnProgress(Self);
end;
// **************************************************

// ( Four Wins ) public
function  TFourWins.MayInsertInto(slot : Integer) : Boolean;
begin
  result := MMMayInsertInto(Feld, Slot)
end;

function  TFourWins.InsertInto(slot : Integer; Player : TPlayer)    : Boolean;
begin
  Result := MayInsertInto(Slot);
  if not result then exit;
  FFeld   := MMInsertInto(Feld, Slot, Player);
  Self.Player := 1-Player;
  if Winner <> -1 then begin FStatus := Finished; if Assigned(FOnGameEnds) then FOnGameEnds(Self); exit; end;
  if assigned(FOnChange) then FOnChange(Self);
  if assigned(FOnInsert) then FOnInsert(Self, Slot);
end;

function  TFourWins.GetValue  : Integer; // Wert ausrechnen
begin
  result := MMGetValue(FFeld);
end;

function TFourWins.ComputerMakeMove : Integer;
var w : TMinMaxResult;
begin
  Max := Player;
  w:=MMMinMax;
  InsertInto(w.Bestmove, Player);
  result := w.Value;
end;

procedure TFourWins.Reset;
var x,y:Integer;
begin
  Player := 0;
  for y := 1 to Rows do for x := 1 to Cols do FFeld[y,x]:=-1;
  for x := 1 to Cols do FFeld[Rows+1,x]:=2; // Stopper
  FStatus := Playing;
  if assigned(FOnReset) then FOnReset(Self);
  if assigned(FOnChange) then FOnChange(Self);
end;


end.

