{-----------------------------------------------------------------------------}
{ A component and a function (use the one you prefer) to encapsulate the      }
{ Win95 style directory selection dialog SHBrowseForFolder().                 }
{ Copyright 1996, Brad Stowers.  All Rights Reserved.                         }
{ This component can be freely used and distributed in commercial and private }
{ environments, provied this notice is not modified in any way and there is   }
{ no charge for it other than nomial handling fees.  Contact me directly for  }
{ modifications to this agreement.                                            }
{-----------------------------------------------------------------------------}
{ Feel free to contact me if you have any questions, comments or suggestions  }
{ at bstowers@pobox.com or 72733,3374 on CompuServe.                          }
{ The lateset version will always be available on the web at:                 }
{   http://www.pobox.com/~bstowers/delphi/delphi.html                         }
{-----------------------------------------------------------------------------}
{ Date last modified:  09/17/96                                               }
{-----------------------------------------------------------------------------}

{ ----------------------------------------------------------------------------}
{ TBrowseDirectory v2.00                                                      }
{ ----------------------------------------------------------------------------}
{ Description:                                                                }
{   A dialog that displays the user's system in a heirarchial manner and      }
{   allows a selection to be made.  It is a wrapper for SHBrowseForFolder(),  }
{   which is rather messy to use directly.                                    }
{ Notes:                                                                      }
{   * Requires Delphi v2.01's ShlObj unit.  If you don't have the 2.01 update }
{     you can get the equivalent using Pat Ritchey's ShellObj unit.  It is    }
{     freely available on his web site at                                     }
{     http://ourworld.compuserve.com/homepages/PRitchey/                      }
{ ----------------------------------------------------------------------------}
{ Revision History:                                                           }
{ 1.00:  + Initial release                                                    }
{ 1.01:  + Now uses Delphi 2.01 ShlObj unit.                                  }
{        + Added callback stuff.  See the DoInitialized and DoSelChanged      }
{          methods, Center property, and OnSelChange event.                   }
{        + Added Center property.                                             }
{        + Added OnSelChange event.                                           }
{ 2.00:  + Calling as a function is no longer supported.  Too complicated to  }
{          support all the new properties that way.                           }
{        + Added StatusText property.  If this property is empty ('') when    }
{          Execute is called, it will not be available during the life of     }
{          the dialog at all.                                                 }
{        + Selected property changed to Selection.  Now available at design   }
{          time.  Setting the value causes that value, if it exists, to be    }
{          selected in the tree.                                              }
{        + Added EnableOKButton property.  Use in conjunction with the        }
{          OnSelChange event to control when the OK button is enabled.        }
{        + Simplified DoSelChanged method.  Boy, was I asleep when I wrote    }
{          that one... Doh.                                                   }
{ ----------------------------------------------------------------------------}

unit BrowseDr;

{$IFNDEF WIN32}
  ERROR!  This unit only available on Win32!
{$ENDIF}

interface

uses Windows, MyShlObj, Controls, Classes, DsgnIntf;

type
  { These are equivalent to the CSIDL_* constants in the Win32 (95?) API.  }
  { They are used to specify the root of the heirarchy.                    }
  { NOTE: the idDesktopExpanded is not docuemnted, but it seems to be used }
  {       by the Win95 Explorer.  I find it useful, but use at your own    }
  {       risk.  It may be "fixed" in some future release of Win95.        }
  TRootID = (
    idDesktop, idDesktopExpanded, idPrograms, idControlPanel, idPrinters,
    idPersonal, idFavorites, idStartup, idRecent, idSendTo, idRecycleBin,
    idStartMenu, idDesktopDirectory, idDrives, idNetwork, idNetHood, idFonts,
    idTemplates
   );
  { These are equivalent to the BIF_* constants in the Win32 API.         }
  { They are used to specify what items can be expanded, and what items   }
  { can be selected.                                                      }
  TBrowseFlag = (
    bfDirectoriesOnly, bfDomainOnly, bfAncestors, bfComputers, bfPrinters
   );
  TBrowseFlags = set of TBrowseFlag;

  TBDSelChangedEvent = procedure(Sender: TObject; const NewSel: string) of object;

type
  TBrowseDirectoryDlg = class(TComponent)
  private
    { Internal variables }
    FDlgWnd: HWND;
    { Property variables }
    FTitle: string;
    FRoot: TRootID;
    FOptions: TBrowseFlags;
    FSelection: string;
    FCenter: boolean;
    FStatusText: string;
    FEnableOKButton: boolean;
    FSelChanged: TBDSelChangedEvent;
    FOnCreate: TNotifyEvent;
  protected
    // internal event methods.
    procedure DoInitialized(Wnd: HWND); virtual;
    procedure DoSelChanged(Wnd: HWND; Item: PItemIDList); virtual;
    // property methods
    procedure SetStatusText(const Val: string);
    procedure SetSelection(const Val: string);
    procedure SetEnableOKButton(Val: boolean);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    { Displays the dialog.  Returns true if user selected an item and       }
    { pressed OK, otherwise it returns false.                               }
    function Execute: boolean; virtual;
  published
    { The item selected.  Setting this before calling Execute will cause    }
    { the value to be initially selected when the dialog is first displayed.}
    property Selection: string read FSelection write SetSelection;
    { Text to display at the top of the dialog.                             }
    property Title: string read FTitle write FTitle;
    { Item that is to be treated as the root of the display.                }
    property Root: TRootID read FRoot write FRoot default idDesktopExpanded;
    { Options to control what is allowed to be selected and expanded.       }
    property Options: TBrowseFlags read FOptions write FOptions default [];
    { Center the dialog on screen }
    property Center: boolean read FCenter write FCenter default TRUE;
    { Status text displayed above the tree.                                 }
    property StatusText: string read FStatusText write SetStatusText;
    { Enable or disable the OK button in the dialog.                        }
    property EnableOKButton: boolean read FEnableOKButton write SetEnableOKButton;
    { Event fired every time a new selection is made.                       }
    property OnSelChanged: TBDSelChangedEvent read FSelChanged write FSelChanged;
    { Event fired when dialog has been created.                             }
    property OnCreate: TNotifyEvent read FOnCreate write FOnCreate;     
  end;

  { A component editor (not really) to allow on-the-fly testing of the      }
  { dialog.  Right click the component and select 'Test Dialog', or simply  }
  { double click the component, and the browse dialog will be displayed     }
  { with the current settings.                                              }
  TBrowseDialogEditor = class(TDefaultEditor)
  public
    procedure ExecuteVerb(Index : Integer); override;
    function GetVerb(Index : Integer): string; override;
    function GetVerbCount : Integer; override;
    procedure Edit; override;
  end;

procedure Register;

implementation

uses OLE2, Forms, Dialogs, SysUtils;

// Utility functions used to convert from Delphi set types to API constants.
function ConvertRoot(Root: TRootID): integer;
const
  RootValues: array[TRootID] of integer = (
    CSIDL_DESKTOP, $0001, CSIDL_PROGRAMS, CSIDL_CONTROLS, CSIDL_PRINTERS,
    CSIDL_PERSONAL, CSIDL_FAVORITES, CSIDL_STARTUP, CSIDL_RECENT, CSIDL_SENDTO,
    CSIDL_BITBUCKET, CSIDL_STARTMENU, CSIDL_DESKTOPDIRECTORY, CSIDL_DRIVES,
    CSIDL_NETWORK, CSIDL_NETHOOD, CSIDL_FONTS, CSIDL_TEMPLATES
   );
begin
  Result := RootValues[Root];
end;

function ConvertFlags(Flags: TBrowseFlags): UINT;
const
  FlagValues: array[TBrowseFlag] of UINT = (
    BIF_RETURNONLYFSDIRS, BIF_DONTGOBELOWDOMAIN, BIF_RETURNFSANCESTORS,
    BIF_BROWSEFORCOMPUTER, BIF_BROWSEFORPRINTER
   );
var
  Opt: TBrowseFlag;
begin
  Result := 0;
  { Loop through all possible values }
  for Opt := Low(TBrowseFlag) to High(TBrowseFlag) do
    if Opt in Flags then
      Result := Result OR FlagValues[Opt];
end;


function BrowseCallbackProc(Wnd: HWnd; Msg: UINT; lParam: LPARAM; lData: LPARAM): integer; stdcall;
begin
  Result := 0;
  case Msg of
    BFFM_INITIALIZED:
      if lData <> 0 then
        TBrowseDirectoryDlg(lData).DoInitialized(Wnd);
    BFFM_SELCHANGED:
      if lData <> 0 then
        TBrowseDirectoryDlg(lData).DoSelChanged(Wnd, PItemIDList(lParam));
  end;
end;


function BrowseDirectory(var Dest: string; const AParent: TWinControl;
                         const Title: string; Root: TRootID; Flags: TBrowseFlags;
                         WantStatusText: boolean; Callback: TFNBFFCallBack;
                         Data: Longint): boolean;
var
  ShellMalloc: IMALLOC;
  shBuff: PChar;
  BrowseInfo: TBrowseInfo;
  idRoot, idBrowse: PItemIDList;
begin
  Result := FALSE; // Assume the worst.
  Dest := ''; // Clear it out.
  SetLength(Dest, MAX_PATH);  // Make sure their will be enough room in dest.
  if SHGetMalloc(ShellMalloc) = NOERROR then begin
    try
      shBuff := PChar(ShellMalloc.Alloc(MAX_PATH)); // Shell allocate buffer.
      if assigned(shBuff) then begin
        try
          // Get id for desired root item.
          SHGetSpecialFolderLocation(AParent.Handle, ConvertRoot(Root), idRoot);
          try
            with BrowseInfo do begin  // Fill info structure
              hwndOwner := AParent.Handle;
              pidlRoot := idRoot;
              pszDisplayName := shBuff;
              lpszTitle := PChar(Title);
              ulFlags := ConvertFlags(Flags);
              if WantStatusText then
                ulFlags := ulFlags or BIF_STATUSTEXT;
              lpfn := Callback;
              lParam := Data;
            end;
            idBrowse := SHBrowseForFolder(BrowseInfo);
            if assigned(idBrowse) then begin
              try
                SHGetPathFromIDList(idBrowse, shBuff); // Turn into real path.
                Dest := shBuff; // Put it in user's variable.
                Result := TRUE; // Success!
              finally
                ShellMalloc.Free(idBrowse); // Clean up after ourselves
              end;
            end;
          finally
            ShellMalloc.Free(idRoot); // Clean-up.
          end;
        finally
          ShellMalloc.Free(shBuff); // Clean-up.
        end;
      end;
    finally
      ShellMalloc.Release; // Clean-up.
    end;
  end;
end;


constructor TBrowseDirectoryDlg.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDlgWnd := 0;
  FTitle := '';
  FRoot := idDesktopExpanded;
  FOptions := [];
  FSelection := '';
  FCenter := TRUE;
  FSelChanged := NIL;
  FStatusText := '';
end;

destructor TBrowseDirectoryDlg.Destroy;
begin
  inherited Destroy;
end;

function TBrowseDirectoryDlg.Execute: boolean;
var
  S: string;
  Parent: TWinControl;
begin
  { Determine who the parent is. }
  if Owner is TWinControl then
    Parent := Owner as TWinControl
  else
    Parent := Application.MainForm;

  { Call the function }
  Result := BrowseDirectory(S, Parent, FTitle, FRoot, FOptions, FStatusText <> '',
                            BrowseCallbackProc, LongInt(Self));
  { If selection made, update property }
  if Result then
    FSelection := S
  else
    FSelection := ''
end;

procedure TBrowseDirectoryDlg.DoInitialized(Wnd: HWND);
var
  Rect: TRect;
begin
  FDlgWnd := Wnd;
  if FCenter then begin
    GetWindowRect(Wnd, Rect);
    SetWindowPos(Wnd, 0,
      (GetSystemMetrics(SM_CXSCREEN) - Rect.Right + Rect.Left) div 2,
      (GetSystemMetrics(SM_CYSCREEN) - Rect.Bottom + Rect.Top) div 2,
      0, 0, SWP_NOACTIVATE or SWP_NOSIZE or SWP_NOZORDER);
  end;
  // Documentation for BFFM_ENABLEOK is incorrect.  Value sent in LPARAM, not WPARAM.
  SendMessage(FDlgWnd, BFFM_ENABLEOK, 0, LPARAM(FEnableOKButton));
  if FStatusText <> '' then
    SendMessage(Wnd, BFFM_SETSTATUSTEXT, 0, LPARAM(FStatusText));
  if FSelection <> '' then
    SendMessage(FDlgWnd, BFFM_SETSELECTION, 1, LPARAM(FSelection));
  if assigned(FOnCreate) then
    FOnCreate(Self);
end;

procedure TBrowseDirectoryDlg.DoSelChanged(Wnd: HWND; Item: PItemIDList);
var
  Name: string;
begin
  if not assigned(FSelChanged) then exit;
  Name := '';
  SetLength(Name, MAX_PATH);
  SHGetPathFromIDList(Item, PChar(Name));
  SetLength(Name, StrLen(PChar(Name)));
  FSelChanged(Self, Name);
end;

procedure TBrowseDirectoryDlg.SetStatusText(const Val: string);
begin
  if FStatusText = Val then exit;
  FStatusText := Val;
  if FDlgWnd <> 0 then
    SendMessage(FDlgWnd, BFFM_SETSTATUSTEXT, 0, LPARAM(FStatusText));
end;

procedure TBrowseDirectoryDlg.SetSelection(const Val: string);
begin
  if FSelection = Val then exit;
  FSelection := Val;
  // need to remove any trailing backslash characters.
  if FSelection[Length(FSelection)] = '\' then
    SetLength(FSelection, Length(FSelection)-1);
  if FDlgWnd <> 0 then
    SendMessage(FDlgWnd, BFFM_SETSELECTION, 1, LPARAM(FSelection));
end;

procedure TBrowseDirectoryDlg.SetEnableOKButton(Val: boolean);
begin
  if FEnableOKButton = Val then exit;
  FEnableOKButton := Val;
  if FDlgWnd <> 0 then
    // Documentation for BFFM_ENABLEOK is incorrect.  Value sent in LPARAM, not WPARAM.
    SendMessage(FDlgWnd, BFFM_ENABLEOK, 0, LPARAM(FEnableOKButton));
end;



// Component Editor (not really) to allow on the fly testing of the dialog
procedure TBrowseDialogEditor.ExecuteVerb(Index: Integer);
begin
  {we only have one verb, so exit if this ain't it}
  if Index <> 0 then Exit;
  Edit;
end;

function TBrowseDialogEditor.GetVerb(Index: Integer): AnsiString;
begin
  Result := 'Test Dialog';
end;

function TBrowseDialogEditor.GetVerbCount: Integer;
begin
  Result := 1;
end;

procedure TBrowseDialogEditor.Edit;
begin
  with TBrowseDirectoryDlg(Component) do
    if Execute then
      MessageDlg(Format('Item selected:'#13#13'%s', [Selection]),
                 mtInformation, [mbOk], 0);
end;


procedure Register;
begin
  { You may prefer it on the Dialogs page, I like it on Win95 because it is
    only available on Win95.                                                }
  RegisterComponents('Win95', [TBrowseDirectoryDlg]);
  RegisterComponentEditor(TBrowseDirectoryDlg, TBrowseDialogEditor);
end;


end.
