
unit Bitwise;

{   Bitwise components manipulate a single bit within a data field.
    TDBBitwiseCheckBox works just like TDBCheckBox except that you also
    specify a value; typically a power of 2 (1,2,4,8,16,32,.. 16384) If the
    box is checked, that value is OR'd with the field's value. If the
    box is unchecked, that value is NOT AND'd with the field's value.
    This is set with CheckValue.

    Caveat: the underlying data field MUST be an integer.

    TDBBitwiseRadioButton works the same way except it looks like a radio
    button. Here, since the "other" radio buttons in the group presumably
    correspond to other enumerated values, the field is set to CheckValue
    when the button is pressed. For example, you might have four radio
    buttons with checkValues 1,2,3,4 to indicate one of four choices. For
    the CheckBoxes, you would have one of four values 1,2,4,8 for the
    sixteen possible permutations.

    Caveat: the underlying data field MUST be an integer.
}

interface

uses    DB, DBTables, DBCtrls, LiteChk, Forms, Menus, Controls, SysUtils,
        Messages, Classes, Graphics, StdCtrls;

type
    TDBBitwiseCheckBox = Class( TLiteCheckBox )

    private
    myCheckValue:   Integer;

    FDataLink:      TFieldDataLink;

    function    GetDataField: String;
    function    GetDataSource: TDataSource;
    procedure   SetDataField( const value: String );
    procedure   SetDataSource( value: TDataSource );

    procedure   DataChange( sender: TObject );
    procedure   UpdateData( Sender: TObject );

    procedure   CMExit( var Message: TCMExit ); message CM_EXIT;


    protected
    procedure   Toggle; override;
    procedure   KeyPress( var Key: Char ); override;
    procedure   Notification( AComponent: TComponent; Operation: TOperation ); override;

    public
    constructor Create( owner: TComponent ); override;
    destructor  Destroy; override;

    property    State;

    published
    property    DataField:  String read GetDataField write SetDataField;
    property    DataSource: TDataSource read GetDataSource write SetDataSource;
    property    CheckValue: Integer read myCheckValue write myCheckValue;
    end;


    TDBBitwiseRadioButton = Class( TRadioButton )

    private
    myCheckValue:   Integer;

    FDataLink:      TFieldDataLink;

    function    GetDataField: String;
    function    GetDataSource: TDataSource;
    procedure   SetDataField( const value: String );
    procedure   SetDataSource( value: TDataSource );

    procedure   DataChange( sender: TObject );
    procedure   UpdateData( Sender: TObject );

    procedure   CMExit( var Message: TCMExit ); message CM_EXIT;
    procedure   CMToggle( var Message: TWMNCLButtonUp ); message WM_NCLBUTTONUP;

    protected
    procedure   KeyPress( var Key: Char ); override;
    procedure   Notification( AComponent: TComponent; Operation: TOperation ); override;

    public
    constructor Create( owner: TComponent ); override;
    destructor  Destroy; override;
    procedure   MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer ); override;

    published
    property    DataField:  String read GetDataField write SetDataField;
    property    DataSource: TDataSource read GetDataSource write SetDataSource;
    property    CheckValue: Integer read myCheckValue write myCheckValue;
    end;

procedure Register;

implementation

procedure Register;

{   VCL Registration function.
}

begin
    RegisterComponents( 'Midnight', [ TDBBitwiseCheckBox ] );
    RegisterComponents( 'Midnight', [ TDBBitwiseRadioButton ] )
end;

{ TDBBitwiseCheckBox }

function TDBBitwiseCheckBox.GetDataField: String;
begin
    Result := FDataLink.FieldName
end;

function TDBBitwiseCheckBox.GetDataSource: TDataSource;
begin
    Result := FDataLink.DataSource
end;


procedure TDBBitwiseCheckBox.SetDataField( const value: String );
begin
    FDataLink.FieldName := value
end;

procedure TDBBitwiseCheckBox.SetDataSource( value: TDataSource );
begin
    FDataLink.DataSource := value
end;


procedure TDBBitwiseCheckBox.UpdateData( Sender: TObject );
begin
    if ( State = cbGrayed ) then
        FDataLink.Field.AsInteger := FDataLink.Field.AsInteger AND ( $7FFF - myCheckValue )
    else if ( Checked ) then
        FDataLink.Field.AsInteger := FDataLink.Field.AsInteger OR myCheckValue
    else
        FDataLink.Field.AsInteger := FDataLink.Field.AsInteger AND ( $7FFF - myCheckValue )
end;


procedure TDBBitwiseCheckBox.DataChange( sender: TObject );

{   A change to the underlying data is reflected in the state of the checkbox.
}

begin
    if ( FDataLink.Field = NIL ) then
        Checked := FALSE
    else if ( myCheckValue < 1 ) then
        Checked := FALSE
    else if ( ( FDataLink.Field.AsInteger AND myCheckValue ) = myCheckValue ) then
        Checked := TRUE
    else
        Checked := FALSE
end;

constructor TDBBitwiseCheckBox.Create( owner: TComponent );
begin
    inherited Create( owner );

    myCheckValue := 0;
    FDataLink := TFieldDataLink.Create;
    FDataLink.Control := Self;
    FDataLink.OnDataChange := DataChange;
    FDataLink.OnUpdateData := UpdateData
end;

destructor TDBBitwiseCheckBox.Destroy;
begin
    FDataLink.Free;

    inherited Destroy
end;


procedure TDBBitwiseCheckBox.CMExit( var Message: TCMExit );
begin
    try
        FDataLink.UpdateRecord;
    except
        SetFocus;
        raise;
    end;

    inherited
end;


procedure TDBBitwiseCheckBox.Toggle;
begin
    if FDataLink.Edit then
    begin
        inherited Toggle;
        FDataLink.Modified
    end
end;

procedure TDBBitwiseCheckBox.KeyPress(var Key: Char);
begin
    inherited KeyPress( Key );
    case Key of
        #8, ' ':
        FDataLink.Edit;

        #27:
        FDataLink.Reset
    end
end;

procedure TDBBitwiseCheckBox.Notification( AComponent: TComponent; Operation: TOperation );
begin
    inherited Notification( AComponent, Operation );

    if ( ( Operation = opRemove ) and ( FDataLink <> nil ) and ( AComponent = DataSource ) ) then
        DataSource := nil
end;


{ TDBBitwiseRadioButton }

function TDBBitwiseRadioButton.GetDataField: String;
begin
    Result := FDataLink.FieldName
end;

function TDBBitwiseRadioButton.GetDataSource: TDataSource;
begin
    Result := FDataLink.DataSource
end;


procedure TDBBitwiseRadioButton.SetDataField( const value: String );
begin
    FDataLink.FieldName := value
end;

procedure TDBBitwiseRadioButton.SetDataSource( value: TDataSource );
begin
    FDataLink.DataSource := value
end;


procedure TDBBitwiseRadioButton.UpdateData( Sender: TObject );
begin
    if ( Checked ) then
        FDataLink.Field.AsInteger := myCheckValue

    { for now we assume that one of the buttons must be selected... }
end;


procedure TDBBitwiseRadioButton.DataChange( sender: TObject );

{   A change to the underlying data is reflected in the state of the RadioButton.
}

begin
    if ( FDataLink.Field = NIL ) then
        Checked := FALSE
    else if ( myCheckValue < 1 ) then
        Checked := FALSE
    else if ( FDataLink.Field.AsInteger = myCheckValue ) then
        Checked := TRUE
    else
        Checked := FALSE
end;

constructor TDBBitwiseRadioButton.Create( owner: TComponent );
begin
    inherited Create( owner );

    myCheckValue := 0;
    FDataLink := TFieldDataLink.Create;
    FDataLink.Control := Self;
    FDataLink.OnDataChange := DataChange;
    FDataLink.OnUpdateData := UpdateData
end;

destructor TDBBitwiseRadioButton.Destroy;
begin
    FDataLink.Free;

    inherited Destroy
end;


procedure TDBBitwiseRadioButton.CMExit( var Message: TCMExit );
begin
    try
        FDataLink.UpdateRecord;
    except
        SetFocus;
        raise;
    end;

    inherited
end;


procedure TDBBitwiseRadioButton.KeyPress(var Key: Char);
begin
    inherited KeyPress( Key );
    case Key of
        #8, ' ':
        FDataLink.Edit;

        #27:
        FDataLink.Reset
    end
end;

procedure TDBBitwiseRadioButton.Notification( AComponent: TComponent; Operation: TOperation );
begin
    inherited Notification( AComponent, Operation );

    if ( ( Operation = opRemove ) and ( FDataLink <> nil ) and ( AComponent = DataSource ) ) then
        DataSource := nil
end;


procedure TDBBitwiseRadioButton.CMToggle( var Message: TWMNCLButtonUp );
begin
    if FDataLink.Edit then
        FDataLink.Modified
end;

procedure TDBBitwiseRadioButton.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer ); 
begin
    inherited MouseUp( button, shift, x, y );
    if ( button = mbLeft ) then
        if FDataLink.Edit then
            FDataLink.Modified
end;

end.

