{
OBJECT      : Component Design: Persistance, Property Editors
              TMyRec    : user-specific type
              TMyRecObj : wrapper around data
              TComp     : component with data-object

LAST CHANGES: 16.12.1996

AUTHOR      : Kurt Spitzley,(101.10490@germanynet.de or Kurt.Spitzley@rz-online.de)

COPYRIGHT   : (c) 1996 by Kurt Spitzley, All rights reserved.
              This software should not be SOLD by anyone. It is distributed as
              freeware and therefore may be used free of charge.

DISCLAIMER  : I have provided this software to the public free of charge. You
              accept this software AS IS without any representation or warranty
              of any kind, including but not limited to the warranty of
              merchantability or fitness for a particular purpose.

DESCRIPTION:

A components properties reflect the natural interface for setting up a components
startup-state. All changes made within the object-inspector while in design-mode
will be stored in the appropriate dfm-file after closing the project. Delphi
supports a lot of standard-types which can be edited directly in the object-
inspector and which will be stored/restored automatically in the dfm-file.
But what, if you decide to define a property with a user-specific type, for
example a record-type ?
Then you have to consider that
- Delphi doesn't support record-types as properties
- object-types are supported so you should define a wrapper-object around your data
- the object-type is persistent
- you add code for saving/loading the data respecting Delphi's stream-machinery

The example defines a simple component with member data of user-specific type
TMyRec. TMyRecObj is the object-wrapper around the data TMyRec. Changes will
be performed through the property-editor TMyRecObjEditor. There are many comments
added to the code so I think the example should be self-explaining.
}

unit Comp;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DsgnIntf,PropEd;

type
  // this is the type of user-specific data to be stored later on in the dfm-file
  TMyRec=record
    s: string;
    l: longint;
  end;

  TMyRecObj=class(TPersistent) // wrapper-object around user-data
  private
    FMyRec:TMyRec; // holds user-data
    FHasData: Boolean;  // indicates that FMyRec contains new data
  protected
    procedure ReadData(Reader: TReader); virtual;// read user-data from stream
    procedure WriteData(Writer: TWriter); virtual;// write user-data to stream
    {alternative, if DefineBinaryProperty is used:
    procedure ReadData(Stream: TStream); virtual;
    procedure WriteData(Stream: TStream); virtual;}
    procedure DefineProperties(Filer: TFiler); override;
  public
    procedure SetMyRec(MyRec:TMyRec); // set..
    function GetMyRec:TMyRec;         // ..and get user-data
  end;

  TComp = class(TComponent) // example-component with TMyRecObj-member data
  private
    FMyRecObj: TMyRecObj;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property MyRecObj: TMyRecObj read FMyRecObj write FMyRecObj stored true;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('KSDKB', [TComp]);
end;

// method for library to read data from stream
procedure TMyRecObj.ReadData(Reader: TReader);
begin
   with FMyRec do
     begin
       s:=Reader.ReadString;
       l:=Reader.ReadInteger;
     end;
end;
{alternative, if DefineBinaryProperty is used:
procedure TMyRecObj.ReadData(Stream:TStream);
begin
  Code, to read data from stream (e.g. Stream.ReadBuffer...)
end;}

// method for library to write data to stream
procedure TMyRecObj.WriteData(Writer: TWriter);
begin
   with FMyRec do
     begin
       Writer.WriteString(s);
       Writer.WriteInteger(l);
     end;
end;
{alternative, if DefineBinaryProperty is used:
procedure TMyRecObj.WriteData(Stream: TStream);
begin
  Code, to write data to stream (e.g. Stream.ReadBuffer...)
end;}

procedure TMyRecObj.SetMyRec(MyRec:TMyRec);
begin
  FHasData:=true; // indicates new data to be stored by Delphi
  FMyRec:=MyRec;
end;

function TMyRecObj.GetMyRec:TMyRec;
begin
  Result:=FMyRec;
end;

// method to allow data to be read and written by the library
procedure TMyRecObj.DefineProperties(Filer: TFiler);
begin
  Filer.DefineProperty('MyRecObjData', ReadData, WriteData, FHasData);
  {alternative, if User-Data shall be stored binary
  Filer.DefineBinaryProperty('MyRecObjData', ReadData, WriteData, FHasData);}
end;

constructor TComp.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FMyRecObj:=TMyRecObj.Create;
end;

destructor TComp.Destroy;
begin
  FMyRecObj.Free;
  inherited Destroy;
end;

end.

