{*************
 *
 * Project....: Experiences and Examples
 * File.......: ATabCtrl.pas
 * Created by : Anton A. Mints
 * Date.......: 1 December 1996
 * Purpose....: Tab Control with glyph on the tabs.
 *
 * $Archive: $
 *
 * $History: $
 *
 *************}

unit
  ATabCtrl;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, CommCtrl;

type

{ TCEOTabControl }
  TCEOTabControl = class (TTabControl)
  private
    FImages: TImageList;
    FImageChangeLink: TChangeLink;

    procedure ImageListChange (Sender: TObject);
    procedure SetImages (Value: TImageList);

    procedure SetImage (Index: Integer; imIndex: Integer);
    function GetImage (Index: Integer): Integer;

  public
    constructor Create (AOwner: TComponent); override;
    destructor Destroy; override;

    property Image[Index: Integer]: Integer read GetImage write SetImage; default;

  published
    property Images: TImageList read FImages write SetImages;
    
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Additional', [TCEOTabControl]);
end;

{*******************************************************************}
{*                     CLASS TCEOTabControl                        *}
{*******************************************************************}

constructor TCEOTabControl.Create (AOwner: TComponent);
begin
  inherited Create (AOwner);
  FImageChangeLink := TChangeLink.Create;
  FImageChangeLink.OnChange := ImageListChange;
end;

destructor TCEOTabControl.Destroy;
begin
  FImageChangeLink.Free;
  inherited Destroy;
end;


{*************
* Name.......: ImageListChange
* Description: Be called whenever a change in the ImageList occurs to
*             notify TabControl about the changes.
*}
procedure TCEOTabControl.ImageListChange(Sender: TObject);
begin
  if HandleAllocated then
    SendMessage(Handle, TCM_SETIMAGELIST, 0, Longint(TImageList(Sender).Handle));
end;


{*************
* Name.......: SetImages
* Description: Assign ImageList to internal class variable and register
*              TChangeLink object.
*}
procedure TCEOTabControl.SetImages (Value: TImageList);
begin
  if Assigned (FImages) then
    Images.UnRegisterChanges(FImageChangeLink);

  FImages := Value;

  if Assigned (FImages) then
    begin
      Images.RegisterChanges (FImageChangeLink);
      SendMessage(Handle, TCM_SETIMAGELIST, 0, Longint(Images.Handle));
    end
  else
    SendMessage(Handle, TCM_SETIMAGELIST, 0, Longint(0));
end;


{*************
* Name.......: SetImage
* Description: Display image from ImageList on Index tab.
*}
procedure TCEOTabControl.SetImage (Index: Integer; imIndex: Integer);
var
  imItem: TTCItem;
begin
  imItem.iImage := imIndex;
  imItem.mask := TCIF_IMAGE;
  SendMessage (Handle, TCM_SETITEM, Index, Longint(@imItem));
  Invalidate;
end;


{*************
* Name.......: GetImage
*}
function TCEOTabControl.GetImage (Index: Integer): Integer;
var
  imItem: TTCItem;
begin
  if Assigned (FImages) then
    begin
      imItem.mask := TCIF_IMAGE;
      SendMessage (Handle, TCM_GETITEM, Index, Longint(@imItem));
      Result := imItem.iImage;
    end
  else
    Result := -1
end;

end.
