//Written by Varian Software Services NL
//Subject: Async component for Delphi 2.01
//Version: Beta 0.69
//Date: 26 Feb 1997
//Last update 2 July 1997
//Release: Freeware

//if you make any modifications to the source, please send us a copy.
//We will verify your changes and give you proper credit when included.

//Please send any questions, suggestions or remarks to our following
//address: Varian@worldaccess.nl

//Credits and our thanks go to:
//- Zuo Hang, from Beijing, P. R. China
//- Sten Karlson
//- Boris V. Evsikov
//- Thomas E. Paulsen

//Latest updates

//Version Beta 0.25

//-Replaced SetLinestate function in seperate function calls so users don't
// need to worry about any win32 variable references. Just turn it On/Off.

//-Received a notice that the communication device couldn't close/and reopen
// again in one session. Fixed.

//-Async32 now uses overlapped IO in EventThread

//-Changed the TComm32.write buffer parameter in a constant. The var
// parameter would sometimes block all data written through it. Fixed.

//-Moved TComm32.enabled function to the public declarations part so
// users can check the state of the communicationsdevice.

//Version 0.68, tested

//-added an OnOpen event wich notifies the user of the port status.
// expect error 0 when no failure occured.

//-TComm32 wouldn't respond correctly under win NT during read/write.
// In earlier releases we had added the overlapped flag to the open
// statement. This caused the problem under NT. Fixed.
// by  Sten Karlson <sten.karlson@daprod.se>
//     Boris V. Evsikov <borikevs@geocities.com>

//-added global reference FCT (TComStat)

//-added global reference FDCB (TDCB)

//-made some internal changes to DTR/RTS/BREAK functions

//Version 0.69

//-added GetOverlappedResult in WaitEvent Thread
// There seemed to be some data-loss when this call wasn't implemented.
// by  Thomas E. Paulsen <tep@iau.dtu.dk>


//Remarks:
//Never try to read more data then available. This will block all other
//operations until some bytes arrive at the comport. Just verify the
//Count parameter used in the RxCharEvent.

//"Final" version 1.0 will be released for Delphi 3 only!


unit async32;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TC32Event = procedure(Sender: TObject; Status: dword) of object;
  TC32EventState = (evBreak, evCTS, evDSR, evError, evRing,
    evRlsd, evRxChar, evRxFlag, evTxEmpty);
  TC32EventType = set of TC32EventState;

  TC32Thread = class(TThread)
  private
    FHandle: THandle;
    FStatus: dword;
    FOnSignal: TC32Event;
    hCloseEvent: THandle;
  protected
    procedure Execute; override;
    procedure DoOnSignal;
  public
    constructor Create(Handle: THandle; Events: TC32EventType);
    destructor Destroy; override;
    procedure ReleaseThread;
    property OnSignal: TC32Event read FOnSignal write FOnSignal;
  end;

  TBaudRate = (cbr110, cbr300, cbr600, cbr1200, cbr2400, cbr4800,
               cbr9600, cbr14400, cbr19200, cbr38400, cbr56000,
               cbr57600, cbr115200, cbr128000, cbr256000);
  TParity = (paNone,paOdd,paEven,paMark,paSpace);
  TStopbits = (sb1_0,sb1_5,sb2_0);
  TDatabits=(da4, da5, da6, da7, da8);

  TC32RxCharEvent = procedure(Sender: TObject; Count: Integer) of object;
  TC32StateEvent = procedure(Sender: TObject; State: Boolean) of object;
  TC32ErrorEvent = procedure(Sender: TObject; Errors: Integer) of object;
  TC32OpenEvent = procedure(Sender: TObject; Error: Integer) of object;

  TComm32 = class(TComponent)
  private
    FHandle: THandle;
    FCT: TComStat;
    FDCB: TDCB;
    FTC32Thread: TC32Thread;
    FDeviceName: String;
    FBaudRate: TBaudRate;
    FParity: TParity;
    FStopbits: TStopbits;
    FDatabits: TDatabits;
    FMonitorEvents: TC32EventType;
    FReadBufferSize: Integer;
    FWriteBufferSize: Integer;

    FOnOpen: TC32OpenEvent;
    FOnBreakSignal: TNotifyEvent;
    FOnCTSSignal: TC32StateEvent;
    FOnDSRSignal: TC32StateEvent;
    FOnErrorSignal: TC32ErrorEvent;
    FOnRingSignal: TNotifyEvent;
    FOnRLSDSignal: TNotifyEvent;
    FOnRxCharSignal: TC32RxCharEvent;
    FOnRxFlagSignal: TNotifyEvent;
    FOnTxEmptySignal: TNotifyEvent;

    procedure HandleTC32Event(Sender: TObject; Status: dword);
    procedure SetBaudRate(Value: TBaudRate);
    procedure SetParity(Value: TParity);
    procedure SetStopbits(Value: TStopBits);
    procedure SetDatabits(Value: TDatabits);
  protected
    {Protected declarations}
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Open;
    procedure Close;
    function Enabled: Boolean;
    function Write(const Buf; Count: Integer): Integer;
    function Read(var Buf; Count: Integer): Integer;
    function InQueCount: Integer;
    function OutQueCount: Integer;
    procedure PurgeIn;
    procedure PurgeOut;
    function DTR(State: Boolean): boolean;
    function RTS(State: Boolean): boolean;
    function BREAK(State: Boolean): boolean;
  published
    property DeviceName: string read FDeviceName write FDeviceName;
    property BaudRate: TBaudRate read FBaudRate write SetBaudRate;
    property Parity: TParity read FParity write SetParity;
    property Stopbits: TStopbits read FStopbits write SetStopbits;
    property Databits: TDatabits read FDatabits write SetDatabits;
    property MonitorEvents: TC32EventType read FMonitorEvents write FMonitorEvents;
    property ReadBufferSize: Integer read FReadBufferSize write FReadBufferSize;
    property WriteBufferSize: Integer read FWriteBufferSize write FWriteBufferSize;

    {Comm signal events}
    property OnOpen: TC32OpenEvent read FOnOpen write FOnOpen;
    property OnBreakSignal: TNotifyEvent read FOnBreakSignal write FOnBreakSignal;
    property OnCTSSignal: TC32StateEvent read FOnCTSSignal write FOnCTSSignal;
    property OnDSRSignal: TC32StateEvent read FOnDSRSignal write FOnDSRSignal;
    property OnErrorSignal: TC32ErrorEvent read FOnErrorSignal write FOnErrorSignal;
    property OnRingSignal: TNotifyEvent read FOnRingSignal write FOnRingSignal;
    property OnRLSDSignal: TNotifyEvent read FOnRLSDSignal write FOnRLSDSignal;
    property OnRxCharSignal: TC32RxCharEvent read FOnRxCharSignal write FOnRxCharSignal;
    property OnRxFlagSignal: TNotifyEvent read FOnRxFlagSignal write FOnRxFlagSignal;
    property OnTxEmptySignal: TNotifyEvent read FOnTxEmptySignal write FOnTxEmptySignal;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Varian', [TComm32]);
end;

{TC32Thread}

constructor TC32Thread.Create(Handle: Integer; Events: TC32EventType);
const
  EvList: array[TC32EventState] of dword = (EV_BREAK, EV_CTS, EV_DSR, EV_ERR,
    EV_RING, EV_RLSD, EV_RXCHAR, EV_RXFLAG, EV_TXEMPTY);
var
  EvIndex: TC32EventState;
  AttrWord: dword;
begin
  Inherited Create(false);
  Priority := tpHighest;
  FHandle := Handle;
  AttrWord := 0;
  for EvIndex := evBREAK to evTXEMPTY do
    if EvIndex in Events then
      AttrWord := AttrWord or EvList[EvIndex];
  SetCommMask(FHandle, AttrWord);
end;

destructor TC32Thread.Destroy;
begin
  CloseHandle(hCloseEvent);
  Inherited Destroy;
end;

procedure TC32Thread.Execute;
var
  HandlesToWaitFor: array[0..2] of THandle;
  dwHandleSignaled: DWORD;
  BytesTransferred: DWORD; //Dummy, not valid for WaitCommEvent
  OverlappedCommEvent: TOverlapped;
begin
  FillChar(OverlappedCommEvent, Sizeof(OverlappedCommEvent), 0);
  hCloseEvent := CreateEvent(nil, True, False, nil);
  OverlappedCommEvent.hEvent := CreateEvent(nil, True, True, nil);
  HandlesToWaitFor[0] := hCloseEvent;
  HandlesToWaitFor[1] := OverlappedCommEvent.hEvent;

  repeat
    WaitCommEvent(FHandle, FStatus, @OverlappedCommEvent);
    dwHandleSignaled := WaitForMultipleObjects(2, @HandlesToWaitFor, False, INFINITE);
    case dwHandleSignaled of
      WAIT_OBJECT_0    : Break;
      WAIT_OBJECT_0 + 1:
          if GetOverlappedResult(FHandle, OverlappedCommEvent,
             BytesTransferred, false) then Synchronize(DoOnSignal);
      else
        Break  {This should never occur}
    end;
  until Terminated;
  PurgeComm(FHandle, PURGE_RXABORT + PURGE_RXCLEAR);
  CloseHandle(OverlappedCommEvent.hEvent);
end;

procedure TC32Thread.ReleaseThread;
begin
  SetEvent(hCloseEvent);
end;

procedure TC32Thread.DoOnSignal;
begin
  if Assigned(FOnSignal) then FOnSignal(Self, FStatus);
end;

{TComm32}

constructor TComm32.Create(AOwner: TComponent);
begin
  Inherited Create(AOwner);
  FHandle := INVALID_HANDLE_VALUE;
  FDeviceName := 'COM2';
  FBaudRate := cbr2400;
  FParity := paNone;
  FStopbits := sb1_0;
  FDatabits := da8;
  FMonitorEvents := [evBreak, evCTS, evDSR, evError, evRing,
                     evRlsd, evRxChar, evRxFlag, evTxEmpty];
  FReadBufferSize := 4096;
  FWriteBufferSize := 2048;
end;

destructor TComm32.Destroy;
begin
  Close;
  Inherited Destroy;
end;

function TComm32.Enabled: Boolean;
begin
  Result := FHandle <> INVALID_HANDLE_VALUE;
end;

procedure TComm32.Open;
var
  CommTimeouts: TCommTimeouts;
begin
  Close;
  FHandle := CreateFile(PChar(FDeviceName), GENERIC_READ or GENERIC_WRITE,
                        0, nil, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, 0);
  if Enabled then
  begin
    SetupComm(FHandle, FReadBufferSize, FWriteBufferSize);

    GetCommTimeOuts(FHandle, CommTimeouts);
    CommTimeouts.ReadIntervalTimeout := 250;
    CommTimeouts.ReadTotalTimeoutMultiplier := 0;
    CommTimeouts.ReadTotalTimeoutConstant := 0;
    CommTimeouts.WriteTotalTimeoutMultiplier := 0;
    CommTimeouts.WriteTotalTimeoutConstant := 0;
    SetCommTimeouts(FHandle, CommTimeouts);

    SetBaudrate(FBaudrate);
    SetParity(FParity);
    SetStopbits(FStopbits);
    SetDatabits(FDatabits);

    FTC32Thread := TC32Thread.Create(FHandle, FMonitorEvents);
    FTC32Thread.OnSignal := HandleTC32Event;
  end;
  if assigned(FOnOpen) then
     FOnOpen(self, GetLastError);
end;

procedure TComm32.Close;
begin
  if Enabled then
  begin
    if FTC32Thread <> nil then
      FTC32Thread.ReleaseThread;
    FTC32Thread.Free;
    CloseHandle(FHandle);
    FHandle := INVALID_HANDLE_VALUE;
  end;
end;

function TComm32.Write(const Buf; Count: Integer): Integer;
var
  OS: TOverlapped;
begin
  Fillchar(OS, Sizeof(OS), 0);
  if not WriteFile(FHandle, Buf, Count, Result, @OS) then Result := -1;
end;

function TComm32.Read(var Buf; Count: Integer): Integer;
var
  OS: TOverlapped;
begin
  Fillchar(OS, Sizeof(OS), 0);
  if not ReadFile(FHandle, Buf, Count, Result, @OS) then Result := -1;
end;

{Errorflags for OnErrorSignal
 CE_BREAK       The hardware detected a break condition.
 CE_DNS	        Windows 95 only: A parallel device is not selected.
 CE_FRAME       The hardware detected a framing error.
 CE_IOE	        An I/O error occurred during communications with the device.
 CE_MODE        The requested mode is not supported, or the hFile parameter
                is invalid. If this value is specified, it is the only valid error.
 CE_OOP	        Windows 95 only: A parallel device signaled that it is out of paper.
 CE_OVERRUN     A character-buffer overrun has occurred. The next character is lost.
 CE_PTO	        Windows 95 only: A time-out occurred on a parallel device.
 CE_RXOVER      An input buffer overflow has occurred. There is either no
                room in the input buffer, or a character was received after
                the end-of-file (EOF) character.
 CE_RXPARITY    The hardware detected a parity error.
 CE_TXFULL      The application tried to transmit a character, but the output
                buffer was full.}

procedure TComm32.HandleTC32Event(Sender: TObject; Status: dword);
var
  Errors: dword;
begin
  ClearCommError(FHandle, Errors, @FCT);
  if Status and EV_BREAK > 0 then
    if assigned(FOnBreakSignal) then FOnBreakSignal(self);
  if Status and EV_CTS > 0 then
    if assigned(FOnCTSSignal) then FOnCTSSignal(self, not (fCtlHold in FCT.Flags));
  if Status and EV_DSR > 0 then
    if assigned(FOnDSRSignal) then FOnDSRSignal(self, not (fDsrHold in FCT.Flags));
  if Status and EV_ERR > 0 then
    if assigned(FOnErrorSignal) then FOnErrorSignal(self, Errors);
  if Status and EV_RING > 0 then
    if assigned(FOnRingSignal) then FOnRingSignal(self);
  if Status and EV_RLSD > 0 then
    if assigned(FOnRLSDSignal) then FOnRLSDSignal(self);
  if Status and EV_RXCHAR > 0 then
    if assigned(FOnRxCharSignal) then FOnRxCharSignal(self, FCT.cbInQue);
  if Status and EV_RXFLAG > 0 then
    if assigned(FOnRxFlagSignal) then FOnRxFlagSignal(self);
  if Status and EV_TXEMPTY > 0 then
    if assigned(FOnTxEmptySignal) then FOnTxEmptySignal(self);
end;

procedure TComm32.SetBaudRate(Value: TBaudRate);
const
  CBR: array[TBaudRate] of Integer = (CBR_110, CBR_300, CBR_600, CBR_1200, CBR_2400,
                       CBR_4800, CBR_9600, CBR_14400, CBR_19200, CBR_38400,
                       CBR_56000, CBR_57600, CBR_115200, CBR_128000, CBR_256000);
begin
  FBaudRate := Value;
  if Enabled then
  begin
    GetCommState(FHandle, FDCB);
    FDCB.BaudRate := CBR[FBaudRate];
    SetCommState(FHandle, FDCB);
  end;
end;

procedure TComm32.SetParity(Value: TParity);
const
  PAR: array[TParity] of byte = (NOPARITY, ODDPARITY, EVENPARITY,
                                 MARKPARITY, SPACEPARITY);
begin
  FParity := Value;
  if Enabled then
  begin
    GetCommState(FHandle, FDCB);
    FDCB.Parity := PAR[FParity];
    SetCommState(FHandle, FDCB);
  end;
end;

procedure TComm32.SetStopbits(Value: TStopbits);
const
  STB: array[TStopbits] of byte = (ONESTOPBIT, ONE5STOPBITS, TWOSTOPBITS);
begin
  FStopbits := Value;
  if Enabled then
  begin
    GetCommState(FHandle, FDCB);
    FDCB.Stopbits := STB[FStopbits];
    SetCommState(FHandle, FDCB);
  end;
end;

procedure TComm32.SetDataBits(Value: TDatabits);
const
  DAB: array[TDatabits] of byte = (4, 5, 6, 7, 8);
begin
  FDataBits:=Value;
  if Enabled then
  begin
    GetCommState(FHandle, FDCB);
    FDCB.Bytesize := DAB[FDatabits];
    SetCommState(FHandle, FDCB);
  end;
end;

function TComm32.InQueCount: Integer;
var
  Errors: dword;
begin
  ClearCommError(FHandle, Errors, @FCT);
  Result := FCT.cbInQue;
end;

function TComm32.OutQueCount: Integer;
var
  Errors: dword;
begin
  ClearCommError(FHandle, Errors, @FCT);
  Result := FCT.cbOutQue;
end;

procedure TComm32.PurgeIn;
begin
  PurgeComm(FHandle, PURGE_RXABORT or PURGE_RXCLEAR);
end;

procedure TComm32.PurgeOut;
begin
  PurgeComm(FHandle, PURGE_TXABORT or PURGE_TXCLEAR);
end;

function TComm32.DTR(State: boolean): boolean;
const
   DTR: array[boolean] of byte = (CLRDTR, SETDTR);
begin
  Result := EscapeCommFunction(FHandle, DTR[State]);
end;

function TComm32.RTS(State: boolean): boolean;
const
  RTS: array[boolean] of byte = (CLRRTS, SETRTS);
begin
  Result := EscapeCommFunction(FHandle, RTS[State]);
end;

function TComm32.BREAK(State: Boolean): boolean;
const
  BREAK: array[boolean] of byte = (CLRBREAK, SETBREAK);
begin
  Result := EscapeCommFunction(FHandle, BREAK[State]);
end;

end.
