unit TMWMsg;
(*********************************************************************************
  Version 1.3

  The TMessagePanel component is used simply to display messages on a panel
  for any form. Just tell it how to act and voila!, a nice little message
  display panel. There are settings for normal message display and for error
  message display. You can set the font and color scheme of each one. You can
  also control the panels size and position manually if you like. It has options
  that allow it to center itself on it's parent control and also resize itself
  based on the displayed string. If you turn off the automatic position or sizing,
  then you must set the associated properties (height, width, top, left) manually.

  There are 2 ways to display information, using either the DisplayInfo
  procedure or the DisplayInfoPause procedure. The DisplayInfo procedure
  takes 2 parameters: s: string; Error: boolean. If the string passed is empty,
  the panel is hidden. If it isn't blank, then the message will be displayed
  using the font and color based on the Error parameter. Error = true will
  use the error font and color.

  Here are some examples to go by:

       ex: Normal message displays

          DisplayInfo('Querying database now...');
          .. do stuff here..
          DisplayInfo('');

       ex: Error message displays

          DisplayInfo('Error while querying data...', true);
          .. do stuff here ..
          DisplayInfo('');

       ex: display message for 2.5 seconds

          DisplayInfoPause('Please rethink your decision.', 2.5);

       ex: display error message for 5 seconds

          DisplayInfoPause('Error, Please leave me alone now.', 5, true);

  Fixes and modifications:

    10/4/01
      1. Fixed bug dealing with showing blinking dots. It should
         now position correctly.
      2. Added Dots property so you can set your own dots or whatever
         characters you want to have blinking at the end of the message

    10/22/01
      1. Added Boundaries property to set the centering restrictions
         to a given area of the parent. For instance, to center the
         display panel in the lower third of a form, you could set the
         Top to Trunc(form.height * 2/3) and the Bottom to form.height.
      2. Added Bevel properties to control the panels borders. You can
         set the inner, outer and width of the panel bevel.

*********************************************************************************)
interface
  uses
    Windows, Forms, Graphics, SysUtils, Classes, Controls, ExtCtrls, StdCtrls;

type

  { The TBoundarySize and TBoundaries code is based on the Delphi
    Constraints property for TControl. It is used to specify the
    boundaries for centering the message panel within a particular
    area of the form.
  }
  TBoundarySize = 0..MaxInt;

  TBoundaries = class(TPersistent)
  private
    FControl: TControl;
    FLeft: TBoundarySize;
    FTop: TBoundarySize;
    FRight: TBoundarySize;
    FBottom: TBoundarySize;
    procedure SetBoundaries(Index: Integer; Value: TBoundarySize);
  public
    constructor Create(Control: TControl); virtual;
  published
    property Left: TBoundarySize index 0 read FLeft write SetBoundaries default 0;
    property Top: TBoundarySize index 1 read FTop write SetBoundaries default 0;
    property Right: TBoundarySize index 2 read FRight write SetBoundaries default 0;
    property Bottom: TBoundarySize index 3 read FBottom write SetBoundaries default 0;
  end;

  TMessagePanel = class(TComponent)
  private
    fOwner                          : TComponent;
    DisplayPanel                    : TPanel;
    fDots,
    fMessageInfo                    : string;
    fErrorCode, fAutoCenter,
    fAutoSize, fKeepPause,
    fDotsOn                         : boolean;
    eColor, fColor                  : TColor;
    fDotFont, eFont, fFont          : TFont;
    fMargin, fLeft, fTop, fHeight,
    fMaxPause, fMinWidth, fWidth,
    fBevelWidth                     : integer;
    fHeightMultiple, fDotsInterval  : double;
    fOnDestroy, fOnChange           : TNotifyEvent;
    fDotLabel                       : TLabel;
    fTimer                          : TTimer;
    fBoundaries                     : TBoundaries;
    fBevelInner, fBevelOuter        : TPanelBevel;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    { DisplayInfo(s: string): used to display any string s, the optional
       Error parameter tells it to use standard (default) font and
       colors or error settings. To clear the message and hide the
       message panel, call DisplayInfo with a blank string
    }
    procedure DisplayInfo(s: string; Error: boolean = false);
    { DisplayInfoPause(s: string; m: double): used to display a string s
       for a given m number of seconds, it will then reset the message
       to whatever it was before calling DisplayInfoPause. If there was
       no previous message being displayed, then it will hide the panel
    }
    procedure DisplayInfoPause(s: string; m: double; Error: boolean = false);
    { GetDisplayInfo: string: used to retrieve the current string being
       displayed
    }
    function GetDisplayInfo: string;
    { PanelParent: returns the owner of the TMessagePanel component
    }
    property PanelParent: TComponent read fOwner write fOwner;
    { HeightMultiple: used to set the height of the message panel based
        on the height of the font being displayed. (Default = 2.5)
        I have found that a minimum of 1.4 is needed for descending
        letters such a g, y
    }
    property HeightMultiple: double read fHeightMultiple write fHeightMultiple;

  private
    { DotTimer is the procedure called by the timer component to
      hide/unhide the dots
    }
    procedure DotTimer(Sender: TObject);

  published
    { PUBLISHED PROPERTIES
        AutoCenter: set to true to let component center itself
        AutoSize: set to true to let component size itself based on text
        BevelWidth: set the bevel width of the displayed panel
        Boundaries: used to control the placement of the message panel
                    within a certain area of the parent form. If AutoCenter
                    is true and the boundaries are set, the panel is
                    centered in this area.
        Color: the panel color for normal messages
        Dots: string of characters to blink at end of text (default "...")
        DotsInterval: seconds to wait between blinking dots
        DotsOn: true/false depending on whether to show blinking dots
        ErrorColor: the color of the panel for error messages
        ErrorFont: the font for error messages
        Font: the font for normal messages
        KeepOnPause: option to keep existing message after using the pause method
        PanelHeight: height of the display panel if manually controlling it
        PanelLeft: left side of the panel if manually positioning it
        Margin: pixels to provide spacing for messages when AutoSize is true
                15 is the minimum used when DotsOn is true to ensure enough
                space between the edges of the panel and the text for the "..."
        MaxPause: maximum number os seconds to use when DisplayInfoPause is
                  used. This keeps you from typing errors in your code and
                  having a message displayed indefinately. (default 10 seconds)
                  Set this to zero to ignore the maximum seconds.
        MinWidth: minimum width of panel when AutoSize is true (default 300 pixels)
        PanelTop: top of the panel when manually controlling it
        PanelWidth: width of the panel when manually controlling it
    }
    property AutoCenter: boolean read fAutoCenter write fAutoCenter;
    property AutoSize: boolean read fAutoSize write fAutoSize;
    property BevelInner: TPanelBevel read fBevelInner write fBevelInner;
    property BevelOuter: TPanelBevel read fBevelOuter write fBevelOuter;
    property BevelWidth: integer read fBevelWidth write fBevelWidth;
    property Boundaries: TBoundaries read fBoundaries write fBoundaries;
    property Color: TColor read fColor write fColor;
    property Dots: string read fDots write fDots;
    property DotFont: TFont read fDotFont write fDotFont;
    property DotsInterval: double read fDotsInterval write fDotsInterval;
    property DotsOn: boolean read fDotsOn write fDotsOn;
    property ErrorColor: TColor read eColor write eColor;
    property ErrorFont: TFont read eFont write eFont;
    property Font: TFont read fFont write fFont;
    property KeepOnPause: boolean read fKeepPause write fKeepPause;
    property PanelHeight: integer read fHeight write fHeight;
    property PanelLeft: integer read fLeft write fLeft;
    property Margin: integer read fMargin write fMargin;
    property MaxPause: integer read fMaxPause write fMaxPause;
    property MinWidth: integer read fMinWidth write fMinWidth;
    property PanelTop: integer read fTop write fTop;
    property PanelWidth: integer read fWidth write fWidth;
    property OnChange: TNotifyEvent read fOnChange write fOnChange;
    property OnDestroy: TNotifyEvent read fOnDestroy write fOnDestroy;
  end;

procedure Register;

implementation

var
  AreDotsOn: boolean;

{ You can change the location this component gets installed if you
  want it somewhere besides the Additional tab. }
procedure Register;
begin
  RegisterComponents('Additional', [TMessagePanel]);
end;

{ TBoundaries }

constructor TBoundaries.Create(Control: TControl);
begin
  inherited Create;
  FControl := Control;
end;

procedure TBoundaries.SetBoundaries(Index: Integer;
  Value: TBoundarySize);
begin
  case Index of
    0:
      if Value <> FLeft then
        FLeft := Value;
    1:
      if Value <> FTop then
        FTop := Value;
    2:
      if Value <> FRight then
        FRight := Value;
    3:
      if Value <> FBottom then
        FBottom := Value;
  end;
end;

{ TMessagePanel }

constructor TMessagePanel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fOwner := AOwner;
  fAutoSize := true;
  fAutoCenter := true;
  fDotFont := TFont.Create;
  eFont := TFont.Create;
  fFont := TFont.Create;
  fBoundaries := TBoundaries.Create(nil);
  fDotFont.Name := 'Arial';
  fDotFont.Size := 14;
  fDotFont.Style := [fsBold];
  fDotFont.Color := clYellow;
  fFont.Assign(fDotFont);
  eFont.Assign(fDotFont);
  eFont.Color := clWhite;
  eColor := clRed;
  fColor := clNavy;
  fHeight := 80;
  fWidth := 260;
  fMargin := 15;
  fMaxPause := 10;
  fMinWidth := 300;
  fBevelWidth := 3;
  fHeightMultiple := 2.5;
  fErrorCode := false;
  fKeepPause := true;
  fDotsOn := false;
  fDots := '...';
  fTimer := nil;
  fDotLabel := nil;
  fDotsInterval := 0.5;
  fBevelInner := bvLowered;
  fBevelOuter := bvRaised;
  DisplayPanel := nil;
end;

destructor TMessagePanel.Destroy;
begin
  if Assigned(fOnDestroy) then
    fOnDestroy(self);
  if Assigned(fDotLabel) then
    fDotLabel.Free;
  if Assigned(fTimer) then
    fTimer.Free;
  if Assigned(DisplayPanel) then
    DisplayPanel.Free;
  fBoundaries.Free;
  fDotFont.Free;
  eFont.Free;
  fFont.Free;
  inherited;
end;

procedure TMessagePanel.DisplayInfo(s: string; Error: boolean);
var
  x: integer;
  GoodOwner: boolean;

  { GetTextWidth returns the width of AText based on the font AFont }
  function GetTextWidth(const AText: string; AControl: TControl;
                        AFont: TFont): integer;
  begin
    with TControlCanvas.Create do
      try
        Control := AControl;
        Font.Assign(AFont);
        Result := TextWidth(AText);
      finally
        Free;
      end;
  end;

  { GetTextHeght returns the height of AText based on the font AFont }
  function GetTextHeight(const AText: string; AControl: TControl;
                        AFont: TFont): integer;
  begin
    with TControlCanvas.Create do
      try
        Control := AControl;
        Font.Assign(AFont);
        Result := TextHeight(AText);
      finally
        Free;
      end;
  end;

begin
  fMessageInfo := s;
  fErrorCode := Error;
  if Assigned(fOnChange) then
    fOnChange(self);
  { if message is blank, free up the panel }
  if (s = '') then begin
    if Assigned(fDotLabel) then
      fDotLabel.Free;
    fDotLabel := nil;
    DisplayPanel.Free;
    DisplayPanel := nil;
    if Assigned(fTimer) then begin
      fTimer.Enabled := false;
      fTimer.OnTimer := nil;
      fTimer.Free;
      fTimer := nil;
    end;
    exit;
  end;
  { make sure the owner is a valid window control }
  GoodOwner := Assigned(fOwner) and (fOwner is TControl);
  { if the panel isn't currently being shown, create it }
  if not Assigned(DisplayPanel) then begin
    DisplayPanel := TPanel.Create(nil);
    with DisplayPanel do begin
      Top := 0;
      Left := 0;
      Height := 1;
      Width := 1;
      BevelInner := fBevelInner;
      BevelOuter := fBevelOuter;
      BevelWidth := fBevelWidth;
      if GoodOwner then
        Parent := TWinControl(fOwner);
      Enabled := false;
      BringToFront;
    end;
  end;
  { set the correct font for the current message }
  if Error then begin
    DisplayPanel.Color := eColor;
    DisplayPanel.Font.Assign(eFont);
  end else begin
    DisplayPanel.Color := fColor;
    DisplayPanel.Font.Assign(fFont);
  end;
	{ set panel caption }
	DisplayPanel.Caption := s;
  Application.ProcessMessages;
  { if automatic sizing is on, calculate the width and height needed }
  if fAutoSize and GoodOwner then begin
    DisplayPanel.Height := Trunc(GetTextHeight(s, DisplayPanel,
        DisplayPanel.Font) * fHeightMultiple) + (DisplayPanel.BevelWidth * 2);
    x := GetTextWidth(s, DisplayPanel, DisplayPanel.Font);
    if x < fMinWidth then
  	  x := fMinWidth;
    if not DotsOn then
 	    DisplayPanel.Width := x + ((DisplayPanel.BevelWidth +fMargin) * 2)
    else
 	    DisplayPanel.Width := x + ((DisplayPanel.BevelWidth +fMargin) * 2) +
            GetTextWidth(fDots+fDots, DisplayPanel, fDotFont);
    if DisplayPanel.Width > TControl(fOwner).Width then
      DisplayPanel.Width := TControl(fOwner).Width;
  end else begin
    DisplayPanel.Height := fHeight;
    DisplayPanel.Width := fWidth;
  end;
  { if automatic centering is on, calculate the top and left properties
    based on whether any boundaries exist }
  if fAutoCenter and GoodOwner then begin
    if ((fBoundaries.Left > 0) or (fBoundaries.Right > 0)) and
        (fBoundaries.Left < fBoundaries.Right) then
      DisplayPanel.Left := fBoundaries.Left +
        (((fBoundaries.Right - fBoundaries.Left) - DisplayPanel.Width) div 2)
    else
      DisplayPanel.Left := ((TControl(fOwner).Width - DisplayPanel.Width) div 2);
    if ((fBoundaries.Top > 0) or (fBoundaries.Bottom > 0)) and
        (fBoundaries.Top < fBoundaries.Bottom) then
      DisplayPanel.Top := fBoundaries.Top +
        (((fBoundaries.Bottom - fBoundaries.Top) - DisplayPanel.Height) div 2)
    else
      DisplayPanel.Top := ((TControl(fOwner).Height - DisplayPanel.Height) div 2);
  end else begin
    DisplayPanel.Left := fLeft;
    DisplayPanel.Top := fTop;
  end;
  { show the panel if it has a valid owner }
  if GoodOwner then
    DisplayPanel.Show;
  Application.ProcessMessages;
  { if using blinking dots, set up the dot label }
  if fDotsOn then begin
    if not Assigned(fDotLabel) then begin
      fDotLabel := TLabel.Create(nil);
      fDotLabel.Parent := DisplayPanel;
      fDotLabel.Caption := fDots;
      fDotLabel.Autosize := true;
      fDotLabel.Alignment := taRightJustify;
      fDotLabel.Visible := false;
    end;
    { place the dots at the end of the message }
    fDotLabel.Font.Assign(fDotFont);
    fDotLabel.Left := DisplayPanel.Width -
          (fMargin + GetTextWidth(Dots,
            DisplayPanel, DisplayPanel.Font));
    { center the label vertically on the panel }
    fDotLabel.Top := (DisplayPanel.Height - fDotLabel.Height) div 2;
    if not Assigned(fTimer) then begin
      fTimer := TTimer.Create(self);
      fTimer.Interval := Trunc(fDotsInterval * 1000);
      fTimer.OnTimer := DotTimer;
      fTimer.Enabled := true;
    end;
  end;
  Application.ProcessMessages;
end;

procedure TMessagePanel.DisplayInfoPause(s: string; m: double; Error: boolean);
var
  oldError: boolean;
  oldMsg: string;
  n: TDateTime;

begin
  oldError := fErrorCode;
  oldMsg := fMessageInfo;
  DisplayInfo(s, Error);
  { calculate the time from now based on pause time desired }
  if (m > fMaxPause) and (fMaxPause > 0) then
    m := fMaxPause;
  m := m * (1 / (60 * 60 * 24));
  n := Now + m;
  { pause for desired time period }
  while (Now < n) do
    Application.ProcessMessages;
  { if KeepOnPause is true, reset the message to what it was before }
  if fKeepPause then
    DisplayInfo(oldMsg, oldError)
  else
    DisplayInfo('');
end;

function TMessagePanel.GetDisplayInfo: string;
begin
  result := fMessageInfo;
end;

procedure TMessagePanel.DotTimer(Sender: TObject);
begin
  { timer to display blinking dots }
  if Assigned(fDotLabel) then
    fDotLabel.Visible := AreDotsOn;
  { toggle the dot indicator }
  AreDotsOn := not AreDotsOn;
end;
end.
