{Author:	Poul Bak}
{Copyright  2000-2001 : BakSoft-Denmark (Poul Bak). All rights reserved.}
{http://home11.inet.tele.dk/BakSoft/}
{Mailto: baksoft-denmark@dk2net.dk}
{}
{Component Version: 4.00.00.00}
{}
{TPBPrinterSetupDialog is a TPrinterSetupDialog component with capability
of getting/setting printersetup-values (orientation, papersize etc) at
designtime and runtime. Settings can be saved so users don't have to setup
the printer every time they run your program.}
{Standard PrinterSetupDialog is used as propertyeditor.}
{At runtime you can switch between 4 setuptypes: stDefault (like standard
PrinterSetupDialog), stInitial (settings set at designtime), stSaved (user
defined settings saved) and stUser (settings set when the dialog has executed).}
{AutoSave and ForceInitialSetupValues.}
//
{How to use: First set the InitialSetupOptions (the values you want to change
from default).}
{Then either: Click the ellipse-button by InitialSetupValues (or double-click
the component on the form) to launch a PrinterSetupDialog, change the values
there and press 'Ok'.}
{Or: Double-click +InitialSetupValues to expand the sub-properties. Set the
values by picking from the lists or enter your own values.}
{That's enough - use the other properties if you like.}
//
{Note: When you have selected the PBPrinterSetupDialog component on the form,
the printersettings will follow the settings of the component until you deselect
the component again (or close your project). Pay attention if you print
from Delphi (deselect the component before you press 'Print' to restore the
original settings in Delphi.}

unit PBPrinterSetupDialog;

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
	Printers, WinSpool, DsgnIntf;

type
{The types for SetupType.}
	TSetupType = (stDefault, stInitial, stSaved, stUser);
{TInitialSetup is used as a set in TInitialSetupOptions.}
{Used by InitialSetupOptions and ThisPrinterOptions.}
	TInitialSetup = (isCollate, isColor, isCopies, isDefaultSource, isDitherType,
	isDuplex,	isFormName, isICMIntent, isICMMethod, isMediatype, isOrientation,
	isPaperLength, isPaperSize,	isPaperWidth, isPrinterName, isPrintQuality,
	isScale, isTTOption, isYResolution);
{See TInitialSetup.}
	TInitialSetupOptions = set of TInitialSetup;
{A 'dummy' type definition necessary so that the property editor
can show a list of values.}
	TSmallIntWithList = SmallInt;
{A 'dummy' type definition necessary so that the property editor
can show a list of values.}
	TCardinalWithList = Cardinal;
{A 'dummy' type definition necessary so that the property editor
can show a list of values.}
	TStringWithList = string;

{TInitialSetupValues is the class that contains the printersetup properties.}
{See InitialSetupValues and ThisPrinterValues.}
{See 'DEVMODE' in Delphi help.}
	TInitialSetupValues = class(TPersistent)
	private
		{ Private declarations }
		FCollate,	FColor : TSmallIntWithList;
		FComponent : TComponent;
		FCopies : SmallInt;
		FDefaultSource : TSmallIntWithList;
		FDitherType : TCardinalWithList;
		FDuplex : TSmallIntWithList;
		FFormName : TStringWithList;
		FICMIntent, FICMMethod, FMediatype : TCardinalWithList;
		FOrientation, FPaperSize : TSmallIntWithList;
		FPaperLength, FPaperWidth, FScale, FYResolution : SmallInt;
		FPrinterName : TStringWithList;
		FPrintQuality, FTTOption : TSmallIntWithList;
		function StoreFormName : Boolean;
		procedure SetCollate(Value : TSmallIntWithList);
		procedure SetColor(Value : TSmallIntWithList);
		procedure SetCopies(Value : SmallInt);
		procedure SetDefaultSource(Value : TSmallIntWithList);
		procedure SetDitherType(Value : TCardinalWithList);
		procedure SetDuplex(Value : TSmallIntWithList);
		procedure SetFormName(Value : TStringWithList);
		procedure SetICMIntent(Value : TCardinalWithList);
		procedure SetICMMethod(Value : TCardinalWithList);
		procedure SetMediaType(Value : TCardinalWithList);
		procedure SetOrientation(Value : TSmallIntWithList);
		procedure SetPaperLength(Value : SmallInt);
		procedure SetPaperSize(Value : TSmallIntWithList);
		procedure SetPaperWidth(Value : SmallInt);
		procedure SetPrinterName(Value : TStringWithList);
		procedure SetPrintQuality(Value : TSmallIntWithList);
		procedure SetScale(Value : SmallInt);
		procedure SetTTOption(Value : TSmallIntWithList);
		procedure SetYResolution(Value : SmallInt);
	published
{The way the printer handles several copies.}
{Note: Since DMCOLLATE_FALSE is equal to 0 it will show in InitialSetupValues
even though Collate is not included in InitialSetupOptions.}
		property Collate : TSmallIntWithList read FCollate write SetCollate default 0;
{Printing in color or monocrome.}
		property Color : TSmallIntWithList read FColor write SetColor default 0;
{The number of copies the printer produces.}
		property Copies : SmallInt read FCopies write SetCopies default 0;
{The bin the printer gets the paper from.}
		property DefaultSource : TSmallIntWithList read FDefaultSource write SetDefaultSource default 0;
{The way the printer handles dithering.}
		property DitherType : TCardinalWithList read FDitherType write SetDitherType default 0;
{The way the printer handles printing on both sides.}
		property Duplex : TSmallIntWithList read FDuplex write SetDuplex default 0;
{ FormName is the name of PaperSize.}
{Note: FormName is localized by Windows. Do not use if you plan to export your
application.}
{See PaperSize - PaperHeight - PaperWidth}
		property FormName : TStringWithList read FFormName write SetFormName stored StoreFormName;
{Image color matching (ICM) profile.}
		property ICMIntent : TCardinalWithList read FICMIntent write SetICMIntent default 0;
{Image color matching (ICM) method.}
		property ICMMethod : TCardinalWithList read FICMMethod write SetICMMethod default 0;
{The kind of paper being used.}
		property Mediatype : TCardinalWithList read FMediatype write SetMediatype default 0;
{Orientation is either portrait (default for most printers) or landscape.}
		property Orientation : TSmallIntWithList read FOrientation write SetOrientation default 0;
{See PaperSize.}
		property PaperLength : SmallInt read FPaperLength write SetPaperLength default 0;
{Set the papersize to one of the predefined values or use a custom value.}
{PaperSize works together with FormName and PaperHeight and PaperWidth.}
{Either set PaperSize or FormName or PaperHeight and PaperWidth.}
{If you specify more than one then the priority is:
{Priority: FormName has the lowest priority. Then PaperSize and finally
PaperHeight and PaperWidth.}
		property PaperSize : TSmallIntWithList read FPaperSize write SetPaperSize default 0;
{See PaperSize.}
		property PaperWidth : SmallInt read FPaperWidth write SetPaperWidth default 0;
{Set PrinterName property if you use a non-default printer to set up the
InitialSetupValues or if you want to use a particular printer at runtime.
Of course a printer of that name has to be installed on the users computer or
the default printer will be used.}
{If you specify a printer the name will be saved (even when not included in
InitialSetupOptions) to make designtime setup easier but it will only be used at
runtime, if you include the isPrinterName option.}
		property PrinterName : TStringWithList read FPrinterName write SetPrinterName;
{Either a predefined constant or a positive number indicating the Dots Per Inch
horizontally together with a positive number for YResolution indicating Dots Per
Inch vertically.}
		property PrintQuality : TSmallIntWithList read FPrintQuality write SetPrintQuality default 0;
{Scaling in percent of normal (default is 100).}
		property Scale : SmallInt read FScale write SetScale default 0;
{Indicates the way the printer handles True Type fonts.}
		property TTOption : TSmallIntWithList read FTTOption write SetTTOption default 0;
{See PrintQuality.}
		property YResolution : SmallInt read FYResolution write SetYResolution default 0;
	end;
{A record used internally.}
	TPrinterConfig = record
		PrinterName, Driver, Port : array[0..CCHDEVICENAME] of Char;
		SizeOfDeviceMode : Cardinal;
	end;

{Author:	Poul Bak}
{Copyright  2000-2001 : BakSoft-Denmark (Poul Bak). All rights reserved.}
{http://home11.inet.tele.dk/BakSoft/}
{Mailto: baksoft-denmark@dk2net.dk}
{}
{Component Version: 4.00.00.00}
{}
{TPBPrinterSetupDialog is a TPrinterSetupDialog component with capability
of getting/setting printersetup-values (orientation, papersize etc) at
designtime and runtime. Settings can be saved so users don't have to setup
the printer every time they run your program.}
{Standard PrinterSetupDialog is used as propertyeditor.}
{At runtime you can switch between 4 setuptypes: stDefault (like standard
PrinterSetupDialog), stInitial (settings set at designtime), stSaved (user
defined settings saved) and stUser (settings set when the dialog has exuted).}
{AutoSave and ForceInitialSetupValues.}
//
{How to use: First set the InitialSetupOptions (the values you want to change
from default).}
{Then either: Click the ellipse-button by InitialSetupValues (or double-click
the component on the form) to launch a PrinterSetupDialog, change the values
there and press 'Ok'.}
{Or: Double-click +InitialSetupValues to expand the sub-properties. Set the
values by picking from the lists or enter your own values.}
{That's enough - use the other properties if you like.}
//
{Note: When you have selected the PBPrinterSetupDialog component on the form,
the printersettings will follow the settings of the component until you deselect
the component again (or close your project). Pay attention if you print
from Delphi (deselect the component before you press 'Print' to restore the
original settings in Delphi.}

	TPBPrinterSetupDialog = class(TPrinterSetupDialog)
	private
		{ Private declarations }
		FFileName, TempFileName : TFileName;
		FAutoSave, FForceInitialSetupValues : Boolean;
		FPrinterRecord : TPrinterConfig;
		FConfigFile : file of TPrinterConfig;
		FDeviceModeFile : file of Char;
		PPrinterDevMode : PDevMode;
		DevModeHandle : THandle;
		FSetupType : TSetupType;
		FVersion, DefaultPrinter : string;
		FInitialSetupOptions, FThisPrinterOptions : TInitialSetupOptions;
		FInitialSetupValues, FThisPrinterValues : TInitialSetupValues;
		function InheritedExecute : Boolean;
		function GetPrinterSetup: Boolean;
		function SetPrinterSetup: Boolean;
		procedure Dummy(Value : string);
		procedure Dummy1(Value : TInitialSetupOptions);
		procedure Dummy2(Value : TInitialSetupValues);
		procedure SetSetupType(Value : TSetupType);
		procedure SetInitialSetupOptions(Value : TInitialSetupOptions);
		procedure SetInitialSetupValues;
		procedure SetThisPrinterValues;
		procedure SetFileName(Value : TFileName);
	public
		{ Public declarations }
		constructor Create(AOwner: TComponent); override;
		destructor Destroy; override;
		procedure Loaded; override;
{Call Execute at runtime to let user setup his/her printer.}
{See AutoSave and ForceInitialSetupValues.}
		function Execute: Boolean; override;
{Loads a saved Setup from SetupFileName. Includes printername and all settings.}
{Same as setting SetupType to stSaved.}
		procedure LoadSetup;
{Sets printername to the standard printer and sets the printersetup to the default values.}
{Same as setting SetupType to stDefault.}
		procedure DefaultSetup;
{Saves the current printersetup to SetupFileName. Includes printername and all settings.}
{See SetupType.}
		procedure SaveSetup;
{Sets the printersetup to the values you set at designtime.}
{Same as setting SetupType to stInitial.}
		procedure InitialSetup;
	published
{If True then printersetup will be automatically saved when PBPrinterSetupDialog
has Executed, so users don't have to setup the printer again. If False you can
manually call SaveSetup to save.}
		property AutoSave : Boolean read FAutoSave write FAutoSave default True;
{If True then users can only change the non-important values (the values you
don't include in InitialSetupOptions) because every time PBPrinterSetupDialog
has executed it calls InitialSetup.}
		property ForceInitialSetupValues : Boolean read FForceInitialSetupValues
			write FForceInitialSetupValues default False;
{A set of options that decides which setup-values that must be set and which
values should be default. Only options included here will be changed when
calling InitialSetup. Other options won't be affected.}
{You can only set 'public' values of the devmode structure here.}
{Note: FormName is localized by Windows. Do not use if you plan to export your
application.}
{See InitialSetupValues, ThisPrinterOptions and ThisPrinterValues.}
		property InitialSetupOptions : TInitialSetupOptions
			read FInitialSetupOptions write SetInitialSetupOptions default [];
{The actual values that you want to set. You can change the values either by
clicking the ellipse-button which will execute a standard PrinterSetupDialog
(the easy way) or by setting the individual values manually.}
{If you want to change a setting that your own printer(s) doesn't support,
you must set the value manually either by picking a value from the list or type
the value.}
{Changing a value at designtime will automatically update the current
printersettings. If you change a value at runtime the printer will only be
updated if SetupType is stInitial or you can call InitialSetup or set SetupType
to stInitial to update the printersettings.}
{Note: FormName is localized by Windows. Do not use if you plan to export your
application.}
{See InitialSetupOptions, ThisPrinterOptions and ThisPrinterValues.}
{See DEVMODE in Dephi help file to learn more.}
		property InitialSetupValues : TInitialSetupValues
			read FInitialSetupValues write Dummy2;
{The (path and) filename where the setup is saved. Both the printername and all
settings (both 'public' and 'private' members of the devmode structure) are saved.}
{Because of that it is very effective at runtime, but the file should not be
distributed with your application, unless you know that the user has the same
printer as you (in a network for instance).}
{Changing the filename at runtime gives you the possibility of having more than
one setup-file to use in different parts of your application.}
		property SetupFileName : TFileName read FFileName write SetFileName;
{Can have 4 values: stDefault, stInitial, stSaved and stUser.}
{Read to determine the current SetupType, Set to change the SetupType.}
{stDefault: The default setup-values. This is the default printer with default setup
 (both 'private' and 'public' settings are set to default values).}
{stInitial: The values you set in the Object Inspector at designtime. Setting
SetupType to stInitial changes only the values you have included in
InitialSetupOptions.}
{stSaved: Changes all values (incl. printername, 'private' and 'public' members
of the devmode structure) to the values saved in SetupFileName. If SetupFileName
doesn't exist or doesn't fit to one of the connected printers, setting SetupType
to stSaved will have no effect.}
{stUser: A readonly value set when the user might have changed the setup (by
executing PBPrinterSetupDialog) and AutoSave is False.}
{Note: Switching between different SetupTypes does not change InitialSetupValues,
the values stay intact unless you change them individually.}
		property SetupType : TSetupType read FSetupType write SetSetupType;
{A readonly property for informational purpose. Here you can see at designtime,
which Setup values are initialized on the current printer. These values can
normally also be set. Values that have not been initialized might still be available
 though. It depends on the printerdriver.}
		property ThisPrinterOptions : TInitialSetupOptions
			read FThisPrinterOptions write Dummy1 stored False;
{A readonly property for informational purpose. Here you can see at designtime
(and check at runtime) the actual Setup values which have been initialized on
the current printer. These values can normally also be set.}
		property ThisPrinterValues : TInitialSetupValues
			read FThisPrinterValues write Dummy2 stored False;
		property Version : string read FVersion write Dummy stored False;
	end;

procedure Register;

// -------------------- implementation -------------------------
implementation
{$I+}

const
	NUMBER_OF_FIELDS = 19;
	dmFieldConstants : array[0..NUMBER_OF_FIELDS - 1] of DWord = (DM_COLLATE,
		DM_COLOR, DM_COPIES, DM_DEFAULTSOURCE, DM_DITHERTYPE, DM_DUPLEX, DM_FORMNAME,
		DM_ICMINTENT, DM_ICMMETHOD, DM_MEDIATYPE, DM_ORIENTATION, DM_PAPERLENGTH,
		DM_PAPERSIZE,	DM_PAPERWIDTH, 0, DM_PRINTQUALITY, DM_SCALE, DM_TTOPTION,
		DM_YRESOLUTION);

constructor TPBPrinterSetupDialog.Create(AOwner : TComponent);
var
	TempPath, TempName : array[0..MAX_PATH] of Char;
begin
	inherited Create(AOwner);
	FVersion := '4.00.00.00';
	FFileName := 'PrinterSetup.cfg';
	if csDesigning in ComponentState then
	begin
		GetTempPath(MAX_PATH, TempPath);
		GetTempFileName(TempPath, 'PBP', 0, TempName);
		TempFileName := TFileName(TempName);
	end
	else TempFileName := FFileName;
	FAutoSave := True;
	FForceInitialSetupValues := False;
	FSetupType := stSaved;
	FInitialSetupOptions := [];
	FInitialSetupValues := TInitialSetupValues.Create;
	FInitialSetupValues.FComponent := Self;
	FThisPrinterValues := TInitialSetupValues.Create;
	FThisPrinterValues.FComponent := nil;
end;

procedure TPBPrinterSetupDialog.Loaded;
begin
	inherited Loaded;
	if (Printer.Printers.Count = 0) or (csDesigning in ComponentState) then Exit;
	if (FSetupType >= stInitial) then InitialSetup
	else if (FSetupType >= stSaved) then
	begin
		if FileExists(TempFileName) then LoadSetup
		else FSetupType := stInitial;
	end;
end;

destructor TPBPrinterSetupDialog.Destroy;
begin
	if (csDesigning in ComponentState) then DeleteFile(TempFileName);
	FInitialSetupValues.Free;
	FInitialSetupValues := nil;
	FThisPrinterValues.Free;
	FThisPrinterValues := nil;
	inherited Destroy;
end;

procedure TPBPrinterSetupDialog.Dummy(Value : string); begin end;
procedure TPBPrinterSetupDialog.Dummy1(Value : TInitialSetupOptions); begin end;
procedure TPBPrinterSetupDialog.Dummy2(Value : TInitialSetupValues); begin end;
function TInitialSetupValues.StoreFormName : Boolean;
begin
	Result := (FFormName <> '');
end;

procedure TPBPrinterSetupDialog.SaveSetup;
begin
	if Printer.Printers.Count = 0 then Exit;
	if GetPrinterSetup then
	begin
		AssignFile(FConfigFile, TempFileName);
		ReWrite(FConfigFile);
		Write(FConfigFile, FPrinterRecord);
		CloseFile(FConfigFile);
		AssignFile(FDeviceModeFile, TempFileName);
		Reset(FDeviceModeFile);
		Seek(FDeviceModeFile, FileSize(FDeviceModeFile));
		with FPrinterRecord do
		begin
			BlockWrite(FDeviceModeFile, PPrinterDevMode^, SizeOfDeviceMode);
		end;
		CloseFile(FDeviceModeFile);
		if not (csDesigning in ComponentState)then FSetupType := stSaved;
		GlobalUnLock(DevModeHandle);
	end;
end;

procedure TPBPrinterSetupDialog.LoadSetup;
var
	Index : integer;
	TempRecord : TPrinterConfig;
begin
	if Printer.Printers.Count = 0 then Exit;
	if FileExists(TempFileName) then
	begin
		AssignFile(FConfigFile, TempFileName);
		Reset(FConfigFile);
		Read(FConfigFile, TempRecord);
		CloseFile(FConfigFile);
		with TempRecord do
		begin
			Index := 0;
			while (Index < Printer.Printers.Count)
				and (AnsiUpperCase(Copy(Printer.Printers[Index], 1, StrLen(PrinterName)))
				<> AnsiUpperCase(PrinterName)) do Inc(Index);
		end;
		if Index < Printer.Printers.Count then
		begin
			FPrinterRecord := TempRecord;
			with FPrinterRecord do
			begin
				DevModeHandle:=GlobalAlloc(GHND, SizeOfDeviceMode);
				PPrinterDevMode := GlobalLock(DevModeHandle);
				AssignFile(FDeviceModeFile, TempFileName);
				Reset(FDeviceModeFile);
				Seek(FDeviceModeFile, SizeOf(FPrinterRecord));
				BlockRead(FDeviceModeFile, PPrinterDevMode^, SizeOfDeviceMode);
				CloseFile(FDeviceModeFile);
				SetPrinterSetup;
				if not (csDesigning in ComponentState)then FSetupType := stSaved;
			end;
		end;
	end;
end;

function TPBPrinterSetupDialog.Execute : Boolean;
begin
	if Printer.Printers.Count = 0 then Result := False
	else
	begin
		Result := inherited Execute;
		if Result then
		begin
			if FForceInitialSetupValues then InitialSetup;
			if FAutoSave then SaveSetup
			else FSetupType := stUser;
			SetThisPrinterValues;
		end;
	end;
end;

procedure TPBPrinterSetupDialog.DefaultSetup;
begin
	if Printer.Printers.Count = 0 then Exit;
	Printer.PrinterIndex := -1;
	SetThisPrinterValues;
	FSetupType := stDefault;
end;

function TPBPrinterSetupDialog.GetPrinterSetup : Boolean;
begin
	if Printer.Printers.Count = 0 then Result := False
	else
	begin
		Result := True;
		with FPrinterRecord do
		begin
			Printer.GetPrinter(PrinterName, Driver, Port, DevModeHandle);
			SizeOfDeviceMode := GlobalSize(DevModeHandle);
			if SizeOfDeviceMode = 0 then Result := False
			else
			begin
				PPrinterDevMode := GlobalLock(DevModeHandle);
				if PPrinterDevMode = nil then Result := False;
			end;
		end;
	end;
end;

function TPBPrinterSetupDialog.SetPrinterSetup : Boolean;
begin
	if Printer.Printers.Count = 0 then Result := False
	else
	begin
		with FPrinterRecord do Printer.SetPrinter(PrinterName, Driver, Port, DevModeHandle);
		GlobalUnLock(DevModeHandle);
		SetThisPrinterValues;
		Result := True;
	end;
end;

procedure TPBPrinterSetupDialog.SetSetupType(Value : TSetupType);
begin
	if FSetupType <> Value then FSetupType := Value;
	if (Printer.Printers.Count = 0) or (csDesigning in ComponentState) then Exit;
	if Value = stDefault then DefaultSetup
	else if Value = stInitial then
	begin
		InitialSetup;
		FSetupType := stInitial;
	end
	else if Value = stSaved then LoadSetup;
end;

procedure TPBPrinterSetupDialog.SetFileName(Value : TFileName);
begin
	if FFileName <> Value then
	begin
		FFileName := Value;
		if not (csDesigning in ComponentState) then TempFileName := Value;
	end;
end;

procedure TPBPrinterSetupDialog.InitialSetup;
var
	Index : integer;
	t : Byte;
begin
	if Printer.Printers.Count = 0 then Exit;
	if ((csDesigning in ComponentState) or (isPrinterName in FInitialSetupOptions))
		and (FInitialSetupValues.FPrinterName <> '') then
	begin
		Index := Printer.Printers.Count - 1;
		while (Index > -1)
			and (AnsiUpperCase(Copy(Printer.Printers[Index], 1,
			Length(FInitialSetupValues.FPrinterName)))
			<> AnsiUpperCase(FInitialSetupValues.FPrinterName)) do Dec(Index);
		Printer.PrinterIndex := Index;
	end;
	if GetPrinterSetup then with PPrinterDevMode^ do
	begin
		if isCollate in FInitialSetupOptions then dmCollate := FInitialSetupValues.FCollate;
		if isColor in FInitialSetupOptions then	dmColor := FInitialSetupValues.FColor;
		if isCopies in FInitialSetupOptions then dmCopies := FInitialSetupValues.FCopies;
		if isDefaultSource in FInitialSetupOptions then	dmDefaultSource := FInitialSetupValues.FDefaultSource;
		if isDitherType in FInitialSetupOptions then dmDitherType := FInitialSetupValues.FDitherType;
		if isDuplex in FInitialSetupOptions then dmDuplex := FInitialSetupValues.FDuplex;
		if isFormName in FInitialSetupOptions then StrPCopy(dmFormName, FInitialSetupValues.FFormName);
		if isICMIntent in FInitialSetupOptions then	dmICMIntent := FInitialSetupValues.FICMIntent;
		if isICMMethod in FInitialSetupOptions then	dmICMMethod := FInitialSetupValues.FICMMethod;
		if isMediatype in FInitialSetupOptions then dmMediatype := FInitialSetupValues.FMediatype;
		if isOrientation in FInitialSetupOptions then	dmOrientation := FInitialSetupValues.FOrientation;
		if isPaperLength in FInitialSetupOptions then	dmPaperLength := FInitialSetupValues.FPaperLength;
		if isPaperSize in FInitialSetupOptions then	dmPaperSize := FInitialSetupValues.FPaperSize;
		if isPaperWidth in FInitialSetupOptions then dmPaperWidth := FInitialSetupValues.FPaperWidth;
		if isPrintQuality in FInitialSetupOptions then dmPrintQuality := FInitialSetupValues.FPrintQuality;
		if isScale in FInitialSetupOptions then	dmScale := FInitialSetupValues.FScale;
		if isTTOption in FInitialSetupOptions then dmTTOption := FInitialSetupValues.FTTOption;
		if isYResolution in FInitialSetupOptions then dmYResolution := FInitialSetupValues.FYResolution;
		for t := 0 to NUMBER_OF_FIELDS - 1 do
		begin
			if TInitialSetup(t) in FInitialSetupOptions then dmFields := dmFields or dmFieldConstants[t];
		end;
		SetPrinterSetup;
	end;
end;

procedure TPBPrinterSetupDialog.SetInitialSetupValues;
begin
	if Printer.Printers.Count = 0 then Exit;
	if not GetPrinterSetup then Exit;
	if (isPrinterName in FInitialSetupOptions) or (FPrinterRecord.PrinterName
		<> DefaultPrinter) then	FInitialSetupValues.PrinterName :=
		FPrinterRecord.PrinterName
	else FInitialSetupValues.PrinterName := '';
	with PPrinterDevMode^ do
	begin
		if isCollate in FInitialSetupOptions then FInitialSetupValues.FCollate := dmCollate
		else FInitialSetupValues.FCollate := 0;
		if isColor in FInitialSetupOptions then FInitialSetupValues.FColor := dmColor
		else FInitialSetupValues.FColor := 0;
		if isCopies in FInitialSetupOptions then FInitialSetupValues.FCopies := dmCopies
		else FInitialSetupValues.FCopies := 0;
		if isDefaultSource in FInitialSetupOptions then FInitialSetupValues.FDefaultSource := dmDefaultSource
		else FInitialSetupValues.FDefaultSource := 0;
		if isDitherType in FInitialSetupOptions then FInitialSetupValues.FDitherType := dmDitherType
		else FInitialSetupValues.FDitherType := 0;
		if isDuplex in FInitialSetupOptions then FInitialSetupValues.FDuplex := dmDuplex
		else FInitialSetupValues.FDuplex := 0;
		if isFormName in FInitialSetupOptions then FInitialSetupValues.FFormName := dmFormName
		else FInitialSetupValues.FFormName := '';
		if isICMIntent in FInitialSetupOptions then FInitialSetupValues.FICMIntent := dmICMIntent
		else FInitialSetupValues.FICMIntent := 0;
		if isICMMethod in FInitialSetupOptions then FInitialSetupValues.FICMMethod := dmICMMethod
		else FInitialSetupValues.FICMMethod := 0;
		if isMediatype in FInitialSetupOptions then FInitialSetupValues.FMediatype := dmMediatype
		else FInitialSetupValues.FMediatype := 0;
		if isOrientation in FInitialSetupOptions then FInitialSetupValues.FOrientation := dmOrientation
		else FInitialSetupValues.FOrientation := 0;
		if isPaperLength in FInitialSetupOptions then FInitialSetupValues.FPaperLength := dmPaperLength
		else FInitialSetupValues.FPaperLength := 0;
		if isPaperSize in FInitialSetupOptions then FInitialSetupValues.FPaperSize := dmPaperSize
		else FInitialSetupValues.FPaperSize := 0;
		if isPaperWidth in FInitialSetupOptions then FInitialSetupValues.FPaperWidth := dmPaperWidth
		else FInitialSetupValues.FPaperWidth := 0;
		if isPrintQuality in FInitialSetupOptions then FInitialSetupValues.FPrintQuality := dmPrintQuality
		else FInitialSetupValues.FPrintQuality := 0;
		if isScale in FInitialSetupOptions then FInitialSetupValues.FScale := dmScale
		else FInitialSetupValues.FScale := 0;
		if isTTOption in FInitialSetupOptions then FInitialSetupValues.FTTOption := dmTTOption
		else FInitialSetupValues.FTTOption := 0;
		if isYResolution in FInitialSetupOptions then FInitialSetupValues.FYResolution := dmYResolution
		else FInitialSetupValues.FYResolution := 0;
	end;
	GlobalUnlock(DevModeHandle);
end;

procedure TPBPrinterSetupDialog.SetThisPrinterValues;
var
	t : Byte;
begin
	if Printer.Printers.Count = 0 then Exit;
	if not GetPrinterSetup then Exit;
	for t := 0 to NUMBER_OF_FIELDS - 1 do
	begin
		if (dmFieldConstants[t] and PPrinterDevMode^.dmFields) = dmFieldConstants[t]
			then FThisPrinterOptions := FThisPrinterOptions + [TInitialSetup(t)]
		else FThisPrinterOptions := FThisPrinterOptions - [TInitialSetup(t)];
	end;
	FThisPrinterValues.FPrinterName := FPrinterRecord.PrinterName;
	with PPrinterDevMode^ do
	begin
		FThisPrinterValues.FCollate := dmCollate;
		FThisPrinterValues.FColor := dmColor;
		FThisPrinterValues.FCopies := dmCopies;
		FThisPrinterValues.FDefaultSource := dmDefaultSource;
		FThisPrinterValues.FDitherType := dmDitherType;
		FThisPrinterValues.FDuplex := dmDuplex;
		FThisPrinterValues.FFormName := dmFormName;
		FThisPrinterValues.FICMIntent := dmICMIntent;
		FThisPrinterValues.FICMMethod := dmICMMethod;
		FThisPrinterValues.FMediatype := dmMediatype;
		FThisPrinterValues.FOrientation := dmOrientation;
		FThisPrinterValues.FPaperLength := dmPaperLength;
		FThisPrinterValues.FPaperSize := dmPaperSize;
		FThisPrinterValues.FPaperWidth := dmPaperWidth;
		FThisPrinterValues.FPrintQuality := dmPrintQuality;
		FThisPrinterValues.FScale := dmScale;
		FThisPrinterValues.FTTOption := dmTTOption;
		FThisPrinterValues.FYResolution := dmYResolution;
	end;
	GlobalUnlock(DevModeHandle);
end;

procedure TPBPrinterSetupDialog.SetInitialSetupOptions(Value : TInitialSetupOptions);
begin
	if FInitialSetupOptions <> Value then
	begin
		FInitialSetupOptions := Value;
		if not (csReading	in ComponentState) and (csDesigning in ComponentState)
			then with FInitialSetupValues do
		begin
			if not (isCollate in FInitialSetupOptions) then FCollate := 0
			else if FCollate = 0 then FCollate := FThisPrinterValues.FCollate;
			if not (isColor in FInitialSetupOptions) then FColor := 0
			else if FColor = 0 then FColor := FThisPrinterValues.FColor;
			if not (isCopies in FInitialSetupOptions) then FCopies := 0
			else if FCopies = 0 then FCopies := FThisPrinterValues.FCopies;
			if not (isDefaultSource in FInitialSetupOptions) then FDefaultSource := 0
			else if FDefaultSource = 0 then FDefaultSource := FThisPrinterValues.FDefaultSource;
			if not (isDitherType in FInitialSetupOptions) then FDitherType := 0
			else if FDitherType = 0 then FDitherType := FThisPrinterValues.FDitherType;
			if not (isDuplex in FInitialSetupOptions) then FDuplex := 0
			else if FDuplex = 0 then FDuplex := FThisPrinterValues.FDuplex;
			if not (isFormName in FInitialSetupOptions) then FFormName := ''
			else if FFormName = '' then FFormName := FThisPrinterValues.FFormName;
			if not (isICMIntent in FInitialSetupOptions) then FICMIntent := 0
			else if FICMIntent = 0 then FICMIntent := FThisPrinterValues.FICMIntent;
			if not (isICMMethod in FInitialSetupOptions) then FICMMethod := 0
			else if FICMMethod = 0 then FICMMethod := FThisPrinterValues.FICMMethod;
			if not (isMediaType in FInitialSetupOptions) then FMediaType := 0
			else if FMediaType = 0 then FMediaType := FThisPrinterValues.FMediatype;
			if not (isOrientation in FInitialSetupOptions) then FOrientation := 0
			else if FOrientation = 0 then FOrientation := FThisPrinterValues.FOrientation;
			if not (isPaperLength in FInitialSetupOptions) then FPaperLength := 0
			else if FPaperLength = 0 then FPaperLength := FThisPrinterValues.FPaperLength;
			if not (isPaperSize in FInitialSetupOptions) then FPaperSize := 0
			else if FPaperSize = 0 then FPaperSize := FThisPrinterValues.FPaperSize;
			if not (isPaperWidth in FInitialSetupOptions) then FPaperWidth := 0
			else if FPaperWidth = 0 then FPaperWidth := FThisPrinterValues.FPaperWidth;
			if not (isPrintQuality in FInitialSetupOptions) then FPrintQuality := 0
			else if FPrintQuality = 0 then FPrintQuality := FThisPrinterValues.FPrintQuality;
			if not (isScale in FInitialSetupOptions) then FScale := 0
			else if FScale = 0 then FScale := FThisPrinterValues.FScale;
			if not (isTTOption in FInitialSetupOptions) then FTTOption := 0
			else if FTTOption = 0 then FTTOption := FThisPrinterValues.FTTOption;
			if not (isYResolution in FInitialSetupOptions) then FYResolution := 0
			else if FYResolution = 0 then FYResolution := FThisPrinterValues.FYResolution;
		end;
		if not (csReading	in ComponentState) and ((csDesigning in ComponentState) or (FSetupType = stInitial)) then	InitialSetup;
	end;
end;

function TPBPrinterSetupDialog.InheritedExecute : Boolean;
begin
	InitialSetup;
	Result := inherited Execute;
	if Result then
	begin
		SetInitialSetupValues;
		SetThisPrinterValues;
	end;
end;

//  ------------------- TInitialSetupValues ----------------------
procedure TInitialSetupValues.SetCollate(Value : TSmallIntWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FCollate <> Value) and (isCollate in FInitialSetupOptions) then
		begin
			FCollate := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetColor(Value : TSmallIntWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FColor <> Value) and (isColor in FInitialSetupOptions) then
		begin
			FColor := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetCopies(Value : SmallInt);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FCopies <> Value) and (isCopies in FInitialSetupOptions) then
		begin
			FCopies := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetDefaultSource(Value : TSmallIntWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FDefaultSource <> Value) and (isDefaultSource in FInitialSetupOptions) then
		begin
			FDefaultSource := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetDitherType(Value : TCardinalWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FDitherType <> Value) and (isDitherType in FInitialSetupOptions) then
		begin
			FDitherType := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetDuplex(Value : TSmallIntWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FDuplex <> Value) and (isDuplex in FInitialSetupOptions) then
		begin
			FDuplex := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetFormName(Value : TStringWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FFormName <> Value) and (isFormName in FInitialSetupOptions) then
		begin
			FFormName := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetICMIntent(Value : TCardinalWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FICMIntent <> Value) and (isICMIntent in FInitialSetupOptions) then
		begin
			FICMIntent := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetICMMethod(Value : TCardinalWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FICMMethod <> Value) and (isICMMethod in FInitialSetupOptions) then
		begin
			FICMMethod := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetMediaType(Value : TCardinalWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FMediaType <> Value) and (isMediaType in FInitialSetupOptions) then
		begin
			FMediaType := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetOrientation(Value : TSmallIntWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FOrientation <> Value) and (isOrientation in FInitialSetupOptions) then
		begin
			FOrientation := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetPaperLength(Value : SmallInt);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FPaperLength <> Value) and (isPaperLength in FInitialSetupOptions) then
		begin
			FPaperLength := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetPaperSize(Value : TSmallIntWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FPaperSize <> Value) and (isPaperSize in FInitialSetupOptions) then
		begin
			FPaperSize := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetPaperWidth(Value : SmallInt);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FPaperWidth <> Value) and (isPaperWidth in FInitialSetupOptions) then
		begin
			FPaperWidth := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetPrinterName(Value : TStringWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FPrinterName <> Value) then
		begin
			FPrinterName := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetPrintQuality(Value : TSmallIntWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FPrintQuality <> Value) and (isPrintQuality in FInitialSetupOptions) then
		begin
			FPrintQuality := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetScale(Value : SmallInt);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FScale <> Value) and (isScale in FInitialSetupOptions) then
		begin
			FScale := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetTTOption(Value : TSmallIntWithList);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FTTOption <> Value) and (isTTOption in FInitialSetupOptions) then
		begin
			FTTOption := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

procedure TInitialSetupValues.SetYResolution(Value : SmallInt);
begin
	if FComponent is TPBPrinterSetupDialog then with (FComponent as TPBPrinterSetupDialog) do
	begin
		if (FYResolution <> Value) and (isYResolution in FInitialSetupOptions) then
		begin
			FYResolution := Value;
			if not (csReading	in ComponentState) and ((csDesigning in ComponentState)
				or (FSetupType = stInitial)) then	InitialSetup;
		end;
	end;
end;

// --------------------- PropertyEditors--------------------------
type
	TInitialSetupProperty = class(TClassProperty)
	public
		function GetAttributes : TPropertyAttributes; override;
		procedure Edit; override;
	end;

	TIntegerWithListProperty = class(TIntegerProperty)
	protected
		FValueList : TStringList;
		destructor Destroy; override;
		procedure Initialize; override;
	public
		function GetAttributes : TPropertyAttributes; override;
		function GetValue : string; override;
		procedure GetValues(Proc: TGetStrProc); override;
		procedure SetValue(const Value : string); override;
	end;

	TCollateSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TColorSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TDefaultSourceSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TDitherTypeSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TDuplexSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TStringWithListProperty = class(TStringProperty)
	protected
		FValueList : TStringList;
		destructor Destroy; override;
		procedure Initialize; override;
	public
		function GetAttributes : TPropertyAttributes; override;
		procedure GetValues(Proc: TGetStrProc); override;
	end;

	TFormNameSetupProperty = class(TStringWithListProperty)
	public
		procedure GetValues(Proc: TGetStrProc); override;
	end;

	TICMIntentSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TICMMethodSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TMediaTypeSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TOrientationSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TPaperSizeSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TPrinterNameProperty = class(TStringWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TPrintQualitySetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

	TTTOptionSetupProperty = class(TIntegerWithListProperty)
	protected
		procedure Initialize; override;
	end;

// --------------------- ComponentEditor --------------------------
{$IFDEF VER90}
	IFormDesigner = TFormDesigner;
{$ENDIF}
{$IFDEF VER100}
	IFormDesigner = TFormDesigner;
{$ENDIF}
	TPBPrinterSetupEditor = class(TDefaultEditor)
	protected
		constructor Create(AComponent: TComponent; ADesigner: IFormDesigner); override;
		destructor Destroy; override;
		procedure EditProperty(PropertyEditor : TPropertyEditor;
			var Continue, FreeEditor : Boolean); override;
	public
		procedure ExecuteVerb(Index : Integer); override;
		function GetVerb(Index : Integer) : string; override;
		function GetVerbCount : Integer; override;
	end;

// ------------- PropertyEditor for InitialSetupValues  ----------------------
function TInitialSetupProperty.GetAttributes : TPropertyAttributes;
begin
	Result := [paDialog, paSubProperties, paRevertable];
end;

procedure TInitialSetupProperty.Edit;
begin
	if Printer.Printers.Count = 0 then Exit;
	if GetName = 'InitialSetupValues' then
	begin
		if TPBPrinterSetupDialog(GetComponent(0)).InheritedExecute then Modified;
	end;
end;

// --------- Ancestor PropertyEditor for all IntegerWithLists  ------------
destructor TIntegerWithListProperty.Destroy;
begin
	FValueList.Free;
	FValueList := nil;
	inherited Destroy;
end;

function TIntegerWithListProperty.GetAttributes : TPropertyAttributes;
begin
	Result := [paValueList, paSortList, paRevertable];
end;

function TIntegerWithListProperty.GetValue : string;
var
	t : integer;
	Name : string;
begin
	Result := inherited GetValue;
	t := 0;
	while t < FValueList.Count do
	begin
		Name := FValueList.Names[t];
		if FValueList.Values[Name] = Result then
		begin
			Result := Name;
			Exit;
		end;
		Inc(t);
	end;
end;

procedure TIntegerWithListProperty.Initialize;
begin
	inherited Initialize;
	FValueList := TStringList.Create;
end;

procedure TIntegerWithListProperty.GetValues(Proc: TGetStrProc);
var
	t : integer;
begin
	t := 0;
	while t < FValueList.Count do
	begin
		Proc(FValueList.Names[t]);
		Inc(t);
	end;
end;

procedure TIntegerWithListProperty.SetValue(const Value : string);
begin
	if FValueList.Values[Value] <> '' then inherited SetValue(FValueList.Values[Value])
	else inherited SetValue(Value);
end;

// --- Specialized PropertyEditors (descends from TIntegerWithListProperty) ---
procedure TCollateSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMCOLLATE_TRUE=' + IntToStr(DMCOLLATE_TRUE));
	FValueList.Add('DMCOLLATE_FALSE=' + IntToStr(DMCOLLATE_FALSE));
end;

procedure TColorSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMCOLOR_COLOR=' + IntToStr(DMCOLOR_COLOR));
	FValueList.Add('DMCOLOR_MONOCHROME=' + IntToStr(DMCOLOR_MONOCHROME));
end;

procedure TDefaultSourceSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMBIN_UPPER=' + IntToStr(DMBIN_UPPER));
	FValueList.Add('DMBIN_LOWER=' + IntToStr(DMBIN_LOWER));
	FValueList.Add('DMBIN_MIDDLE=' + IntToStr(DMBIN_MIDDLE));
	FValueList.Add('DMBIN_MANUAL=' + IntToStr(DMBIN_MANUAL));
	FValueList.Add('DMBIN_ENVELOPE=' + IntToStr(DMBIN_ENVELOPE));
	FValueList.Add('DMBIN_ENVMANUAL=' + IntToStr(DMBIN_ENVMANUAL));
	FValueList.Add('DMBIN_TRACTOR=' + IntToStr(DMBIN_TRACTOR));
	FValueList.Add('DMBIN_SMALLFMT=' + IntToStr(DMBIN_SMALLFMT));
	FValueList.Add('DMBIN_LARGEFMT=' + IntToStr(DMBIN_LARGEFMT));
	FValueList.Add('DMBIN_LARGECAPACITY=' + IntToStr(DMBIN_LARGECAPACITY));
	FValueList.Add('DMBIN_FORMSOURCE=' + IntToStr(DMBIN_FORMSOURCE));
end;

procedure TDitherTypeSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMDITHER_NONE=' + IntToStr(DMDITHER_NONE));
	FValueList.Add('DMDITHER_COARSE=' + IntToStr(DMDITHER_COARSE));
	FValueList.Add('DMDITHER_FINE=' + IntToStr(DMDITHER_FINE));
	FValueList.Add('DMDITHER_LINEART=' + IntToStr(DMDITHER_LINEART));
	FValueList.Add('DMDITHER_GRAYSCALE=' + IntToStr(DMDITHER_GRAYSCALE));
end;

procedure TDuplexSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMDUP_SIMPLEX=' + IntToStr(DMDUP_SIMPLEX));
	FValueList.Add('DMDUP_HORIZONTAL=' + IntToStr(DMDUP_HORIZONTAL));
	FValueList.Add('DMDUP_VERTICAL=' + IntToStr(DMDUP_VERTICAL));
end;

procedure TICMIntentSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMICM_SATURATE=' + IntToStr(DMICM_SATURATE));
	FValueList.Add('DMICM_CONTRAST=' + IntToStr(DMICM_CONTRAST));
	FValueList.Add('DMICM_COLORMETRIC=' + IntToStr(DMICM_COLORMETRIC));
end;

procedure TICMMethodSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMICMMETHOD_NONE=' + IntToStr(DMICMMETHOD_NONE));
	FValueList.Add('DMICMMETHOD_SYSTEM=' + IntToStr(DMICMMETHOD_SYSTEM));
	FValueList.Add('DMICMMETHOD_DRIVER=' + IntToStr(DMICMMETHOD_DRIVER));
	FValueList.Add('DMICMMETHOD_DEVICE=' + IntToStr(DMICMMETHOD_DEVICE));
end;

procedure TMediaTypeSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMMEDIA_STANDARD=' + IntToStr(DMMEDIA_STANDARD));
	FValueList.Add('DMMEDIA_TRANSPARENCY=' + IntToStr(DMMEDIA_TRANSPARENCY));
	FValueList.Add('DMMEDIA_GLOSSY=' + IntToStr(DMMEDIA_GLOSSY));
end;

procedure TOrientationSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMORIENT_PORTRAIT=' + IntToStr(DMORIENT_PORTRAIT));
	FValueList.Add('DMORIENT_LANDSCAPE=' + IntToStr(DMORIENT_LANDSCAPE));
end;

procedure TPaperSizeSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMPAPER_LETTER=' + IntToStr(DMPAPER_LETTER));
	FValueList.Add('DMPAPER_LETTERSMALL=' + IntToStr(DMPAPER_LETTERSMALL));
	FValueList.Add('DMPAPER_TABLOID=' + IntToStr(DMPAPER_TABLOID));
	FValueList.Add('DMPAPER_LEDGER=' + IntToStr(DMPAPER_LEDGER));
	FValueList.Add('DMPAPER_LEGAL=' + IntToStr(DMPAPER_LEGAL));
	FValueList.Add('DMPAPER_STATEMENT=' + IntToStr(DMPAPER_STATEMENT));
	FValueList.Add('DMPAPER_EXECUTIVE=' + IntToStr(DMPAPER_EXECUTIVE));
	FValueList.Add('DMPAPER_A3=' + IntToStr(DMPAPER_A3));
	FValueList.Add('DMPAPER_A4=' + IntToStr(DMPAPER_A4));
	FValueList.Add('DMPAPER_A4SMALL=' + IntToStr(DMPAPER_A4SMALL));
	FValueList.Add('DMPAPER_A5=' + IntToStr(DMPAPER_A5));
	FValueList.Add('DMPAPER_B4=' + IntToStr(DMPAPER_B4));
	FValueList.Add('DMPAPER_B5=' + IntToStr(DMPAPER_B5));
	FValueList.Add('DMPAPER_FOLIO=' + IntToStr(DMPAPER_FOLIO));
	FValueList.Add('DMPAPER_QUARTO=' + IntToStr(DMPAPER_QUARTO));
	FValueList.Add('DMPAPER_10X14=' + IntToStr(DMPAPER_10X14));
	FValueList.Add('DMPAPER_11X17=' + IntToStr(DMPAPER_11X17));
	FValueList.Add('DMPAPER_NOTE=' + IntToStr(DMPAPER_NOTE));
	FValueList.Add('DMPAPER_ENV_9=' + IntToStr(DMPAPER_ENV_9));
	FValueList.Add('DMPAPER_ENV_10=' + IntToStr(DMPAPER_ENV_10));
	FValueList.Add('DMPAPER_ENV_11=' + IntToStr(DMPAPER_ENV_11));
	FValueList.Add('DMPAPER_ENV_12=' + IntToStr(DMPAPER_ENV_12));
	FValueList.Add('DMPAPER_ENV_14=' + IntToStr(DMPAPER_ENV_14));
	FValueList.Add('DMPAPER_CSHEET=' + IntToStr(DMPAPER_CSHEET));
	FValueList.Add('DMPAPER_DSHEET=' + IntToStr(DMPAPER_DSHEET));
	FValueList.Add('DMPAPER_ESHEET=' + IntToStr(DMPAPER_ESHEET));
	FValueList.Add('DMPAPER_ENV_DL=' + IntToStr(DMPAPER_ENV_DL));
	FValueList.Add('DMPAPER_ENV_C5=' + IntToStr(DMPAPER_ENV_C5));
	FValueList.Add('DMPAPER_ENV_C3=' + IntToStr(DMPAPER_ENV_C3));
	FValueList.Add('DMPAPER_ENV_C4=' + IntToStr(DMPAPER_ENV_C4));
	FValueList.Add('DMPAPER_ENV_C6=' + IntToStr(DMPAPER_ENV_C6));
	FValueList.Add('DMPAPER_ENV_C65=' + IntToStr(DMPAPER_ENV_C65));
	FValueList.Add('DMPAPER_ENV_B4=' + IntToStr(DMPAPER_ENV_B4));
	FValueList.Add('DMPAPER_ENV_B5=' + IntToStr(DMPAPER_ENV_B5));
	FValueList.Add('DMPAPER_ENV_B6=' + IntToStr(DMPAPER_ENV_B6));
	FValueList.Add('DMPAPER_ENV_ITALY=' + IntToStr(DMPAPER_ENV_ITALY));
	FValueList.Add('DMPAPER_ENV_MONARCH=' + IntToStr(DMPAPER_ENV_MONARCH));
	FValueList.Add('DMPAPER_ENV_PERSONAL=' + IntToStr(DMPAPER_ENV_PERSONAL));
	FValueList.Add('DMPAPER_FANFOLD_US=' + IntToStr(DMPAPER_FANFOLD_US));
	FValueList.Add('DMPAPER_FANFOLD_STD_GERMAN=' + IntToStr(DMPAPER_FANFOLD_STD_GERMAN));
	FValueList.Add('DMPAPER_FANFOLD_LGL_GERMAN=' + IntToStr(DMPAPER_FANFOLD_LGL_GERMAN));
	FValueList.Add('DMPAPER_ISO_B4=' + IntToStr(DMPAPER_ISO_B4));
	FValueList.Add('DMPAPER_JAPANESE_POSTCARD=' + IntToStr(DMPAPER_JAPANESE_POSTCARD));
	FValueList.Add('DMPAPER_9X11=' + IntToStr(DMPAPER_9X11));
	FValueList.Add('DMPAPER_10X11=' + IntToStr(DMPAPER_10X11));
	FValueList.Add('DMPAPER_15X11=' + IntToStr(DMPAPER_15X11));
	FValueList.Add('DMPAPER_ENV_INVITE=' + IntToStr(DMPAPER_ENV_INVITE));
	FValueList.Add('DMPAPER_LETTER_EXTRA=' + IntToStr(DMPAPER_LETTER_EXTRA));
	FValueList.Add('DMPAPER_LEGAL_EXTRA=' + IntToStr(DMPAPER_LEGAL_EXTRA));
	FValueList.Add('DMPAPER_TABLOID_EXTRA=' + IntToStr(DMPAPER_TABLOID_EXTRA));
	FValueList.Add('DMPAPER_A4_EXTRA=' + IntToStr(DMPAPER_A4_EXTRA));
	FValueList.Add('DMPAPER_LETTER_TRANSVERSE=' + IntToStr(DMPAPER_LETTER_TRANSVERSE));
	FValueList.Add('DMPAPER_A4_TRANSVERSE=' + IntToStr(DMPAPER_A4_TRANSVERSE));
	FValueList.Add('DMPAPER_LETTER_EXTRA_TRANSVERSE=' + IntToStr(DMPAPER_LETTER_EXTRA_TRANSVERSE));
	FValueList.Add('DMPAPER_A_PLUS=' + IntToStr(DMPAPER_A_PLUS));
	FValueList.Add('DMPAPER_B_PLUS=' + IntToStr(DMPAPER_B_PLUS));
	FValueList.Add('DMPAPER_LETTER_PLUS=' + IntToStr(DMPAPER_LETTER_PLUS));
	FValueList.Add('DMPAPER_A4_PLUS=' + IntToStr(DMPAPER_A4_PLUS));
	FValueList.Add('DMPAPER_A5_TRANSVERSE=' + IntToStr(DMPAPER_A5_TRANSVERSE));
	FValueList.Add('DMPAPER_B5_TRANSVERSE=' + IntToStr(DMPAPER_B5_TRANSVERSE));
	FValueList.Add('DMPAPER_A3_EXTRA=' + IntToStr(DMPAPER_A3_EXTRA));
	FValueList.Add('DMPAPER_A5_EXTRA=' + IntToStr(DMPAPER_A5_EXTRA));
	FValueList.Add('DMPAPER_B5_EXTRA=' + IntToStr(DMPAPER_B5_EXTRA));
	FValueList.Add('DMPAPER_A2=' + IntToStr(DMPAPER_A2));
	FValueList.Add('DMPAPER_A3_TRANSVERSE=' + IntToStr(DMPAPER_A3_TRANSVERSE));
	FValueList.Add('DMPAPER_A3_EXTRA_TRANSVERSE=' + IntToStr(DMPAPER_A3_EXTRA_TRANSVERSE));
	FValueList.Add('DMPAPER_USER=' + IntToStr(DMPAPER_USER));
end;

procedure TPrintQualitySetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMRES_DRAFT=' + IntToStr(DMRES_DRAFT));
	FValueList.Add('DMRES_LOW=' + IntToStr(DMRES_LOW));
	FValueList.Add('DMRES_MEDIUM=' + IntToStr(DMRES_MEDIUM));
	FValueList.Add('DMRES_HIGH=' + IntToStr(DMRES_HIGH));
end;

procedure TTTOptionSetupProperty.Initialize;
begin
	inherited Initialize;
	FValueList.Add('DMTT_BITMAP=' + IntToStr(DMTT_BITMAP));
	FValueList.Add('DMTT_DOWNLOAD=' + IntToStr(DMTT_DOWNLOAD));
	FValueList.Add('DMTT_SUBDEV=' + IntToStr(DMTT_SUBDEV));
	FValueList.Add('DMTT_DOWNLOAD_OUTLINE=' + IntToStr(DMTT_DOWNLOAD_OUTLINE));
end;

// --------------- Ancestor PropertyEditor StringWithList  --------------------
destructor TStringWithListProperty.Destroy;
begin
	FValueList.Free;
	FValueList := nil;
	inherited Destroy;
end;

function TStringWithListProperty.GetAttributes : TPropertyAttributes;
begin
	Result := [paValueList, paSortList, paRevertable];
end;

procedure TStringWithListProperty.Initialize;
begin
	inherited Initialize;
	FValueList := TStringList.Create;
end;

procedure TStringWithListProperty.GetValues(Proc: TGetStrProc);
var
	t : integer;
begin
	t := 0;
	while t < FValueList.Count do
	begin
		Proc(FValueList[t]);
		Inc(t);
	end;
end;

// --- Specialized PropertyEditors (descends from TStringWithListProperty) ---
procedure TFormNameSetupProperty.GetValues(Proc: TGetStrProc);
type
	TPaperNameArray = array [1..256, 0..63] of Char;
	PPapernameArray = ^TPaperNameArray;
var
	NumberOfPaperNames, t : integer;
	PPaperNames : PPapernameArray;
begin
	FValueList.Clear;
	with TPBPrinterSetupDialog(GetComponent(0)) do
	begin
		if GetPrinterSetup then with FPrinterRecord do
		begin
			NumberOfPaperNames := DeviceCapabilities(PrinterName, Port, DC_PAPERNAMES, nil, PPrinterDevMode);
			if NumberOfPaperNames > 0 then
			begin
				GetMem(PPaperNames, NumberOfPaperNames * 64);
				try
					DeviceCapabilities(PrinterName, Port, DC_PAPERNAMES, Pchar(PPaperNames), PPrinterDevMode);
					for t := 1 to NumberOfPaperNames do	FValueList.Add(PPaperNames^[t]);
				finally
					FreeMem(PPaperNames);
				end;
			end;
			GlobalUnlock(DevModeHandle);
		end;
	end;
	inherited GetValues(Proc);
end;

procedure TPrinterNameProperty.Initialize;
var
	t, p : integer;
begin
	inherited Initialize;
	if Printer.Printers.Count = 0 then Exit;
	for t := 0 to Printer.Printers.Count - 1 do
	begin
		p := Pos(' on ', Printer.Printers[t]);
		if p > 0 then FValueList.Add(Copy(Printer.Printers[t], 1, p - 1))
		else FValueList.Add(Printer.Printers[t]);
	end;
end;

// -------------------- ComponentEditor -------------------
constructor TPBPrinterSetupEditor.Create(AComponent: TComponent; ADesigner: IFormDesigner);
begin
	inherited Create(AComponent, ADesigner);
	with (Component as TPBPrinterSetupDialog) do
	begin
		SaveSetup;
		InitialSetup;
	end;
end;

destructor TPBPrinterSetupEditor.Destroy;
begin
	(Component as TPBPrinterSetupDialog).LoadSetup;
	inherited Destroy;
end;

procedure TPBPrinterSetupEditor.EditProperty(PropertyEditor : TPropertyEditor;
	var Continue, FreeEditor : Boolean);
begin
	if CompareText(PropertyEditor.GetName, 'InitialSetupValues') = 0 then
	begin
		PropertyEditor.Edit;
		Continue := False;
	end;
end;

procedure TPBPrinterSetupEditor.ExecuteVerb(Index : Integer);
begin
	if Index = 0 then Edit;
end;

function TPBPrinterSetupEditor.GetVerb(Index : Integer) : string;
begin
	if Index = 0 then Result := 'Edit InitialSetupValues'
	else Result := '';
end;

function TPBPrinterSetupEditor.GetVerbCount : Integer;
begin
	Result :=  1;
end;

procedure Register;
begin
	RegisterComponents('PB', [TPBPrinterSetupDialog]);
	RegisterPropertyEditor(TypeInfo(TInitialSetupValues), TPBPrinterSetupDialog, '', TInitialSetupProperty);
	RegisterPropertyEditor(TypeInfo(TSmallIntWithList), TInitialSetupValues, 'Collate', TCollateSetupProperty);
	RegisterPropertyEditor(TypeInfo(TSmallIntWithList), TInitialSetupValues, 'Color', TColorSetupProperty);
	RegisterPropertyEditor(TypeInfo(TSmallIntWithList), TInitialSetupValues, 'DefaultSource', TDefaultSourceSetupProperty);
	RegisterPropertyEditor(TypeInfo(TCardinalWithList), TInitialSetupValues, 'DitherType', TDitherTypeSetupProperty);
	RegisterPropertyEditor(TypeInfo(TSmallIntWithList), TInitialSetupValues, 'Duplex', TDuplexSetupProperty);
	RegisterPropertyEditor(TypeInfo(TStringWithList), TInitialSetupValues, 'FormName', TFormNameSetupProperty);
	RegisterPropertyEditor(TypeInfo(TCardinalWithList), TInitialSetupValues, 'ICMIntent', TICMIntentSetupProperty);
	RegisterPropertyEditor(TypeInfo(TCardinalWithList), TInitialSetupValues, 'ICMMethod', TICMMethodSetupProperty);
	RegisterPropertyEditor(TypeInfo(TCardinalWithList), TInitialSetupValues, 'MediaType', TMediaTypeSetupProperty);
	RegisterPropertyEditor(TypeInfo(TSmallIntWithList), TInitialSetupValues, 'Orientation', TOrientationSetupProperty);
	RegisterPropertyEditor(TypeInfo(TSmallIntWithList), TInitialSetupValues, 'PaperSize', TPaperSizeSetupProperty);
	RegisterPropertyEditor(TypeInfo(TStringWithList), TInitialSetupValues, 'PrinterName', TPrinterNameProperty);
	RegisterPropertyEditor(TypeInfo(TSmallIntWithList), TInitialSetupValues, 'PrintQuality', TPrintQualitySetupProperty);
	RegisterPropertyEditor(TypeInfo(TSmallIntWithList), TInitialSetupValues, 'TTOption', TTTOptionSetupProperty);
	RegisterComponentEditor(TPBPrinterSetupDialog, TPBPrinterSetupEditor);
end;

end.

