{---------------------------------------------------------------------------------------
  Class:       TFileCompare
  Version:     1.0
  Date:        12 July 2000
  Author:      Mike GIBBARD
  Email:       mike.gibbard@burnsecs.com

  Description: TFileCompare is a simple class for comparing one file (CompareFileName)
               with another (BaseFileName) using the CompareMem function in SysUtils to
               perform a binary compare of Length bytes of memory referenced by
               BaseFileName to that of CompareFileName.

  Properties:  * BaseFileName: string - Fully qualified path and filename of the file
                 against which the comparison will be made

               * CompareFileName: string - Fully qualified path and filename of the file
                 to compared

  Copyright:   Copyright 2000, Mike Gibbard (hereafter 'the Author').
               All Rights Reserved.

  IMPORTANT:   This software is provided 'as-is', without any express or implied
               warranty. In no event will the author be held liable for any damages
               arising from the use of this software.
               Permission is granted to anyone to use this software for any purpose,
               including commercial applications, and to alter it and redistribute it
               freely, subject to the following restrictions:
               1. The origin of this software must not be misrepresented, you must not
                  claim that you wrote the original software. If you use this software
                  in a product, an acknowledgment in the product documentation would be
                  appreciated but is not required.
               2. Altered source versions must be plainly marked as such, and must not be
                  misrepresented as being the original software.
               3. This notice may not be removed or altered from any source distribution.

  Example:     procedure TForm1.Button1Click(Sender: TObject);
               var
                 FCompare: TFileCompare;
               begin
                 FCompare := TFileCompare.Create;
                 try
                   if (Edit1.Text <> '') and (Edit2.Text <> '') then
                      begin
                        with FCompare do
                          begin
                            BaseFileName := Edit1.Text;
                            CompareFileName := Edit2.Text;
                            if CompareFiles then
                               ShowMessage('Files are the same')
                               else
                               ShowMessage('Files are different');
                          end;
                      end;
                 finally
                   FCompare.Free;
                 end;
               end;

---------------------------------------------------------------------------------------}

unit FileCompare;

interface

uses
Classes, SysUtils;

type
TFileCompare = class(TObject)
  private
    FBaseFileName: string;
    FCompareFileName: string;
    FBaseStream: TMemoryStream;
    FCompareStream: TMemoryStream;
    procedure SetBaseFileName(Val: string);
    procedure SetCompareFileName(Val: string);
  public
    constructor Create;
    destructor Destroy; override;
    function CompareFiles: boolean;
    property BaseFileName: string read FBaseFileName write SetBaseFileName;
    property CompareFileName: string read FCompareFileName write SetCompareFileName;
  end;

  { In Delphi 5, the CompareMem function is exposed in the SysUtils unit whereas, in
    Delphi 2, it's hidden in the Classes unit. I don't know where it appears in Delphi's
    3 and 4, so, for convenience, I've included it here}

  function CompareMem(P1, P2: Pointer; Length: Integer): Boolean; assembler;


implementation

constructor TFileCompare.Create;
begin
  FBaseStream := Nil;
  FCompareStream := Nil;
  FBaseFileName := '';
  FCompareFileName := '';
end;

destructor TFileCompare.Destroy;
begin
  inherited Destroy;
end;

procedure TFileCompare.SetBaseFileName(Val: String);
begin
  if FileExists(Val) then FBaseFileName := Val
     else
     raise Exception.Create(Val+' does not exist');
end;

procedure TFileCompare.SetCompareFileName(Val: String);
begin
  if FileExists(Val) then FCompareFileName := Val
     else
     raise Exception.Create(Val+' does not exist');
end;

function TFileCompare.CompareFiles: boolean;
begin
  Result := FALSE;
  //Instantiate the two memory streams
  FBaseStream := TMemoryStream.Create;
  FCompareStream := TMemoryStream.Create;

  try
    //Load both files into their respective streams...
    FBaseStream.LoadFromFile(FBaseFileName);
    FCompareStream.LoadFromFile(FCompareFileName);

    //...and compare them
    Result := (FBaseStream.Size = FCompareStream.Size) and
               CompareMem(FBaseStream.Memory, FCompareStream.Memory, FBaseStream.Size);
  finally
    //Tidy up
    FBaseStream.Free;
    FCompareStream.Free;
    FBaseStream := Nil;
    FCompareStream := Nil;
  end;
end;

function CompareMem(P1, P2: Pointer; Length: Integer): Boolean; assembler;
asm
        PUSH    ESI
        PUSH    EDI
        MOV     ESI,P1
        MOV     EDI,P2
        MOV     EDX,ECX
        XOR     EAX,EAX
        AND     EDX,3
        SHR     ECX,2
        REPE    CMPSD
        JNE     @@2
        MOV     ECX,EDX
        REPE    CMPSB
        JNE     @@2
@@1:    INC     EAX
@@2:    POP     EDI
        POP     ESI
end;


end.
