unit Panel3D;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls;

type
  TPanelText3DBevel = (bvNone3,bvLowered3,bvRaised3,bvShadow3);
  TPanel3DVertAlignment = (vaTop,vaCenter,vaBottom);
  TPanel3DRePaintCT = (rpAll,rpCaption,rpTitle);
  TPanel3D = class(TPanel)
  private
    FAlignmentTitle: TAlignment;
    FBevelCaption: TPanelText3DBevel;
    FBevelTitle: TPanelText3DBevel;
    FBorderTitle: TPanelBevel;
    FColorTitle: TColor;
    FFontTitle: TFont;
    FHColorCaption: TColor;
    FHColorTitle: TColor;
    FRePaintCT: TPanel3DRePaintCT;
    FSColorCaption: TColor;
    FSColorTitle: TColor;
    FShadowSizeCaption: Byte;
    FShadowSizeTitle: Byte;
    FVertAlignment: TPanel3DVertAlignment;
    FTitle: String;
    procedure DoDrawCaption(var Rect: TRect;Flags: Word);
    procedure DoDrawShadowText(ColorHigh,ColorLow: TColor;Rect: TRect;
      SH: Byte;Text: String;BevelState: TPanelText3DBevel;Flags: Word;
      AFont: TFont);
    procedure DoDrawTitle(var Rect: TRect;Flags: Word);
    function IsFontTitleStored: Boolean;
    procedure SetAlignmentTitle(Value: TAlignment);
    procedure SetBevelCaption(Value: TPanelText3DBevel);
    procedure SetBevelTitle(Value: TPanelText3DBevel);
    procedure SetBorderTitle(Value: TPanelBevel);
    procedure SetColorTitle(Value: TColor);
    procedure SetFontTitle(Value: TFont);
    procedure SetHColorCaption(Value: TColor);
    procedure SetHColorTitle(Value: TColor);
    procedure SetRePaintCT(Value: TPanel3DRePaintCT);
    procedure SetSColorCaption(Value: TColor);
    procedure SetSColorTitle(Value: TColor);
    procedure SetShadowSizeCaption(Value: Byte);
    procedure SetShadowSizeTitle(Value: Byte);
    procedure SetTitle(Value: String);
    procedure SetVertAlignment(Value: TPanel3DVertAlignment);
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property AlignmentTitle: TAlignment read FAlignmentTitle
      write SetAlignmentTitle default taLeftJustify;
    property BevelCaption: TPanelText3DBevel read FBevelCaption
      write SetBevelCaption default bvRaised3;
    property BevelTitle: TPanelText3DBevel read FBevelTitle
      write SetBevelTitle default bvRaised3;
    property BorderTitle: TPanelBevel read FBorderTitle
      write SetBorderTitle default bvNone;
    property ColorTitle: TColor read FColorTitle
      write SetColorTitle default clBtnFace;
    property FontTitle: TFont read FFontTitle
      write SetFontTitle stored IsFontTitleStored;
    property HColorCaption: TColor read FHColorCaption
      write SetHColorCaption default clBtnHighLight;
    property HColorTitle: TColor read FHColorTitle
      write SetHColorTitle default clBtnHighLight;
    property RePaintCT: TPanel3DRePaintCT read FRePaintCT
      write SetRePaintCT default rpAll;
    property SColorCaption: TColor read FSColorCaption
      write SetSColorCaption default clBtnShadow;
    property SColorTitle: TColor read FSColorTitle
      write SetSColorTitle default clBtnShadow;
    property ShadowSizeCaption: Byte read FShadowSizeCaption
      write SetShadowSizeCaption default 1;
    property ShadowSizeTitle: Byte read FShadowSizeTitle
      write SetShadowSizeTitle default 1;
    property Title: String read FTitle write SetTitle;
    property VertAlignment: TPanel3DVertAlignment read FVertAlignment
      write SetVertAlignment default vaCenter;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('3DElements', [TPanel3D]);
end;

{
  class TPanel3D
  ~~~~~~~~~~~~~~
}
constructor TPanel3D.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAlignmentTitle:=taLeftJustify;
  FBevelCaption:=bvRaised3;
  FBevelTitle:=bvRaised3;
  FBorderTitle:=bvNone;
  FColorTitle:=clBtnFace;
  FFontTitle:=TFont.Create;
  FHColorCaption:=clBtnHighLight;
  FHColorTitle:=clBtnHighLight;
  FRePaintCT:=rpAll;
  FSColorCaption:=clBtnShadow;
  FSColorTitle:=clBtnShadow;
  FShadowSizeCaption:=1;
  FShadowSizeTitle:=1;
  FTitle:='';{Name;}
  FVertAlignment:=vaCenter;
end;

destructor TPanel3D.Destroy;
begin
  FFontTitle.Free;
  inherited Destroy;
end;


procedure TPanel3D.DoDrawShadowText(ColorHigh,ColorLow: TColor;Rect: TRect;
  SH: Byte;Text: String;BevelState: TPanelText3DBevel;Flags: Word;AFont: TFont);
var
  R: TRect;
begin
  {High shadow text}
  if BevelState <> bvShadow3 then
  begin
    R:=Rect;
    OffsetRect(R,-SH,-SH);
    Canvas.Font.Color:=ColorHigh;
    if Enabled then
      DrawText(Canvas.Handle,PChar(Text),Length(Text),R,Flags);
  end;
  {Low shadow text}
  R:=Rect;
  OffsetRect(R,SH,SH);
  Canvas.Font.Color:=ColorLow;
  DrawText(Canvas.Handle,PChar(Text),Length(Text),R,Flags);
  {Text}
  Canvas.Font:=AFont;
  if not Enabled then Canvas.Font.Color:=clGrayText;
  DrawText(Canvas.Handle,PChar(Text),Length(Text),Rect,Flags);
end;

procedure TPanel3D.DoDrawCaption(var Rect: TRect;Flags: Word);
begin
  Canvas.Font:=Font;
  if FBevelCaption = bvNone3 then
    DrawText(Canvas.Handle,PChar(Caption),Length(Caption),Rect,Flags)
  else
    case FBevelCaption of
      bvRaised3 : DoDrawShadowText(FHColorCaption,FSColorCaption,Rect,
                   FShadowSizeCaption,Caption,FBevelCaption,Flags,Font);
      bvLowered3: DoDrawShadowText(FSColorCaption,FHColorCaption,Rect,
                   FShadowSizeCaption,Caption,FBevelCaption,Flags,Font);
      bvShadow3 : DoDrawShadowText(0,FSColorCaption,Rect,
                   FShadowSizeCaption,Caption,FBevelCaption,Flags,Font);
    end;
end;

procedure TPanel3D.DoDrawTitle(var Rect: TRect;Flags: Word);
begin
  Canvas.Font:=FontTitle;
  if FBevelTitle = bvNone3 then
    DrawText(Canvas.Handle,PChar(Title),Length(Title),Rect,Flags)
  else
    case FBevelTitle of
      bvRaised3 : DoDrawShadowText(FHColorTitle,FSColorTitle,Rect,
                   FShadowSizeTitle,Title,FBevelTitle,Flags,FontTitle);
      bvLowered3: DoDrawShadowText(FSColorTitle,FHColorTitle,Rect,
                   FShadowSizeTitle,Title,FBevelTitle,Flags,FontTitle);
      bvShadow3 : DoDrawShadowText(0,FSColorTitle,Rect,
                   FShadowSizeTitle,Title,FBevelTitle,Flags,FontTitle);
    end;
end;

procedure TPanel3D.Paint;
var
  Rect,R: TRect;
  TopColor,BottomColor: TColor;
  H: Integer;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT,DT_RIGHT,DT_CENTER);

  procedure AdjustColors(Bevel: TPanelBevel);
  begin
    TopColor:=clBtnHighlight;
    if Bevel = bvLowered then TopColor:=clBtnShadow;
    BottomColor:=clBtnShadow;
    if Bevel = bvLowered then BottomColor:=clBtnHighlight;
  end;

begin
  Rect:=GetClientRect;
  Canvas.Brush.Color:=Color;
  Canvas.FillRect(Rect);
  Rect.Top:=Rect.Top-(FontTitle.Height div 2)+FShadowSizeTitle;
  if BevelOuter <> bvNone then
  begin
    AdjustColors(BevelOuter);
    Frame3D(Canvas,Rect,TopColor,BottomColor,BevelWidth);
  end;
  Frame3D(Canvas,Rect,Color,Color,BorderWidth);
  if BevelInner <> bvNone then
  begin
    AdjustColors(BevelInner);
    Frame3D(Canvas,Rect,TopColor,BottomColor,BevelWidth);
  end;
  with Canvas do
  begin
    Brush.Style:=bsClear;
    if FRePaintCT <> rpTitle then
    begin
      Font:=Self.Font;
      R:=Rect;
      H:=DrawText(Canvas.Handle,PChar(Caption),-1,R,
          (DT_CALCRECT or DT_EXPANDTABS or DT_VCENTER or DT_WORDBREAK)
          or Alignments[Alignment]);
      with Rect do
      begin
        if FTitle <> '' then
          Top:=Top-(FontTitle.Height div 2)+FShadowSizeTitle;
        case FVertAlignment of
          vaTop   : begin
                      Top:=Top+FShadowSizeCaption;
                      Bottom:=Top+H+FShadowSizeCaption;
                    end;
          vaCenter: begin
                      Top:=(((Bottom+Top)-H) div 2)+FShadowSizeCaption;;
                      Bottom:=Top+H+FShadowSizeCaption;;
                    end;
          vaBottom: Top:=Bottom-H-FShadowSizeCaption;
        end;
        Left:=Left+FShadowSizeCaption;
        Right:=Right-FShadowSizeCaption;
      end;
      {Draw Caption text}
      DoDrawCaption(Rect,(DT_EXPANDTABS or
        DT_VCENTER or DT_WORDBREAK) or Alignments[Alignment]);
    end;
    {Draw Title}
    if (FTitle <> '') and (FRePaintCT <> rpCaption) then
    begin
      Font:=Self.FontTitle;
      {Rect:=R;}
      Rect:=GetClientRect;
      with Rect do
      begin
        H:=TextWidth(FTitle);
        case FAlignmentTitle of
          taLeftJustify : begin
                            Left:=Left+5;
                            if H < (Width-5) then
                              Right:=Left+H;
                          end;
          taCenter      : begin
                            if H > Width then
                              Left:=0
                            else
                              Left:=(Width div 2)-(H div 2);
                              Right:=(Width div 2)+(H div 2);
                          end;
          taRightJustify: begin
                            Right:=Right-5;
                            if H < (Width-5) then
                              Left:=Right-H;
                          end;
        end;
        Bottom:=Top+TextHeight('W')+FShadowSizeTitle;
      end;
      Brush.Color:=FColorTitle;
      R:=Rect;
      with Rect do
      begin
        Left:=Left-FShadowSizeTitle;
        Right:=Right+FShadowSizeTitle;
      end;
      FillRect(Rect);
      case FBorderTitle of
        bvRaised : Frame3D(Canvas,Rect,clBtnHighlight,clBtnShadow,BevelWidth);
        bvLowered: Frame3D(Canvas,Rect,clBtnShadow,clBtnHighlight,BevelWidth);
      end;
      Rect:=R;
      Brush.Style:=bsClear;
      DoDrawTitle(Rect,(DT_EXPANDTABS or DT_VCENTER));
    end;
  end;
end;

function TPanel3D.IsFontTitleStored: Boolean;
begin
  Result:=not ParentFont;
end;

procedure TPanel3D.SetAlignmentTitle(Value: TAlignment);
begin
  FAlignmentTitle:=Value;
  Invalidate;
end;

procedure TPanel3D.SetBevelCaption(Value: TPanelText3DBevel);
begin
  if FBevelCaption <> Value then
  begin
    FBevelCaption:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetBevelTitle(Value: TPanelText3DBevel);
begin
  if FBevelTitle <> Value then
  begin
    FBevelTitle:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetBorderTitle(Value: TPanelBevel);
begin
  if FBorderTitle <> Value then
  begin
    FBorderTitle:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetColorTitle(Value: TColor);
begin
  if FColorTitle <> Value then
  begin
    FColorTitle:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetFontTitle(Value: TFont);
begin
  FFontTitle.Assign(Value);
  Invalidate;
end;

procedure TPanel3D.SetHColorCaption(Value: TColor);
begin
  if FHColorCaption <> Value then
  begin
    FHColorCaption:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetHColorTitle(Value: TColor);
begin
  if FHColorTitle <> Value then
  begin
    FHColorTitle:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetRePaintCT(Value: TPanel3DRePaintCT);
begin
  if FRePaintCT <> Value then
  begin
    FRePaintCT:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetSColorCaption(Value: TColor);
begin
  if FSColorCaption <> Value then
  begin
    FSColorCaption:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetSColorTitle(Value: TColor);
begin
  if FSColorTitle <> Value then
  begin
    FSColorTitle:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetShadowSizeCaption(Value: Byte);
begin
  if FShadowSizeCaption <> Value then
  begin
    FShadowSizeCaption:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetShadowSizeTitle(Value: Byte);
begin
  if FShadowSizeTitle <> Value then
  begin
    FShadowSizeTitle:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetTitle(Value: String);
begin
  if FTitle <> Value then
  begin
    FTitle:=Value;
    Invalidate;
  end;
end;

procedure TPanel3D.SetVertAlignment(Value: TPanel3DVertAlignment);
begin
  if FVertAlignment <> Value then
  begin
    FVertAlignment:=Value;
    Invalidate;
  end;
end;

end.
