unit Button3D;

interface

uses
  Windows,Messages,SysUtils,Classes,Graphics,Controls,Forms,Dialogs,
  StdCtrls,Buttons;

type
  TButton3DBevel = (bvNone,bvLowered,bvRaised,bvShadow);
  TButton3DPressState = (bvNonePress,bvPress);
  TButton3D = class(TButton)
  private
    FCanvas: TCanvas;
    FHighLightColor: TColor;
    FShadowColor: TColor;
    FBevel: TButton3DBevel;
    FShadowSize: Byte;
    FPressState: TButton3DPressState;
    IsFocused: Boolean;
    IsDown: Boolean;
    procedure CNDrawItem(var Message: TWMDrawItem); message CN_DRAWITEM;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure CNMeasureItem(var Message: TWMMeasureItem); message CN_MEASUREITEM;
    procedure DrawButtonText(Canvas: TCanvas; const Caption: string;
      TextBounds: TRect; State: TButtonState);
    procedure DrawItem(const DrawItemStruct: TDrawItemStruct);
    procedure DoDrawShadowText(ColorHigh,ColorLow: TColor;Rect: TRect);
    procedure SetBevel(Value: TButton3DBevel);
    procedure SetHighLightColor(Value: TColor);
    procedure SetShadowColor(Value: TColor);
    procedure SetShadowSize(Value: Byte);
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure SetButtonStyle(ADefault: Boolean); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property HighLightColor: TColor read FHighLightColor
      write SetHighLightColor default clBtnHighLight;
    property ShadowColor: TColor read FShadowColor
      write SetShadowColor default clBtnShadow;
    property Bevel: TButton3DBevel read FBevel
      write SetBevel default bvRaised;
    property ShadowSize: Byte read FShadowSize
      write SetShadowSize default 1;
    property PressState: TButton3DPressState read FPressState
      write FPressState default bvNonePress;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('3DElements',[TButton3D]);
end;

{
  class TButton3D
  ~~~~~~~~~~~~~~~
}
constructor TButton3D.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCanvas:=TCanvas.Create;
  FHighLightColor:=clBtnHighLight;
  FShadowColor:=clBtnShadow;
  FBevel:=bvRaised;
  FShadowSize:=1;
  FPressState:=bvNonePress;
end;

destructor TButton3D.Destroy;
begin
  FCanvas.Free;
  inherited Destroy;
end;

procedure TButton3D.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do Style:=Style or BS_OWNERDRAW;
end;

procedure TButton3D.CNDrawItem(var Message: TWMDrawItem);
begin
  DrawItem(Message.DrawItemStruct^);
end;

procedure TButton3D.CMFontChanged(var Message: TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TButton3D.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TButton3D.CNMeasureItem(var Message: TWMMeasureItem);
begin
  with Message.MeasureItemStruct^ do
  begin
    itemWidth:=Width;
    itemHeight:=Height;
  end;
end;

procedure TButton3D.DrawButtonText(Canvas: TCanvas; const Caption: string;
  TextBounds: TRect; State: TButtonState);
begin
  with Canvas do
  begin
    Brush.Style:=bsClear;
    if State = bsDisabled then
    begin
      OffsetRect(TextBounds,1,1);
      Font.Color:=clWhite;
      DrawText(Handle,PChar(Caption),Length(Caption),TextBounds,
        DT_CENTER or DT_VCENTER or DT_SINGLELINE);
      OffsetRect(TextBounds,-1,-1);
      Font.Color:=clDkGray;
      DrawText(Handle,PChar(Caption),Length(Caption),TextBounds,
        DT_CENTER or DT_VCENTER or DT_SINGLELINE);
    end
    else
    begin
      Canvas.Font:=Font;
      if FBevel = bvNone then
        DrawText(Canvas.Handle,PChar(Caption),Length(Caption),TextBounds,
          DT_CENTER or DT_VCENTER or DT_SINGLELINE)
      else
        if IsDown and (FPressState = bvPress) then
          case FBevel of
            bvRaised : DoDrawShadowText(FShadowColor,FHighLightColor,TextBounds);
            bvLowered: DoDrawShadowText(FHighLightColor,FShadowColor,TextBounds);
            bvShadow : DoDrawShadowText(0,FShadowColor,TextBounds);
          end
        else
          case FBevel of
            bvRaised : DoDrawShadowText(FHighLightColor,FShadowColor,TextBounds);
            bvLowered: DoDrawShadowText(FShadowColor,FHighLightColor,TextBounds);
            bvShadow : DoDrawShadowText(0,FShadowColor,TextBounds);
          end;
    end;
  end;
end;

procedure TButton3D.DrawItem(const DrawItemStruct: TDrawItemStruct);
var
  IsDefault: Boolean;
  State: TButtonState;
  R: TRect;
  Flags: Longint;
begin
  FCanvas.Handle:=DrawItemStruct.hDC;
  R:=ClientRect;
  with DrawItemStruct do
  begin
    IsDown:=itemState and ODS_SELECTED <> 0;
    IsDefault:=itemState and ODS_FOCUS <> 0;
    if not Enabled then State:=bsDisabled
    else if IsDown then State:=bsDown
    else State:=bsUp;
  end;
  Flags:=DFCS_BUTTONPUSH or DFCS_ADJUSTRECT;
  if IsDown then Flags:=Flags or DFCS_PUSHED;
  if DrawItemStruct.itemState and ODS_DISABLED <> 0 then
    Flags:=Flags or DFCS_INACTIVE;
  { DrawFrameControl doesn't allow for drawing a button as the
      default button,so it must be done here. }
  if IsFocused or IsDefault then
  begin
    FCanvas.Pen.Color:=clWindowFrame;
    FCanvas.Pen.Width:=1;
    FCanvas.Brush.Style:=bsClear;
    FCanvas.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
    { DrawFrameControl must draw within this border }
    InflateRect(R,-1,-1);
  end;
  { DrawFrameControl does not draw a pressed button correctly }
  if IsDown then
  begin
    FCanvas.Pen.Color:=clBtnShadow;
    FCanvas.Pen.Width:=1;
    FCanvas.Brush.Color:=clBtnFace;
    FCanvas.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
    InflateRect(R,-1,-1);
  end
  else
    DrawFrameControl(DrawItemStruct.hDC,R,DFC_BUTTON,Flags);
  if IsFocused then
  begin
    R:=ClientRect;
    InflateRect(R,-1,-1);
  end;
  FCanvas.Font:=Self.Font;
  if IsDown then
    OffsetRect(R,1,1);
  DrawButtonText(FCanvas,Caption,R,State);
  if IsFocused then
  begin
    R:=ClientRect;
    InflateRect(R,-4,-4);
    FCanvas.Pen.Color:=clWindowFrame;
    FCanvas.Brush.Color:=clBtnFace;
    DrawFocusRect(FCanvas.Handle,R);
  end;
  FCanvas.Handle:=0;
end;

procedure TButton3D.DoDrawShadowText(ColorHigh,ColorLow: TColor;Rect: TRect);
var
  R: TRect;
  Flags: Word;
begin
  Flags:=DT_CENTER or DT_VCENTER or DT_SINGLELINE;
  {High shadow text}
  if FBevel <> bvShadow then
  begin
    R:=Rect;
    OffsetRect(R,-FShadowSize,-FShadowSize);
    FCanvas.Font.Color:=ColorHigh;
    DrawText(FCanvas.Handle,PChar(Caption),Length(Caption),R,Flags);
  end;
  {Low shadow text}
  if (FBevel = bvShadow) and (IsDown) then
  else
  begin
    R:=Rect;
    OffsetRect(R,FShadowSize,FShadowSize);
    FCanvas.Font.Color:=ColorLow;
    DrawText(FCanvas.Handle,PChar(Caption),Length(Caption),R,Flags);
  end;
  {Text}
  FCanvas.Font:=Font;
  DrawText(FCanvas.Handle,PChar(Caption),Length(Caption),Rect,Flags);
end;

procedure TButton3D.SetButtonStyle(ADefault: Boolean);
begin
  if ADefault <> IsFocused then
  begin
    IsFocused:=ADefault;
    Refresh;
  end;
end;

procedure TButton3D.SetBevel(Value: TButton3DBevel);
begin
  if FBevel <> Value then
  begin
    FBevel:=Value;
    Invalidate;
  end;
end;

procedure TButton3D.SetHighLightColor(Value: TColor);
begin
  if FHighLightColor <> Value then
  begin
    FHighLightColor:=Value;
    Invalidate;
  end;
end;

procedure TButton3D.SetShadowColor(Value: TColor);
begin
  if FShadowColor <> Value then
  begin
    FShadowColor:=Value;
    Invalidate;
  end;
end;

procedure TButton3D.SetShadowSize(Value: Byte);
begin
  if FShadowSize <> Value then
  begin
    FShadowSize:=Value;
    Invalidate;
  end;
end;

procedure TButton3D.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  Perform(WM_LBUTTONDOWN, Message.Keys, Longint(Message.Pos));
end;


end.
