(*******************************************************************************
 ZComp16.PAS
 Copyright 1999 Matthew Meadows.  All rights reserved.
 

 Author Name       :   Matthew Meadows
 Author Initials   :   MBM
 Start Date        :   02/13/99

 Description
 ===========
            Contains implementation of TZCompress for Win16

            TZCompress
              Wraps the PasZLib compression routines Deflate & Inflate.
              
            This code is provided freely for use in the public domain, 
            but the copyright is retained by Matthew Meadows.  This code is 
            provided on an "as is" basis without any warranty of any kind,
            expressed or implied, including but not limited to the implied 
            warranties of merchantability and fitness for a particular purpose.
            The entire risk as to quality and performance of this code is with
            you.  
            
            Please send donations, updates & bug fixes to: 
              Matthew_Meadows@catapultsystems.com


 Modifications
 =============

*******************************************************************************)

unit ZComp16;

interface

uses
  {MBM: Zlib Uses }
  ZLib, ZUtil, ZDeflate, ZInflate, 

  {MBM: VCL Uses }
  SysUtils, Classes;

type
  TZProgress = procedure (Sender: TObject; zStream: z_stream) of object;
  
  TZCompress = class
    {MBM: No member variables, must be reentrant }
  protected
    {MBM: Deflate routines }
    procedure InitDeflate(var zStream: z_Stream);
    procedure ExecuteDeflate(var zStream: z_Stream);
    procedure FinishDeflate(var zStream: z_Stream);
    {MBM: Inflate routines }
    procedure InitInflate(var zStream: z_Stream);
    procedure ExecuteInflate(var zStream: z_Stream);
    procedure FinishInflate(var zStream: z_Stream);
  public
    procedure StreamDeflate(const msIn, msOut: TMemoryStream; pProgress: TZProgress);
    procedure StreamInflate(const msIn, msOut: TMemoryStream; pProgress: TZProgress);
  end;

type
  ECompressionError  = class(Exception);  { Error returned from compressor }
  
const
  STR_COMPRESSION_ERROR = 'Error in compression routine: ';
  ZLIB_READ  = 8092;
  ZLIB_WRITE = 8092;
  
implementation

procedure TZCompress.InitDeflate(var zStream: z_Stream);
var
  iError : integer;
begin
  {MBM: Initialize zlib stream }
 zStream.zalloc := nil; 
 zStream.zfree  := nil; 
 zStream.opaque := nil; 

  iError := DeflateInit(zStream, Z_BEST_SPEED);
  if iError <> Z_OK then
    raise ECompressionError.Create(STR_COMPRESSION_ERROR + IntToStr(iError));
end;

procedure TZCompress.FinishDeflate(var zStream: z_Stream);
var
  iError : integer;
begin
  iError := DeflateEnd(zStream);
  if iError <> Z_OK then
    raise ECompressionError.Create(STR_COMPRESSION_ERROR + IntToStr(iError));
end;

procedure TZCompress.ExecuteDeflate(var zStream: z_Stream);
var
  iError : integer;
begin
  iError := Deflate(zStream, Z_NO_FLUSH);
  if (iError <> Z_OK) and (iError <> Z_STREAM_END) then
    raise ECompressionError.Create(STR_COMPRESSION_ERROR + IntToStr(iError));
end;

procedure TZCompress.ExecuteInflate(var zStream: z_Stream);
var
  iError : integer;
begin
  iError := Inflate(zStream, Z_NO_FLUSH);
  if (iError <> Z_OK) and (iError <> Z_STREAM_END) then
    raise ECompressionError.Create(STR_COMPRESSION_ERROR + IntToStr(iError));
end;

procedure TZCompress.InitInflate(var zStream: z_Stream);
var
  iError : integer;
begin
  {MBM: Initialize zlib stream }
 zStream.zalloc := nil; 
 zStream.zfree  := nil; 
 zStream.opaque := nil; 

  iError := InflateInit(zStream);
  if iError <> Z_OK then
    raise ECompressionError.Create(STR_COMPRESSION_ERROR + IntToStr(iError));
end;

procedure TZCompress.FinishInflate(var zStream: z_Stream);
var
  iError : integer;
begin
  iError := InflateEnd(zStream);
  if iError <> Z_OK then
    raise ECompressionError.Create(STR_COMPRESSION_ERROR + IntToStr(iError));
end;

procedure TZCompress.StreamDeflate(const msIn, msOut: TMemoryStream; pProgress: TZProgress);
{ 
  Creates a compressed stream from msIn.  The actual output stream format
  looks like this:
  
    SIZE + STREAM
}

var
  zStream : z_Stream;
  iError  : integer;  
  lSize   : longInt;
  pRead   : pointer;
  pWrite  : pointer;
  lRead   : longInt;
  lTotal  : longInt;
  msComp  : TMemoryStream;
begin
  InitDeflate(zStream);

  {MBM: Get buffers for reading & writing streams }
  GetMem(pRead,  ZLIB_READ);
  GetMem(pWrite, ZLIB_WRITE);

  {MBM: 16-bit only: Since setting the size of a TMemoryStream destroys its
        contents, we need to create a temporary stream }
  msComp := TMemoryStream.Create;
  
  try
    {MBM: Preserve the uncompressed size }
    lSize  := msIn.Size;
    msComp.SetSize(lSize + (Round(lSize * 0.01)) + 12); { See algorithm for details }

    {MBM: Setup write buffer }
    zStream.next_out  := pWrite;
    zStream.avail_out := ZLIB_WRITE;

    {MBM: Setup read buffer }
    zStream.next_in := pBytef(pRead);
    lRead := msIn.Read(pRead^, ZLIB_READ);
    zStream.avail_in  := lRead;

    while (zStream.total_in <> lSize) and (zStream.total_out < msComp.Size) do
    begin
      {MBM: Deflate }
      ExecuteDeflate(zStream);
      if @pProgress <> nil then
        pProgress(Self, zStream);

      {MBM: Continue feeding compressor }
      if zStream.avail_in = 0 then
      begin
       zStream.next_in := pBytef(pRead);
        lRead := msIn.Read(pRead^, ZLIB_READ);
       zStream.avail_in  := lRead;
      end;

      {MBM: Flush output stream }
      if zStream.avail_out = 0 then
      begin
        msComp.Write(pWrite^, ZLIB_WRITE);
        zStream.next_out  := pWrite;
        zStream.avail_out := ZLIB_WRITE;
      end;
    end;

    {MBM: Finish the stream }
    while TRUE do
    begin
      {MBM: Flush output stream }
      msComp.Write(pWrite^, pchar(zStream.next_out) - pWrite);
      zStream.next_out := pWrite;
      zStream.avail_out := ZLIB_WRITE;

      {MBM: Deflate }
      iError := Deflate(zStream, Z_FINISH);
      if @pProgress <> nil then
        pProgress(Self, zStream);
      
      if (iError = Z_STREAM_END) then
      begin
        msComp.Write(pWrite^, pchar(zStream.next_out) - pWrite);
        break;
      end;
    end;

    {MBM: Resize msOut }
    lTotal := msComp.Position;
    msOut.SetSize(lTotal + Sizeof(Integer));
    msOut.Seek(0, 0);
    msOut.Write(lSize, Sizeof(LongInt));

    {MBM: Copy the compressed stream to msOut }
    msComp.Seek(0, 0);
    msOut.CopyFrom(msComp, lTotal);
    
  finally  
    msComp.Free;
    FreeMem(pRead,  ZLIB_READ);
    FreeMem(pWrite, ZLIB_WRITE);
    FinishDeflate(zStream);
  end;
end;

procedure TZCompress.StreamInflate(const msIn, msOut: TMemoryStream; pProgress: TZProgress);
{ 
  Creates an uncompressed stream from msIn.  The actual input stream format 
  looks like this:
    SIZE + STREAM
}

var
  iError  : integer;
 zStream : z_stream; 
  lSize   : longInt;
  pRead   : pointer;
  pWrite  : pointer;
  lRead   : longInt;
begin
  InitInflate(zStream);

  {MBM: Get buffers for reading & writing streams }
  GetMem(pRead,  ZLIB_READ);
  GetMem(pWrite, ZLIB_WRITE);
  
  try
    {MBM: Initialize output stream }
    msIn.Read(lSize, Sizeof(longInt));
    msOut.SetSize(msOut.Position + lSize);

    {MBM: Setup write buffer }
    zStream.next_out  := pWrite;
    zStream.avail_out := ZLIB_WRITE;

    {MBM: Setup read buffer }
    zStream.next_in := pBytef(pRead);
    lRead := msIn.Read(pRead^, ZLIB_READ);
    zStream.avail_in := lRead;
    
    while (zStream.total_out < lSize) do
    begin
      {MBM: Inflate }
      ExecuteInflate(zStream);
      if @pProgress <> nil then
        pProgress(Self, zStream);
      
      {MBM: Continue feeding compressor }
      if zStream.avail_in = 0 then
      begin
        zStream.next_in := pBytef(pRead);
        lRead := msIn.Read(pRead^, ZLIB_READ);
        zStream.avail_in  := lRead;
      end;

      {MBM: Flush output stream }
      if zStream.avail_out = 0 then
      begin
        msOut.Write(pWrite^, ZLIB_WRITE);
        zStream.next_out  := pWrite;
        zStream.avail_out := ZLIB_WRITE;
      end;
    end;

    {MBM: Flush output stream }
    msOut.Write(pWrite^, pchar(zStream.next_out) - pWrite);

    {MBM: Deflate }
    iError := Inflate(zStream, Z_FINISH);
    if @pProgress <> nil then
      pProgress(Self, zStream);

    {MBM: Sanity check }  
    if iError <> Z_STREAM_END then
      raise ECompressionError.Create(STR_COMPRESSION_ERROR + IntToStr(iError));
    
  finally
    FreeMem(pRead,  ZLIB_READ);
    FreeMem(pWrite, ZLIB_WRITE);
    FinishInflate(zStream);
  end;
end;


end.
 