(*
    Engine for the "Yawn" game
    Copyright (c) 1995 Alessandro Scotti
*)
unit YawnTool;

interface

const
  (* Score sheet (entries 1-6 are referenced directly by number) *)
  YWN_FIRST   = 1;
  YWN_LAST    = 13;
  YWN_TRIS    = 7;
  YWN_POKER   = 8;
  YWN_FULL    = 9;
  YWN_S4      = 10;
  YWN_S5      = 11;
  YWN_YAWN    = 12;
  YWN_CHANCE  = 13;
type
  TYawnScoreSheet = array[ YWN_FIRST..YWN_LAST ] of integer;
  TYawnScoreTable = record
    Score   : word;
    Bonus   : word;
    Rating  : word;
    Average : integer;
  end;

procedure yawnSetScoreSheet( var SS: TYawnScoreSheet; D: array of word );
procedure yawnSetScoreTable( var SS: TYawnScoreSheet; var ST: TYawnScoreTable );

implementation

const
  (* Max scores for rating evaluation *)
  MAXSCORESHEET : TYawnScoreSheet = (
    5, 10, 15, 20, 25, 30, 30, 30, 25, 30, 40, 50, 30
  );

(*
    Sets each entry in the score sheet.

    Input:
        SS    = score sheet to set
        D     = dice array
    Output:
        none
*)
procedure yawnSetScoreSheet( var SS: TYawnScoreSheet; D: array of word );
var
  I, V: integer;
  S   : word; (* Bit mask use to detect straights *)
  T   : word; (* If not zero, value for three-of-a-kind (tris) *)
  C   : word; (* Sum of all dice *)
begin
  (* Initialize variables *)
  for I:=YWN_FIRST to YWN_LAST do
    SS[I] := 0;
  S := 0;
  T := 0;

  (* Get sum first *)
  C := 0;
  for I:=Low(D) to High(D) do Inc( C, D[I] );
  SS[YWN_CHANCE] := C;
  (* Run thru dice (don't check range, there should be 5 of them) *)
  for I:=Low(D) to High(D) do begin
    V := D[I];
    Inc( SS[V], V );
    if( SS[V] = V*3 ) then T := V; (* Tris *)
    if( SS[V] = V*4 ) then SS[YWN_POKER] := C; (* Poker *)
    if( SS[V] = V*5 ) then SS[YWN_YAWN] := 50; (* Yawn *)
    S := S or (1 shl (V-1)); (* Update mask for straights *)
  end;

  (* Set score sheet *)
  if( T <> 0 ) then begin
    (* Tris, also check for full *)
    SS[YWN_TRIS] := C;
    for I:=1 to 6 do
      if(( SS[I] = 2*I )and( I <> T ))or( SS[YWN_YAWN] <> 0 ) then
        SS[YWN_FULL] := 25;
  end;
  if( S and $0F = $0F )or( S and $1E = $1E )or( S and $3C = $3C ) then
    SS[YWN_S4] := 30; (* 4 in a row (small straight) *)
  if( S and $1F = $1F )or( S and $3E = $3E ) then
    SS[YWN_S5] := 40; (* 5 in a row (straight) *)
end;

(*
    Computes the score for a score sheet.

    Input:
        SS    = score sheet
        ST    = score table to set
    Output:
        none
    Note:
        for rating evaluation, unused fields should be set to
        a negative value
*)
procedure yawnSetScoreTable( var SS: TYawnScoreSheet; var ST: TYawnScoreTable );
var
  I: integer;
  B: word;
  R: word;
begin
  (* Initialize variables *)
  with ST do begin
    Score := 0;
    Bonus := 0;
    Rating := 0;
    Average := 0;
  end;

  (* Compute max score for used entries *)
  B := 0;
  R := 0;
  for I:=YWN_FIRST to YWN_LAST do begin
    if( SS[I] >= 0 ) then begin
      Inc( R, MAXSCORESHEET[I] );
      if( I <= 6 ) then Inc( B );
    end;
  end;
  if( B = 6 ) then Inc( R, 35 );

  (* Compute average and bonus *)
  for I:=1 to 6 do
    if( SS[I] >= 0 ) then
      Inc( ST.Average, SS[I] - 3*I );
  if( ST.Average >= 0 )and( B = 6 ) then
    ST.Bonus := 35;

  (* Compute score *)
  for I:=YWN_FIRST to YWN_LAST do
    if( SS[I] >= 0 ) then
      Inc( ST.Score, SS[I] );
  Inc( ST.Score, ST.Bonus );
  if( R = 0 ) then
    ST.Rating := 0
  else
    ST.Rating := (longint(ST.Score) * 100) div R;
end;

end.

