(*
  Dice component for Delphi
  Copyright (c) 1995 Alessandro Scotti
*)
unit Dice;
interface

uses
  WinTypes, WinProcs,
  Classes, Controls, Graphics, Messages;

const
  MINIMUM_SIZE        = 8;
  DEFAULT_SIZE        = 32;
  DEFAULT_BACKCOLOR   = clWhite;
  DEFAULT_FORECOLOR   = clBlack;
type
  (* A class may be derived by a number of ancestors: *)
  (* the TGraphicControl provides a canvas to paint on *)
  (* but is not a "windowed" control (this means that *)
  (* no window handle is created for this control) *)
  (* so less system resources are used. *)
  TDice = class(TGraphicControl)
    private
      (* The "private" section contains data and methods *)
      (* we don't want to show outside. For a component, *)
      (* this usually means the property fields and the *)
      (* corresponding methods. *)
      fSize     : word;
      fValue    : word;
      fBackColor: TColor;
      fForeColor: TColor;
      procedure SetSize( Value: word );
      procedure SetValue( Value: word );
      procedure SetBackColor( Value: TColor );
      procedure SetForeColor( Value: TColor );
      (* The following methods are not related to *)
      (* property management. *)
      procedure PaintDots( Dots: array of word );
      procedure WMPaint( var M: TWmPaint ); message WM_PAINT;
    protected
      (* The "protected" section contains data and methods *)
      (* which can be seen only by derived classes, but not *)
      (* by applications using *this* class. *)
      procedure Paint; override;
    public
      (* The "public" section contains data and methods *)
      (* accessible to everyone. *)
      (* This procedure "rolls" the die and updates *)
      (* the fValue field with a random number between 1 and 6. *)
      procedure Roll;
      (* It's always a good idea to make public constructors *)
      (* and destructors, otherwise the application could not *)
      (* create objects of this class! *)
      constructor Create( AOwner: TComponent ); override;
      destructor Destroy; override;
    published
      (* The "published" section contains variables and *)
      (* methods that are public *and* visible by the *)
      (* Object Inspector. For a component, variables *)
      (* are most often declared as *properties* in this *)
      (* section, so the class has control over them. *)
      (* In this example we want to intercept changes to the *)
      (* die size (width and height, being a square), so we *)
      (* tell Delphi that it has to call the method SetSize *)
      (* whenever the user wants to change the value of the *)
      (* Size property. We can then change the actual value, *)
      (* stored into the fSize private field, and redisplay *)
      (* the die with the new size. *)
      property Size: word read fSize write SetSize default DEFAULT_SIZE;
      (* We are also interested in changes of the die value, *)
      (* so we can properly redisplay it. *)
      property Value: word read fValue write SetValue;
      (* Allow the user to change back and fore color if needed, *)
      (* force a repaint on writes. *)
      property BackColor: TColor read fBackColor write SetBackColor default DEFAULT_BACKCOLOR;
      property ForeColor: TColor read fForeColor write SetForeColor default DEFAULT_FORECOLOR;
      (* These properties are defined by TGraphicControl, *)
      (* we just need to "publish" them. *)
      property DragCursor;
      property DragMode;
      property ParentShowHint;
      property ShowHint;
      property Visible;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
  end;

procedure Register;

implementation

(*
  The Create constructor is called when the object is
  created. All variables should be initialized here.
*)
constructor TDice.Create( AOwner: TComponent );
begin
  (* Call the inherited constructor first, in this case *)
  (* it will be TGraphicControl.Create *)
  inherited Create( AOwner );
  (* Initialize variables *)
  fSize := DEFAULT_SIZE;
  fValue := 1;
  fForeColor := DEFAULT_FORECOLOR;
  fBackColor := DEFAULT_BACKCOLOR;
  (* Make ancestor know *)
  SetBounds( Left, Top, fSize, fSize );
end;

(*
  The Destroy destructor is called when the object is
  destroyed. Its purpose is to release all resources
  which may be owned by the object (none for this
  component).
  Because this destructor only calls the inherited
  destructor, we could omit the declaration and just
  inherit the Destroy method.
*)
destructor TDice.Destroy;
begin
  inherited Destroy;
end;

(*
  Trap the WM_PAINT message so we can check the control
  size. This is a very simple implementation but it
  allows the die to be resized at design time quite
  easily (at run-time the application is supposed to
  use the Size property).
*)
procedure TDice.WMPaint;
var
  S: word;
begin
  (* Get min( Width, Height ) *)
  S := Height;
  if( Width < Height ) then
    S := Width;
  (* If the size has changed update it, *)
  (* otherwise continue processing the WM_PAINT message *)
  if( S < MINIMUM_SIZE ) then
    S := MINIMUM_SIZE;
  if( S <> fSize ) then
    Size := S
  else
    inherited;
end;

(*
  This little procedure draws the die dots on the
  component canvas.
  To make the job easier, dots are arranged on a 3x3
  grid, with two "slots" unused. Depending on the die
  value the Paint method decides which dots are to be
  actually displayed.
*)
procedure TDice.PaintDots( Dots: array of word );
var
  D, I, Radius: integer;
  X, Y: integer;
begin
  (* Compute radius length based on die size *)
  Radius := fSize * 2 div 7;
  (* Compute offset of first (top left) dot *)
  D := (fSize mod Radius) div 2 + 1;
  (* Draw the specified dots *)
  for I:=Low(Dots) to High(Dots) do begin
    X := (Dots[I] mod 3)*Radius + D;
    Y := (Dots[I] div 3)*Radius + D;
    Canvas.Ellipse( X, Y, X+Radius-1, Y+Radius-1 );
  end;
end;

(*
  Draw the die.
*)
procedure TDice.Paint;
var
  R: TRect;
begin
  with Canvas do begin
    (* Erase background *)
    Brush.Color := fBackColor;
    Brush.Style := bsSolid;
    R := Rect( 0, 0, fSize, fSize );
    FillRect( R );
    (* Draw dots *)
    Brush.Color := fForeColor;
    Pen.Color := fForeColor;
    Pen.Style := psSolid;
    case fValue of
      1: PaintDots( [4] );
      2: PaintDots( [0,8] );
      3: PaintDots( [0,4,8] );
      4: PaintDots( [0,2,6,8] );
      5: PaintDots( [0,2,4,6,8] );
      6: PaintDots( [0,2,3,5,6,8] );
    end;
  end;
end;

(*
  Changes the die size.
*)
procedure TDice.SetSize( Value: word );
begin
  (* Check range *)
  if( Value < MINIMUM_SIZE )or( Value = fSize ) then
    Exit;
  (* Update field (not property!) and force redisplay *)
  fSize := Value;
  SetBounds( Left, Top, fSize, fSize );
  Invalidate;
end;

(*
  Changes the die value.
*)
procedure TDice.SetValue( Value: word );
begin
  (* Check range (0 means a blank die) *)
  if( Value > 6 ) then
    Exit;
  (* Update field (not property!) and force redisplay *)
  fValue := Value;
  Invalidate;
end;

(*
  Sets the background color.
*)
procedure TDice.SetBackColor;
begin
  fBackColor := Value;
  Invalidate;
end;

(*
  Sets the foreground color.
*)
procedure TDice.SetForeColor;
begin
  fForeColor := Value;
  Invalidate;
end;

(*
  Rolls the die.
*)
procedure TDice.Roll;
begin
  (* Assign a random value to the Value property (because *)
  (* we use the property the SetValue method is called and *)
  (* redisplay is automatic) *)
  Value := Random( 6 ) + 1;
end;

(*
  This procedure is called by Delphi to register
  the component into the proper palette page.
*)
procedure Register;
begin
  RegisterComponents( 'Samples', [TDice] );
end;

end.
