Unit WWave;

Interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls;

Type
  TWhereWave = (wwInFile, wwInExe);
  TWaveOption = (woSync, woNoDefault, woLoop, woNoStop);
  TWaveOptions = set of TwaveOption;
  {
  EWaveException = class(Exception);
  }
  TWave = class(TComponent)
    private
      FWhereWave : TWhereWave;
      FWaveName : PChar;
      FResourceType : PChar;
      FOptions : word;
      FPalyOptions : TWaveOptions;
      procedure SetWhereWave(Value : TWhereWave);
      procedure SetWaveName(Value : string);
      function GetWaveName : string;
      procedure SetResourceType(Value : string);
      function GetResourceType : string;
    protected
      constructor create(aOwner : TComponent); override;
      destructor Destroy; override;
    public
      function Execute : boolean;
      function StopSound : boolean;
    published
      property WhereWave : TWhereWave read FWhereWave write SetWhereWave;
      property WaveName : string read GetWaveName write SetWaveName;
      property ResourceType : string read GetResourceType write SetResourceType;
      property Options : TWaveOptions read FPalyOptions write FPalyOptions default [woSync, woNoDefault, woNoStop];
  End;

Procedure Register;

Implementation

Uses mmsystem;

  Constructor TWave.Create(aOwner : TComponent);
  Begin
    Inherited Create(aOwner);
    Options := [woSync, woNoDefault, woNoStop];
    GetMem(FWaveName, 256);
    GetMem(FResourceType, 256);
    WaveName := '';
    ResourceType := '';
  End;

  destructor TWave.Destroy;
  begin
    FreeMem(FResourceType, 256);
    FreeMem(FWaveName, 256);
    Inherited Destroy;
  end;

  Procedure TWave.SetWhereWave(Value : TWhereWave);
  Begin
    If Value <> FWhereWave
    Then
      FWhereWave := Value;
  End;

  Procedure TWave.SetWaveName(Value : String);
  Begin
    FWaveName := StrPCopy(FWaveName, Value);
  End;

  function TWave.GetWaveName : String;
  Begin
    Result := StrPas(FWaveName);
  End;

  Procedure TWave.SetResourceType(Value : String);
  Begin
    FResourceType := StrPCopy(FResourceType, Value);
  End;

  function TWave.GetResourceType : String;
  Begin
    Result := StrPas(FResourceType);
  End;

  Function TWave.Execute : Boolean;
  Var
    FindResHandle : THandle;
    LoadResHandle : THandle;
    LockResPointer : Pointer;
  Begin
    Result := False;
    If WaveOutGetNumDevs <= 0
    Then
      Abort {You have not any sound devices}
      (*
      raise EWaveException.Create('You have not any sound devices')
      *)
    Else
      Begin
        FOptions := 0;
        If woSync In Options
        Then
          FOptions := FOptions Or SND_SYNC
        Else
          FOptions := FOptions Or SND_ASYNC;
        If woNoDefault In Options
        Then
          FOptions := FOptions Or SND_NODEFAULT;
        If woLoop In Options
        Then
          FOptions := FOptions Or SND_LOOP;
        If woNoStop In Options
        Then
          FOptions := FOptions Or SND_NOSTOP;
        If FWhereWave = wwInFile
        Then
          Result := sndPlaySound(FWaveName, FOptions)
        Else
          Begin
            FindResHandle := FindResource(HInstance, FWaveName, FResourceType);
            If FindResHandle = 0
            Then
              Abort {Resource not found}
              (*
              raise EWaveException.Create('Resource not found')
              *)
            Else
              Begin
                LoadResHandle := LoadResource(HInstance, FindResHandle);
                If LoadResHandle = 0
                Then
                  Abort {Resource not load}
                  (*
                  raise EWaveException.Create('Resource not load')
                  *)
                Else
                  Begin
                    LockResPointer := LockResource(LoadResHandle);
                    If LockResPointer = Nil
                    Then
                      begin
                        FreeResource(LoadResHandle);
                        Abort; {Can't lock resource}
                        (*
                        raise EWaveException.Create('Can not lock resource');
                        *)
                      end
                    Else
                      Begin
                        Result := sndPlaySound(LockResPointer, FOptions Or SND_MEMORY);
                        Result := Result Or FreeResource(LoadResHandle);
                      End;
                  End;
              End;
          End;
      End;
  End;

  Function TWave.StopSound : Boolean;
  Begin
    Result := sndPlaySound(#0, 0);
  End;

  Procedure Register;
  Begin
    RegisterComponents('Samples', [TWave]);
  End;

End.
