{**********************************************************************

  TVolumeInformation. A non visual component that encapsulates Win32's
                      GetVolumeInformation function

  Copyright (C) 1997 Andrea Mennini

  Version: 1.01
  Date: Jul 23, 1997

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library in the file LGPL.TXT; if not, write to the
  Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
  MA 02139, USA.

  Comments, suggestions, enhancements and bug reports are obviously welcome.

  The author can be contacted at the following addresses:

  Snail mail:

    Andrea Mennini
    via E. Manfredi 2/2
    I-40138 Bologna BO
    Italy

  E-mail:
    jake@blues.dsnet.it


  Revision history:
  -----------------
  Version: 1.01 - Jul 23, 1997
    - Fixed bug: The volume label of networked drives was not correctly
                 retrieved. By the way, this was a Windows NT bug, not a
                 TVolumeInfo one. Anyway, it's FIXED.

    - New features: Added property: UNCDrive, runtime and readonly. UNCDrive
                    for networked drives is a real UNC file name of the form:

                    \\servername\sharename

                    For local drives, it's the root file name of the form:

                    driveletter:\

  Version: 1.00 - Feb 14, 1997: Initial release

**********************************************************************}
unit Volinfo;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TErrorEvent = procedure(ErrorNumber: integer) of Object;
  TVolumeInformation = class(TComponent)
  private
    { Private declarations }
    FDrive: char;
    FUNCDrive: string;
    FErrorEvent: TErrorEvent;
    FVersion: string;
    function FCaseIsPreserved: boolean;
    function FCaseSensitive: boolean;
    function FUnicodeStoredOnDisk: boolean;
    function FPersistentACLs: boolean;
    function FSupportsFileCompression: boolean;
    function FVolumeIsCompressed: boolean;
    function GetUNCDrive: string;
    function GetVolumeFileSystemFlags: integer;
    function GetVolumeFileSystemName: string;
    function GetVolumeLabel: string;
    function GetVolumeSerialNumber: string;
    procedure SetDrive(DriveLetter: char);
    procedure SetUNCDrive;
  protected
    { Protected declarations }
    procedure ProcessError(Result: Variant);
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    property CaseIsPreserved: boolean read FCaseIsPreserved;
    property CaseSensitive: boolean read FCaseSensitive;
    property UnicodeStoredOnDisk: boolean read FUnicodeStoredOnDisk;
    property PersistentACLs: boolean read FPersistentACLs;
    property SupportsFileCompression: boolean read FSupportsFileCompression;
    property Version: string read FVersion;
    property VolumeIsCompressed: boolean read FVolumeIsCompressed;
    property VolumeFileSystemFlags: integer read GetVolumeFileSystemFlags;
    property VolumeFileSystemName: string read GetVolumeFileSystemName;
    property VolumeLabel: string read GetVolumeLabel;
    property VolumeSerialNumber: string read GetVolumeSerialNumber;
  published
    { Published declarations }
    property Drive: char read FDrive write SetDrive;
    property UNCDrive: string read GetUNCDrive;
    property OnError: TErrorEvent read FErrorEvent write FErrorEvent;
  end;

procedure Register;

implementation

constructor TVolumeInformation.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDrive := 'C';
  FVersion := '1.01'
end;

{*****************************************************************************
  procedure ProcessError - Handles the GetVolumeInformation errors. If there
                           is an User Event defined for the component, it gets
                           called with the error code passed as parameter.
                           Otherwise a default error box is displayed.
                           In any case, a null value is returned: this means
                           a null string ('') or a null integer(0).

                           Result is of type variant in order to make this
                           routine good both for string and integer return
                           types.
 *****************************************************************************}
procedure TVolumeInformation.ProcessError(Result: Variant);
begin
  if Assigned(FErrorEvent) then FErrorEvent(GetLastError)
  else
    Application.MessageBox(PChar('Error ' + IntToStr(GetLastError) +
                           ' in GetVolumeInformation' + #13 + #10 +
                           #13 + #10 +
                           'TVolumeInformation is (C) 1997 by Andrea Mennini, jake@blues.dsnet.it'),
                           'TVolumeInformation',
                           (MB_ICONWARNING or MB_OK));

    if (VarType(Result) and varTypeMask) =  varString then
      Result := ''
    else if (VarType(Result) and varTypeMask) =  varInteger then
      Result := 0
    else
      Application.MessageBox(PChar('Error in ProcessError call!!!!' + #13 + #10 +
                           #13 + #10 +
                           'TVolumeInformation is (C) 1997 by Andrea Mennini, jake@blues.dsnet.it'),
                           'TVolumeInformation',
                           (MB_ICONWARNING or MB_OK));
end;

{*****************************************************************************
  function GetUNCDrive - Gets the Volume label for the specified drive.

  Input parameters: NONE
  Output parameters: The function returns a string containing the UNC file name,
                     that could have two possible forms:

                       <driveletter>:\

                     if FDrive is local, and

                       \\<servername>\<sharename>

                     if FDrive is remote, i.e. networked.

                     Since FUNCDrive could have an ending backslash to override
                     a NT bug with networked drives, it removes also the last
                     '\'.
 *****************************************************************************}
function TVolumeInformation.GetUNCDrive: string;
begin
  if ((FUNCDrive[1] = '\') and (FUNCDrive[2] = '\')) then
    Result := Copy(FUNCDrive, 1, Length(FUncDrive) - 1)
  else
    Result := FUNCDrive;
end;

{*****************************************************************************
  function GetVolumeLabel - Gets the Volume label for the specified drive.

  Input parameters: NONE
  Output parameters: The function returns a string containing the volume name,
                     or an empty string if some error occurs.
 *****************************************************************************}
function TVolumeInformation.GetVolumeLabel: string;
var
  OldErrorMode: Integer;
  NotUsed, VolFlags: Integer;
  Buf: array [0..MAX_PATH] of Char;
  Error: boolean;
begin
  OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    SetUNCDrive;
    Error := GetVolumeInformation(PChar(FUNCDrive),
                                  Buf,
                                  SizeOf(Buf),
                                  nil,
                                  NotUsed,
                                  VolFlags,
                                  nil,
                                  0);
    if (Error = False) then
    begin
      ProcessError(Result);
      Exit;
    end;
    SetString(Result, Buf, StrLen(Buf));
    if FDrive < 'a' then
      Result := AnsiUpperCase(Result)
    else
      Result := AnsiLowerCase(Result);
  finally
    SetErrorMode(OldErrorMode);
  end;
end;

{*****************************************************************************
  function GetVolumeSerialNumber - Gets the Volume serial number for the
                                   specified drive.

  Input parameters: NONE
  Output parameters: The function returns a string containing the volume serial
                     number, or an empty string if some error occurs.
 *****************************************************************************}
function TVolumeInformation.GetVolumeSerialNumber: string;
var
  OldErrorMode: Integer;
  NotUsed, VolFlags: Integer;
  VolSN: DWORD;
  VolumeSerialNumber: string;
  Error: boolean;
begin
  OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    SetUNCDrive;
    Error := GetVolumeInformation(PChar(FUNCDrive),
                                  nil,
                                  0,
                                  @VolSN,
                                  NotUsed,
                                  VolFlags,
                                  nil,
                                  0);

    if (Error = False) then
    begin
      ProcessError(Result);
      Exit
    end;

    VolumeSerialNumber := AnsiUpperCase(IntToHex(VolSN, 8));
    Result := Copy(VolumeSerialNumber, 1, 4) + ':' +
              Copy(VolumeSerialNumber, 5, 4);
  finally
    SetErrorMode(OldErrorMode);
  end;
end;

{*****************************************************************************
  function GetVolumeFileSystemName - Gets the file system name (usually FAT or
                                     NTFS) for the specified drive.

  Input parameters: NONE
  Output parameters: The function returns a string containing the file system
                     name, or an empty string if some error occurs.
 *****************************************************************************}
function TVolumeInformation.GetVolumeFileSystemName: string;
var
  OldErrorMode: Integer;
  NotUsed, VolFlags: Integer;
  Buf: array [0..MAX_PATH] of Char;
  Error: boolean;
begin
  OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    SetUNCDrive;
    Error := GetVolumeInformation(PChar(FUNCDrive),
                                  nil,
                                  0,
                                  nil,
                                  NotUsed,
                                  VolFlags,
                                  Buf,
                                  SizeOf(Buf));
    if (Error = False) then
    begin
      ProcessError(Result);
      Exit
    end;

    SetString(Result, Buf, StrLen(Buf));
    if FDrive < 'a' then
      Result := AnsiUpperCase(Result)
    else
      Result := AnsiLowerCase(Result);
  finally
    SetErrorMode(OldErrorMode);
  end;
end;

{*****************************************************************************
  function GetVolumeFileSystemFlags - Gets the Volume file system flags for
                                      the specified drive.

  Input parameters: NONE
  Output parameters: The function returns an integer containing the volume file
                     system flags, or 0 (zero) if some error occurs.
 *****************************************************************************}
function TVolumeInformation.GetVolumeFileSystemFlags: integer;
var
  OldErrorMode: Integer;
  NotUsed, VolFlags: Integer;
  Buf: array [0..MAX_PATH] of Char;
  Error: boolean;
begin
  OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    SetUNCDrive;
    Error := GetVolumeInformation(PChar(FUNCDrive),
                                  nil,
                                  0,
                                  nil,
                                  NotUsed,
                                  VolFlags,
                                  nil,
                                  0);
    if (Error = False) then
    begin
      ProcessError(Result);
      Exit
    end;

    Result := VolFlags;
  finally
    SetErrorMode(OldErrorMode);
  end;
end;

{*****************************************************************************
  function CaseIsPreserved - Return the FS_CASE_IS_PRESERVED bit status.

  Input parameters: NONE
  Output parameters: The function returns True if FS_CASE_IS_PRESERVED is set,
                     False otherwise.
 *****************************************************************************}
function TVolumeInformation.FCaseIsPreserved: boolean;
begin
  if (GetVolumeFileSystemFlags and FS_CASE_IS_PRESERVED) <> 0 then
    Result := True
  else
    Result := False;
end;

{*****************************************************************************
  function CaseIsPreserved - Return the FS_CASE_SENSITIVE bit status.

  Input parameters: NONE
  Output parameters: The function returns True if FS_CASE_SENSITIVE is set,
                     False otherwise.
 *****************************************************************************}
function TVolumeInformation.FCaseSensitive: boolean;
begin
  if (GetVolumeFileSystemFlags and FS_CASE_SENSITIVE) <> 0 then
    Result := True
  else
    Result := False;
end;

{*****************************************************************************
  function CaseIsPreserved - Return the FS_UNICODE_STORED_ON_DISK bit status.

  Input parameters: NONE
  Output parameters: The function returns True if FS_UNICODE_STORED_ON_DISK is
                     set, False otherwise.
 *****************************************************************************}
function TVolumeInformation.FUnicodeStoredOnDisk: boolean;
begin
  if (GetVolumeFileSystemFlags and FS_UNICODE_STORED_ON_DISK) <> 0 then
    Result := True
  else
    Result := False;
end;

{*****************************************************************************
  function CaseIsPreserved - Return the FS_PERSISTENT_ACLS bit status.

  Input parameters: NONE
  Output parameters: The function returns True if FS_PERSISTENT_ACLS is set,
                     False otherwise.
 *****************************************************************************}
function TVolumeInformation.FPersistentACLs: boolean;
begin
  if (GetVolumeFileSystemFlags and FS_PERSISTENT_ACLS) <> 0 then
    Result := True
  else
    Result := False;
end;

{*****************************************************************************
  function CaseIsPreserved - Return the FS_FILE_COMPRESSION bit status.

  Input parameters: NONE
  Output parameters: The function returns True if FS_FILE_COMPRESSION is set,
                     False otherwise.
 *****************************************************************************}
function TVolumeInformation.FSupportsFileCompression: boolean;
begin
  if (GetVolumeFileSystemFlags and FS_FILE_COMPRESSION) <> 0 then
    Result := True
  else
    Result := False;
end;

{*****************************************************************************
  function CaseIsPreserved - Return the FS_VOL_IS_COMPRESSED bit status.

  Input parameters: NONE
  Output parameters: The function returns True if FS_VOL_IS_COMPRESSED is set,
                     False otherwise.
 *****************************************************************************}
function TVolumeInformation.FVolumeIsCompressed: boolean;
begin
  if (GetVolumeFileSystemFlags and FS_VOL_IS_COMPRESSED) <> 0 then
    Result := True
  else
    Result := False;
end;

procedure TVolumeInformation.SetDrive(DriveLetter: char);
begin
  if FDrive <> DriveLetter then
    FDrive := DriveLetter;
end;

{*****************************************************************************
  procedure SetUNCDrive - Sets the FUNCDrive, that is UNC file name of the root
                          directory of the drive specified by FDrive. If FDrive
                          is local, FUNCDrive is of the form:

                            <driveletter>:\

                         If FDrive is networked, FDrive has the form:

                            \\<servername>\<sharename>

  Input parameters: NONE
  Output parameters: NONE.
  Note: Due to a bug in Windows NT, the UNC name passed to GetVolumeInformation
        must have an extra backslash at its end, e.g. it MUST be of the form:

          \\<servername>\<sharename>\

 *****************************************************************************}
procedure TVolumeInformation.SetUNCDrive;
var RemoteNameInfo: array[0..1023] of Byte;
    Size: integer;
begin
  FUNCDrive := FDrive + ':\';
  if GetDriveType(PChar(FUNCDrive)) = DRIVE_REMOTE then
  begin
    Size := SizeOf(RemoteNameInfo);
    if WNetGetUniversalName(PChar(FUNCDrive), UNIVERSAL_NAME_INFO_LEVEL,
      @RemoteNameInfo, Size) <> NO_ERROR then
    begin
       ProcessError('');
       Exit;
    end;
    FUNCDrive := PRemoteNameInfo(@RemoteNameInfo).lpUniversalName + '\';
  end
end;

procedure Register;
begin
  RegisterComponents('Samples', [TVolumeInformation]);
end;

end.
