unit VarArray;
(*
____________________
Created by Michael Rynn

	michrynn@ozemail.com.au
____________________

Freeware:
	Feel free to use, please share any bug reports, fixes or enhancements;

For all delphi versions.

This unit supplies a base type, TVarArray,for manipulation of
dynamic arrays of fixed size elements. Its probably not much different from other
dynamic array units you may have seen, except its pretty basic.


The distinguished property of the TVarArray
is that  member elements which require futher initialization and/or additional
dynamic storage can be "constructed" and "destructed" via the protected virtual
methods of ConstructItem and DestructItem. Resizing, insertions and deletions
are supported, in which ConstructItem and DestructItem are called appropriately.

An example of this is given below with the class TStrStrVArray, a sorted array
which stores a record type with 2 strings, one being a variable name
and the other its value.

ConstructItem and DestructItem will be called if the TVarArray is
constructed  with CreateBase(itemsize, initItems = TRUE). If initItems is false,
the ConstructItem and DestructItem will not be called.  Please note the base
constructor CreateBase does not allocate any memory for the array. This is done
in derived class constructors by setting the Count propery.

TIntVArray, TDoubleVArray subclass the TVarArray for integer and double types,
without really adding much else.

TIntVArray associates an integer with a Delphi long string type. There doesn't seem
to be much use for it, but I have used it for one project.

____________________
The System ReallocMem routine is used for array re-sizing.
A hidden array element at index (-1) supports a generic element swap.
Because of this, setting count to zero does not deallocate the block,
which can only be freed by setting Count := -1; // done in Destroy

Reallocation changes the BlockPtr (which points to the zeroth element,
not the start of the allocated memory block).
Access to a type-specific pointer is facilitated in sub-classes
by setting FBlockHdl as a pointer to your type-specific pointer, so that
it will always be automatically updated after a ReallocMem call.
____________________
GetItemIndex and Sort routines both use the CompareItems method.
____________________
If setting capacity in advance, turn AutoShrink off so that capacity will not
shrink again after the first change to count eg by insertion or deletion.
Capacity cannot be reduced to below count size.
____________________
Packed record types

eg
{$A+}
   TArrayElement = record
		byte1, byte2, byte3 : Byte;   {4 bytes}
   end;

   TArray = array[0..n] of TArrayElement {400 bytes}

Note that if the word packed is used in the record

  TArrayElement = packed record
		byte1, byte2, byte3 : Byte;    {3 bytes}
   end;

It doesn't seem to matter to the compiler if the array type is
declared as packed or not. But as TVarArray always treats the
memory block as a packed array of type, it should probably be
always declared as such.

TArray = packed array[0..99] of TArrayElement ;  {300 bytes}
TArray = array[0..99] of TArrayElement ;  {300 bytes}

Significant memory savings can sometimes be made with packed record types,
the trade off being the alignment access efficiency.
_____________________
History
MR
May 98:  A quick and dirty array type created for a project, and it grew a little
July 98: Use Assert. TStrStrVArray added. Insert and Delete Items for TVarArray
Sept 98: Delphi 1 support code added
Nov  98: Revised & documented for release 1
*)

interface
Uses SysUtils,Classes;

const
	kMaxIntArraySize = MAXINT div sizeof(INTEGER) - 1;
	kMaxDblArraySize = MAXINT div sizeof(double) - 1;
	kIndexRangeError = 'UArray: Index out of range';
{$IFNDEF WIN32}
    kMaxMemBlock = 65533;
{$ENDIF}

type
	PByte = ^Byte;
	PWord = ^WORD;
	PDouble = ^Double;
	PInteger = ^Integer;

	EVarArray = class(Exception);
	EStrStrVArray = class(EVarArray);

    EVarArrayIndexError = class(EVarArray)
    	constructor Create(ix:integer);
    end;

    TBlockHdl = ^Pointer;

	TVASize = LongInt;

	TVarArray = class(TObject)
    private
		FItemSize : integer;
		FCount : TVASize;
		FCapacity : TVASize;

		FBlock : Pointer;
		FInitItems : boolean; {only use in constructor}
		FAutoShrink : boolean; {don't lose capacity}

        procedure CallConstruct(ixfrom, ccount:integer);
        procedure CallDestruct(ixfrom, dcount:integer);

    protected
 		FBlockHdl : TBlockHdl;

		procedure SetCount(inCount : TVASize); virtual;
		procedure DestructItem(itemPtr : pointer);virtual;
		procedure ConstructItem(itemPtr : pointer);virtual;

		procedure SwapItems(p1, p2 : Pointer);virtual;
		{compare: return  ordered result p1 - p2}
		function  CompareItems(p1, p2 : Pointer):integer;virtual;


		function  SwapItemBuffer : PByte;
		function  GetItemPointer(ix : integer) : Pointer;

		procedure SetCapacity(incapacity:TVASize);

	public
		constructor CreateBase(inItemSize: integer; initItems : boolean);
		destructor Destroy; override;

		procedure BubbleSort; virtual;
		procedure QuickSort; virtual;

 		procedure Clear;

 		{GetItemIndex: if true, ix is the items index,
          if false, ix is the insertion index}
		function  GetItemIndex(var ix : integer; const buf):boolean;

		{IndexOf: >= 0 if valid index, -1 if not found}
		function IndexOfItem(const buf) : integer;
		function  Compare(ix1, ix2 : integer):integer;
		procedure SetItem(ix : integer; const buf);
        procedure GetItem(ix : integer; var buf);
		procedure Swap(i1,i2:integer);

		property Ptr[ix : integer] : Pointer read GetItemPointer;

		procedure InsertItemsAt(ix, incount : integer); { move items up}
		procedure DeleteItemsAt(ix, incount : integer); {delete space and move items down}
		procedure Delete(ix:Integer); {delete 1 item}

		property ItemSize : integer read FItemSize;
		property Count : TVASize read FCount write SetCount;
		property BlockPtr : Pointer read FBlock;

		property AutoShrink : boolean read FAutoShrink write FAutoShrink;
		property Capacity : integer read FCapacity write SetCapacity;

		end;

	TDoubleArrayType = packed array[0..kMaxDblArraySize] of double;
    PDoubleArrayType = ^TDoubleArrayType;

	TDoubleVArray = class(TVarArray)
    private
    	FPtr : PDoubleArrayType;
	public
		constructor Create(inCount : integer);

		function GetValue(ix : integer) : double;
		procedure SetValue(ix : integer; invalue:double);

		property Value[ix:integer] : double read GetValue write SetValue; default;
        property Ptr : PDoubleArrayType read FPtr;
	end;

	TIntVABuf = packed array[0..kMaxIntArraySize-1] of integer;
	PIntVABuf = ^TIntVABuf;

	TIntVArray = class(TVarArray)
	protected
		FIntVABuf : PIntVABuf;
	public
		constructor Create(inCount : integer);

		function GetValue(ix : integer) : integer;
		procedure SetValue(ix : integer; invalue:integer);

		property Value[ix:integer] : integer read GetValue write SetValue; default;
		property Buffer : PIntVABuf read FIntVABuf;
	end;
{$IFNDEF WIN32}
    TVAString = pstring;
{$ELSE}
    TVAString = string;
{$ENDIF}
	TIntStrRec = record
    	ival : integer;
        sval : TVAString;
    end;
    PIntStrRec = ^TIntStrRec;

    TIntStrVArray = class(TVarArray)
    protected
    	procedure DestructItem(itemPtr : pointer);override;
		procedure ConstructItem(itemPtr : pointer);override;

    public
    	constructor Create(inCount : integer);
    	procedure GetIntStr(ix:integer; var vint:integer; var vs : string);
        procedure SetIntStr(ix:integer; vint:integer; const vs:string);

    end;

	TStrStrRec = record
		nval : TVAString;
		sval : TVAString;
	end;
	PStrStrRec = ^TStrStrRec;

	TStrStrVArray = class(TVarArray)
		FCaseSensitive : boolean;

    protected
		procedure DestructItem(itemPtr : pointer);override;
		procedure ConstructItem(itemPtr : pointer);override;

	public
		constructor Create(inCount : integer);

		procedure Assign(ssv :TStrStrVArray);

		procedure GetStrStr(ix:integer; var pname,pvalue:string);
		procedure SetStrStr(ix:integer; const pname,pvalue:string);

		function  GetVariableIndex(var ix : integer; const pname:string) : boolean;
		Function  IndexOf(const pname : string) : integer;

		Procedure SetIndexValue(ix : integer; const s : string);
		Procedure SetIndexName(ix : integer; const s : string); {will need sort}

        Function  GetIndexValue(ix :Integer):string;
        Function  GetIndexName(ix : integer):string;


		function CompareItems(p1, p2 : Pointer):integer; override;
		procedure SetVariable(const pname, pvalue:string);
		procedure  GetVariable(const pname : string; var pvalue:string);
		procedure AssignValueStrings(slist : TStrings);
		procedure AppendValueStrings(slist : TStrings);

		procedure SetValueStrings(slist : TStrings);

		property CaseSensitive : boolean read FCaseSensitive write FCaseSensitive;
	end;

{$IFNDEF WIN32}
procedure ReallocMem16(var FBlock:pointer; blksize:LongInt);
{$ENDIF}

implementation
const
	kAllBitsSet : integer = -1;
    kMaxStepSize = 4096;          

{$IFNDEF WIN32}
         const
              kHdrMemBlock = 2;
{*DEFINE ASSERT_P}

{$IFDEF ASSERT_P}
         procedure ASSERT(cond:boolean; const msg:string);
         begin
              if not cond then
                 raise EVarArray.Create(msg);
         end;
{$ENDIF}
         { the ReallocMem interface is different for delphi 1
           This routine tries to duplicate the functionality of
           the Delphi 3 ReallocMem }

         procedure ReallocMem16(var FBlock:pointer; blksize:LongInt);
         var
            wp : PWord;
            oldsize : WORD;

         begin
            if blksize > kMaxMemBlock then
               raise EVarArray.Create('ReallocMem16: request exceeds maximum');

            if (FBlock=nil) and (blksize > 0) then begin
               oldsize := blksize+ kHdrMemBlock;

               GetMem(wp, oldsize);
               if wp = nil then
               		raise EVarArray.Create('ReallocMem16: GetMem returned nil');

               wp^ := blksize;
               Inc(wp);
               FBlock := wp;
            end else begin
               wp := FBlock;
               Dec(wp);
               oldsize := wp^;
               if (blksize <> oldsize) then begin
                  if blksize = 0 then begin
                     FreeMem(wp, oldsize+kHdrMemBlock);
                     FBlock := nil;
                  end else begin
                     wp := ReallocMem(wp,oldsize+kHdrMemBlock,blksize+kHdrMemBlock);
                     wp^ := blksize;
                     Inc(wp);
                     FBlock := wp;
                  end;
               end;
            end;

         end;

         procedure Finalize(var s : TVAString);
         begin
              if (s <> nil) then
                 DisposeStr(s);
         end;

         function GetTStringsName(const s : string):string;
         var
         	i : integer;
         begin
         	for i := 1 to Length(s) do begin
                   if (s[i] = '=') then begin
                   	Result := Copy(s,1,i-1);
                        exit;
                   end;
                end;
                Result := '';
         end;
{$ELSE}
{$DEFINE ASSERT_P}
{$ENDIF}



constructor EVarArrayIndexError.Create(ix:integer);
begin
	inherited Create('UArray: Index out of range '+IntToStr(ix));
end;


procedure TVarArray.Clear;
begin
	SetCount(0);
end;


constructor TVarArray.CreateBase(inItemSize: integer; initItems : boolean);
begin
	inherited Create;
	FItemSize := inItemSize;
	FInitItems := initItems;
	FAutoShrink := TRUE;
	FCapacity := -1;
end;


function  TVarArray.Compare(ix1, ix2 : integer):integer;
var
	p1, p2 : PByte;
begin
	p1 := PByte(FBlock);
	p2 := p1;
	Inc(p1,ix1 * FItemSize);
	Inc(p2,ix2 * FItemSize);
	Result := CompareItems(p1, p2);
end;


procedure TVarArray.QuickSort;

	procedure RecursiveSort(Left, Right: Integer);
	var
		I, J: Integer;
		M : integer;
	begin
		repeat
			I := Left;
			J := Right;
			M := (Left + Right) div 2;

			repeat

				while Compare(I,M) < 0 do
					Inc(I);
				while Compare(J,M) > 0 do
					Dec(J);
				if I <= J then begin
					Swap(I,J);
					Inc(I);
					Dec(J);
				end;
			until I > J;
			if Left < J then RecursiveSort(Left, J);
			Left := I;
		until I >= Right;
	end;
begin
  RecursiveSort(0, Count-1);
end;

function TVarArray.IndexOfItem(const buf) : integer;
begin
	if not GetItemIndex(Result, buf) then
		Result := -1;
end;

function TVarArray.GetItemIndex(var ix : integer; const buf) : boolean;
var
	s, e, mid : integer;
	match : PByte;
    base :  PByte;
	cpres : integer;
begin
	s := 0;
	e := FCount-1;
    Result := FALSE;
    base := FBlock;
    if s <= e then
    repeat
		mid := (e - s) div 2 + s;

		match := base;
		Inc(match,mid*FItemSize);
        cpres := CompareItems(match,@buf);
        if (cpres > 0) then {spot is below mid}
        	e := mid-1
        else if (cpres < 0) then {spot is above mid}
        	s := mid+1
        else begin
			Result := TRUE;
			ix := mid; {this is a found match index}
            exit;
			end;
    until(s>e);
	ix := s;  {return insert index}
end;

destructor TVarArray.Destroy;
begin
	Count := -1;
    inherited Destroy;
end;

function  TVarArray.GetItemPointer(ix : integer) : Pointer;
begin
	Result := FBlock;
	Inc( PByte(Result), ix * FItemSize);
end;


procedure  TVarArray.BubbleSort;
var
	ix : integer;
	ip1, ip2 : PByte;

begin
	ix := 1;

	while (ix < Count)  do
	begin
		ip1 := Ptr[ix-1];
		ip2 := Ptr[ix];

		if CompareItems(ip1,ip2) > 0 then begin
			SwapItems(ip1,ip2);
			if (ix > 1) then
				Dec(ix);
		end else begin
			Inc(ix);
		end;
	end;
end;

procedure TVarArray.CallConstruct(ixfrom, ccount:integer);
var
	pb : PByte;
    i : integer;
begin
	pb := PByte(FBlock);
	Inc(pb,ixfrom*FItemSize);
    for i := 0 to ccount-1 do begin
 		ConstructItem(pb);
        Inc(pb,FItemSize);
    end;
end;

procedure TVarArray.CallDestruct(ixfrom, dcount:integer);
var
	pb : PByte;
    i : integer;
begin
	pb := PByte(FBlock);
	Inc(pb,ixfrom*FItemSize);
    for i := 0 to dcount-1 do begin
 		DestructItem(pb);
        Inc(pb,FItemSize);
    end;
end;


procedure TVarArray.InsertItemsAt(ix, incount : integer);
var
	oldcount : integer;
	saveInitItems : boolean;
	src,dest : PByte;
	bytesmoved : integer;

begin
	if incount <= 0 then
		exit;
{$IFDEF ASSERT_P}
	ASSERT( ((ix >= 0) AND (ix <= FCOUNT)), 'Insert range error');
{$ENDIF}
	oldcount := FCount;
	saveInitItems := FInitItems;
	FInitItems := FALSE; {new items will be clobbered so turn it off}
try
	SetCount(incount+FCount);  {this creates uninitialised space at end}
	src := PByte(FBlock);
	Inc(src,ix*FItemSize);

	if(ix <> oldcount) then begin
		dest := src;
		bytesmoved := (oldcount-ix)*FItemSize;
 		Inc(dest,incount*FItemSize);
		Move(src^,dest^,bytesmoved);
	end;
finally
	FInitItems := saveInitItems;
end;
	if FInitItems then  {initialise inserted items}
    	CallConstruct(ix,incount);
end;

procedure TVarArray.DeleteItemsAt(ix, incount : integer);
var
	dest,src : PByte;
	bytesmoved : integer;
	saveInitItems : boolean;
begin
	if (incount <= 0) then exit;
{$IFDEF ASSERT_P}
	ASSERT( ((ix >= 0) AND (ix+incount <= FCOUNT)), 'Insert range error');
{$ENDIF}
	dest := PByte(FBlock);
	Inc(dest, ix*FItemSize);
	bytesmoved := (FCount - incount - ix)*FItemSize;

	if FInitItems then begin
    	CallDestruct(ix,incount);
	end;
	if bytesmoved > 0 then begin
 		src := dest;
		Inc(src, incount*FItemSize);
		Move(src^,dest^,bytesmoved);
	end;
	saveInitItems := FInitItems;
	FInitItems := FALSE;  {items have been deleted already so turn it off}
try
	SetCount(FCount - incount);
finally
	FInitItems := saveInitItems;
end;

end;


procedure TVarArray.Delete(ix:Integer);
begin
	DeleteItemsAt(ix,1);
end;

function  TVarArray.SwapItemBuffer : PByte;
begin
	Result := FBlock;
	Dec(Result,FItemSize);
end;

procedure TVarArray.SwapItems(p1, p2 : Pointer);
var
	swap : Pointer;
begin
	swap := SwapItemBuffer;
	System.Move(p1^,swap^,FItemSize);
	System.Move(p2^,p1^,FItemSize);
	System.Move(swap^,p2^,FItemSize);
end;

function TVarArray.CompareItems(p1, p2 : Pointer) : integer;
var
	b1,b2 : PByte;
	i : integer;
begin
	b1 := PByte(p1);
	b2 := PByte(p2);
	for i := 1 to FItemSize do begin
		if (b2^ <> b1^) then begin
			Result := (b1^ - b2^);
			exit;
		end;
	end;
	Result := 0;
end;

procedure TVarArray.SetItem(ix : integer; const buf);
var
	itemPtr:PByte;
begin
	itemPtr := FBlock;
    Inc(itemPtr,ix*FItemSize);
    System.Move(buf,itemPtr^,FItemSize);
end;

procedure TVarArray.GetItem(ix : integer; var buf);
var
	itemPtr:PByte;
begin
	itemPtr := FBlock;
    Inc(itemPtr,ix*FItemSize);
    System.Move(itemPtr^,buf,FItemSize);
end;

procedure TVarArray.Swap(i1,i2:integer);
var
	ip1,ip2:PByte;
begin
	ip1 := FBlock;
    Inc(ip1,i1*FItemSize);
	ip2 := FBlock;
    Inc(ip2,i2*FItemSize);
    SwapItems(ip1,ip2);
end;

procedure TVarArray.DestructItem(itemPtr : pointer);
begin
end;

procedure TVarArray.ConstructItem(itemPtr : pointer);
begin
end;

procedure TVarArray.SetCapacity(incapacity:TVASize);
{$IFNDEF WIN32}
var
	maxCapacity : TVASize;
{$ENDIF}
begin
	if (incapacity < FCount) then {setting Capacity := 0 will always shrink unused capacity}
		incapacity := FCount;

{$IFNDEF WIN32}
	{ 64K block limits for this implementation }
	maxCapacity := (LongInt(kMaxMemBlock) div FItemSize)+1;{reuse delta as maxcapacity}
	if (newCapacity >= maxCapacity) then begin
		if (maxCapacity <= inCount) then
			raise EVarArray.create('Cannot allocate more than '+IntToStr(maxCapacity-1)+' items');
		newCapacity := maxCapacity;
	end;
{$ENDIF}
	if (FBlock <> nil) then
		Dec(PByte(FBlock),FItemSize); {point to spare item buffer at beginning}
{$IFNDEF WIN32}
	ReallocMem16(FBlock,(incapacity+1)*FItemSize);
{$ELSE}
	ReallocMem(FBlock,(incapacity+1)*FItemSize);{capacity plus 1 spare}
{$ENDIF}
	if FBlock <> nil then
		Inc(PByte(FBlock),FItemSize); {point to start of array proper}

	if FBlockHdl <> nil then
		FBlockHdl^ := FBlock;

	FCapacity := incapacity;

end;

procedure TVarArray.SetCount(inCount : TVASize);
var

	delta : TVASize;
	newCapacity : TVASize;
	oldCount : TVASize;
begin
	if (inCount = FCount) then exit;

	if (inCount < 0)  then begin
    	inCount := 0;  {no swap element when count is  0}
		newCapacity := -1;  {force memory deallocation, capacity of 0 not allowed}
    end
	else if (inCount > FCapacity) OR (FAutoShrink) then begin

        { for moderate values of count, the step size is
          approximately 25% of the count value
		}

        if incount <= 16 then
        	delta := 4
        else begin
        	delta := ((incount div 8) AND kAllBitsSet)+1;
            if delta > kMaxStepSize then
            	delta := kMaxStepSize;
		end;
		newCapacity := (( (inCount-1) div delta) + 1) * delta;


	end;

	oldCount := FCount;

	if (FInitItems) and (FCount > 0) and (inCount < FCount) then
      {free items at end of array if decreasing count, prior to reallocation}
		CallDestruct(inCount,oldCount-incount);

	FCount := inCount; {SetCapacity must know what the count is going to be}

	if (newCapacity <> FCapacity) then
		SetCapacity(newCapacity);

	if (FInitItems) and (inCount > 0) and (inCount > oldCount) then
		{construct items at end of array if count increased}
		CallConstruct(oldCount, inCount-oldCount);

end;

constructor TDoubleVArray.Create(inCount : integer);
begin
	inherited CreateBase(sizeof(double),FALSE);
	FBlockHdl := @FPtr;
    Count := inCount;
end;

function TDoubleVArray.GetValue(ix : integer) : double;
var
	p : PDouble;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}

	p := PDouble(FBlock);
	Inc(p,ix);
	Result := p^;
end;

procedure TDoubleVArray.SetValue(ix : integer; invalue:double);
var
	p : PDouble;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}

	p := PDouble(FBlock);
	Inc(p,ix);
	p^:= invalue;
end;


constructor TIntVArray.Create(inCount : integer);
begin
	inherited CreateBase(sizeof(integer),FALSE);
	FBlockHdl := @FIntVABuf;
    Count := inCount;
end;

function TIntVArray.GetValue(ix : integer) : integer;
var
	p : PInteger;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}

	p := PInteger(FBlock);
	Inc(p,ix);
	Result := p^;
end;

procedure TIntVArray.SetValue(ix : integer; invalue:integer);
var
	p : PInteger;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}

	p := PInteger(FBlock);
	Inc(p,ix);
	p^:= invalue;
end;


constructor TIntStrVArray.Create(inCount : integer);
begin
	inherited CreateBase(sizeof(TIntStrRec),TRUE);
    Count := inCount;
end;

procedure TIntStrVArray.DestructItem(itemPtr : pointer);
var
	p : PIntStrRec;
begin
   p := PIntStrRec(itemPtr);
   Finalize(p^.sval);

end;


procedure TIntStrVArray.ConstructItem(itemPtr : pointer);
var
	p : PIntStrRec;
begin
	p := PIntStrRec(itemPtr);
	p^.ival := 0;
	Pointer(p^.sval) := nil;
end;


procedure TIntStrVArray.GetIntStr(ix:integer; var vint:integer; var vs : string);
var
	p:PIntStrRec;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}

    p := PIntStrRec(FBlock);
    Inc(p,ix);
{$IFNDEF WIN32}
    vs := (p^.sval)^;
{$ELSE}
    vs := p^.sval;
{$ENDIF}
    vint := p^.ival;
end;


procedure TIntStrVArray.SetIntStr(ix:integer; vint:integer; const vs:string);
var
	p:PIntStrRec;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}

	p := PIntStrRec(FBlock);
	Inc(p,ix);
{$IFNDEF WIN32}
	AssignStr(p^.sval, vs);
{$ELSE}
	p^.sval := vs;
{$ENDIF}
	p^.ival := vint;
end;

{____________________________________________________}

procedure TStrStrVArray.DestructItem(itemPtr : pointer);
var
	p : PStrStrRec;
begin
   p := PStrStrRec(itemPtr);
   Finalize(p^.nval);
   Finalize(p^.sval);
end;

procedure TStrStrVArray.ConstructItem(itemPtr : pointer);
var
	p : PStrStrRec;
begin
	p := PStrStrRec(itemPtr);
	Pointer(p^.nval) := nil;
	Pointer(p^.sval) := nil;

end;

constructor TStrStrVArray.Create(inCount : integer);
begin
	inherited CreateBase(sizeof(TStrStrRec),TRUE);
    Count := inCount;
end;

procedure TStrStrVArray.GetStrStr(ix:integer; var pname,pvalue:string); 
var
	p : PStrStrRec;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}

	p := PStrStrRec(FBlock);
	Inc(p,ix);
{$IFNDEF WIN32}
	pname := (p^.nval)^;
	pvalue := (p^.sval)^;

{$ELSE}
	pname := p^.nval;
	pvalue := p^.sval;
{$ENDIF}
end;

procedure TStrStrVArray.SetStrStr(ix:integer; const pname,pvalue:string);
var
	p : PStrStrRec;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}

	p := PStrStrRec(FBlock);
	Inc(p,ix);
{$IFNDEF WIN32}
	AssignStr(p^.nval,pname);
	AssignStr(p^.sval,pvalue);

{$ELSE}
	p^.nval :=  pname;
	p^.sval :=  pvalue;
{$ENDIF}
end;



{returns true if ix is an index for insert,
 false if pname was found, and ix is its index
}

function TStrStrVArray.GetVariableIndex(var ix : integer; const pname:string) : boolean;
var
	match : TStrStrRec;
begin
	ConstructItem(@match);
{$IFNDEF WIN32}
	AssignStr(match.nval,pname);
{$ELSE}
	match.nval := pname;
{$ENDIF}
    Result := GetItemIndex(ix,match);
	DestructItem(@match);
end;

{use StrStrArray as sorted variable list }
procedure  TStrStrVArray.SetVariable(const pname, pvalue:string);
var
	ix : integer;
	prec:PStrStrRec;
begin
	if not GetVariableIndex(ix,pname) then begin
		InsertItemsAt(ix,1);
		SetStrStr(ix,pname,pvalue);
	end else begin
		{exists already at ix}
		prec := PStrStrRec(FBlock);
		Inc(prec,ix);
{$IFNDEF WIN32}
	    AssignStr(prec^.sval,pvalue);
{$ELSE}
		prec^.sval := pvalue;

{$ENDIF}
	end;
end;

procedure  TStrStrVArray.AssignValueStrings(slist : TStrings);
begin
	SetCount(0); {clear old}
    AppendValueStrings(slist);
end;

procedure  TStrStrVArray.AppendValueStrings(slist : TStrings);
var
	i : integer;
	name,value : string;
begin
	for i := 0 to slist.Count-1 do begin
{$IFNDEF WIN32}
    	name := GetTStringsName(slist.Strings[i]);
{$ELSE}
    	name := slist.Names[i];

{$ENDIF}
        value := slist.Values[name];
        SetVariable(name,value);
    end;
end;

Function TStrStrVArray.GetIndexValue(ix :Integer):string;
var
	p : PStrStrRec;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}
	p := PStrStrRec(FBlock);
	Inc(p,ix);
{$IFNDEF WIN32}
	Result := (p^.sval)^;
{$ELSE}
	Result := p^.sval;
{$ENDIF}
end;

function TStrStrVArray.CompareItems(p1, p2 : Pointer):integer;
var
	r1,r2 : PStrStrRec;
begin
	r1 := p1;
	r2 := p2;
{$IFNDEF WIN32}
	if FCaseSensitive then
		Result := CompareStr(r1^.nval^,r2^.nval^)
	else
		Result := CompareText(r1^.nval^,r2^.nval^);
{$ELSE}
	if FCaseSensitive then
		Result := CompareStr(r1^.nval,r2^.nval)
	else
		Result := CompareText(r1^.nval,r2^.nval);
{$ENDIF}
end;

Procedure TStrStrVArray.SetIndexValue(ix : integer; const s : string);
var
	p : PStrStrRec;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}
	p := PStrStrRec(FBlock);
	Inc(p,ix);
{$IFNDEF WIN32}
	AssignStr(p^.sval,s);
{$ELSE}
	p^.sval := s;
{$ENDIF}
end;

Procedure TStrStrVArray.SetIndexName(ix : integer; const s : string);
var
	p : PStrStrRec;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}
	p := PStrStrRec(FBlock);
	Inc(p,ix);
{$IFNDEF WIN32}
	AssignStr(p^.nval,s);
{$ELSE}
	p^.nval := s;
{$ENDIF}
end;

Function TStrStrVArray.GetIndexName(ix :Integer):string;
var
	p : PStrStrRec;
begin
{$IFDEF ASSERT_P}
	ASSERT( ((ix>=0)AND(ix<FCount)),kIndexRangeError);
{$ENDIF}
	p := PStrStrRec(FBlock);
	Inc(p,ix);
{$IFNDEF WIN32}
	Result := (p^.nval)^;
{$ELSE}
	Result := p^.nval;
{$ENDIF}
end;

procedure TStrStrVArray.Assign(ssv :TStrStrVArray);
var
	p,q : PStrStrRec;
	ix : integer;
begin
	SetCount(0);
	SetCount(ssv.Count);
	p := PStrStrRec(FBlock);
	q := PStrStrRec(ssv.BlockPtr);
	for ix := 0 to Count-1 do begin
{$IFNDEF WIN32}
	AssignStr(p^.nval,(q^.nval)^);
	AssignStr(p^.sval,(q^.sval)^);

{$ELSE}
		p^.nval := q^.nval;
		p^.sval := q^.sval;
{$ENDIF}
		Inc(p);
		Inc(q);
	end;

end;


Function 	TStrStrVArray.IndexOf(const pname : string) : integer;
begin
	if not GetVariableIndex(Result,pname) then
		Result := -1
end;

procedure   TStrStrVArray.GetVariable(const pname : string; var pvalue:string);
var
	ix : integer;
	prec:PStrStrRec;
begin
	if not GetVariableIndex(ix,pname) then
		raise EStrStrVArray.create('Variable '+pname+' not found');
	prec := PStrStrRec(FBlock);
	Inc(prec,ix);
{$IFNDEF WIN32}
	pvalue := (prec^.sval)^;
{$ELSE}
	pvalue := prec^.sval;
{$ENDIF}

end;

procedure TStrStrVArray.SetValueStrings(slist : TStrings);
var
	i : integer;
    	s : string;
	prec:PStrStrRec;
begin
	slist.clear;
	prec := PStrStrRec(FBlock);
 	for i := 0 to FCount-1 do begin
{$IFNDEF WIN32}
        s := (prec^.nval)^ + '=' + (prec^.sval)^;

{$ELSE}
        s := prec^.nval + '=' + prec^.sval;
{$ENDIF}
        slist.Add(s);
    	Inc(prec);
    end;
end;



end.
