unit UOptions;

interface

uses
  Graphics;

type
  TOptions = class
  private
    FAnimationDelay : Integer;
    FBackgroundColor : TColor;
    FDisplayedInformationColor : TColor;
    FDisplayGeneration : Boolean;
    FDisplayLivingCells : Boolean;
    FGridHeight : Cardinal;
    FGridWidth : Cardinal;
    FLivingCellColor : TColor;
    FModified : Boolean;
    procedure SetAnimationDelay(const Value : Integer);
    procedure SetBackgroundColor(const Value : TColor);
    procedure SetDisplayGeneration(const Value : Boolean);
    procedure SetDisplayedInformationColor(const Value : TColor);
    procedure SetDisplayLivingCells(const Value : Boolean);
    procedure SetGridHeight(const Value : Cardinal);
    procedure SetGridWidth(const Value : Cardinal);
    procedure SetLivingCellColor(const Value : TColor);
  public
    constructor Create;
    procedure LoadFromFile;
    procedure RestoreDefaults;
    procedure SaveToFile;
    property AnimationDelay : Integer read FAnimationDelay write SetAnimationDelay;
    property BackgroundColor : TColor read FBackgroundColor write SetBackgroundColor;
    property DisplayedInformationColor : TColor read FDisplayedInformationColor
      write SetDisplayedInformationColor;
    property DisplayGeneration : Boolean read FDisplayGeneration write SetDisplayGeneration;
    property DisplayLivingCells : Boolean read FDisplayLivingCells write SetDisplayLivingCells;
    property GridHeight : Cardinal read FGridHeight write SetGridHeight;
    property GridWidth : Cardinal read FGridWidth write SetGridWidth;
    property LivingCellColor : TColor read FLivingCellColor write SetLivingCellColor;
    property Modified : Boolean read FModified;
  end;

var
  Options : TOptions;

implementation

uses
  Forms, IniFiles, SysUtils;

const
  DefaultAnimationDelay = 750;
  DefaultBackgroundColor = clBtnFace;
  DefaultDisplayGeneration = True;
  DefaultDisplayLivingCells = True;
  DefaultGridHeight = 10;
  DefaultGridWidth = 10;
  DefaultLivingCellColor = clWindowText;
  DefaultDisplayedInformationColor = DefaultLivingCellColor;

constructor TOptions.Create;

begin
  Inherited Create;
  FAnimationDelay := DefaultAnimationDelay;
  FBackgroundColor := DefaultBackgroundColor;
  FDisplayedInformationColor := DefaultDisplayedInformationColor;
  FDisplayGeneration := DefaultDisplayGeneration;
  FDisplayLivingCells := DefaultDisplayLivingCells;
  FGridHeight := DefaultGridHeight;
  FGridWidth := DefaultGridWidth;
  FLivingCellColor := DefaultLivingCellColor;
  LoadFromFile
end;

procedure TOptions.LoadFromFile;

var
  IniFile : TIniFile;

  function ReadRestrictedInteger(const Section, Ident : String; const Min, Max, Default : Integer) : Integer;

  begin
    Result := IniFile.ReadInteger(Section, Ident, Default);
    if Result < Min then
      Result := Min
    else
      if Result > Max then
        Result := Max
  end;

begin { TOptions.LoadFromFile }
  IniFile := TIniFile.Create(ChangeFileExt(Application.ExeName, '.ini'));
  try
    FGridHeight := ReadRestrictedInteger('Grid Size', 'Down', 10, 100, DefaultGridHeight);
    FGridWidth := ReadRestrictedInteger('Grid Size', 'Across', 10, 100, DefaultGridWidth);
    FBackgroundColor := IniFile.ReadInteger('Color', 'Background', DefaultBackgroundColor);
    FLivingCellColor := IniFile.ReadInteger('Color', 'Living Cells', DefaultLivingCellColor);
    FDisplayGeneration := IniFile.ReadBool('Information', 'Display Generation', DefaultDisplayGeneration);
    FDisplayLivingCells := IniFile.ReadBool('Information', 'Display Living Cells', DefaultDisplayLivingCells);
    FDisplayedInformationColor := IniFile.ReadInteger('Information', 'Color', DefaultDisplayedInformationColor);
    FAnimationDelay := ReadRestrictedInteger('Animation', 'Delay', 0, 30000, DefaultAnimationDelay);
    FModified := False
  finally
    IniFile.Free
  end
end;

procedure TOptions.RestoreDefaults;

begin
  AnimationDelay := DefaultAnimationDelay;
  BackgroundColor := DefaultBackgroundColor;
  DisplayedInformationColor := DefaultDisplayedInformationColor;
  DisplayGeneration := DefaultDisplayGeneration;
  DisplayLivingCells := DefaultDisplayLivingCells;
  GridHeight := DefaultGridHeight;
  GridWidth := DefaultGridWidth;
  LivingCellColor := DefaultLivingCellColor
end;

procedure TOptions.SaveToFile;

var
  IniFile : TIniFile;

begin
  IniFile := TIniFile.Create(ChangeFileExt(Application.ExeName, '.ini'));
  try
    IniFile.WriteInteger('Grid Size', 'Down', GridHeight);
    IniFile.WriteInteger('Grid Size', 'Across', GridWidth);
    IniFile.WriteInteger('Color', 'Background', BackgroundColor);
    IniFile.WriteInteger('Color', 'Living Cells', LivingCellColor);
    IniFile.WriteBool('Information', 'Display Generation', DisplayGeneration);
    IniFile.WriteBool('Information', 'Display Living Cells', DisplayLivingCells);
    IniFile.WriteInteger('Information', 'Color', DisplayedInformationColor);
    IniFile.WriteInteger('Animation', 'Delay', AnimationDelay)
  finally
    IniFile.Free
  end
end;

procedure TOptions.SetAnimationDelay(const Value : Integer);

begin
  if Value <> FAnimationDelay then
  begin
    FAnimationDelay := Value;
    FModified := True
  end
end;

procedure TOptions.SetBackgroundColor(const Value : TColor);

begin
  if Value <> FBackgroundColor then
  begin
    FBackgroundColor := Value;
    FModified := True
  end
end;

procedure TOptions.SetDisplayGeneration(const Value : Boolean);

begin
  if Value <> FDisplayGeneration then
  begin
    FDisplayGeneration := Value;
    FModified := True
  end
end;

procedure TOptions.SetDisplayedInformationColor(const Value : TColor);

begin
  if Value <> FDisplayedInformationColor then
  begin
    FDisplayedInformationColor := Value;
    FModified := True
  end
end;

procedure TOptions.SetDisplayLivingCells(const Value : Boolean);

begin
  if Value <> FDisplayLivingCells then
  begin
    FDisplayLivingCells := Value;
    FModified := True
  end
end;

procedure TOptions.SetGridHeight(const Value : Cardinal);

begin
  if Value <> FGridHeight then
  begin
    FGridHeight := Value;
    FModified := True
  end
end;

procedure TOptions.SetGridWidth(const Value : Cardinal);

begin
  if Value <> FGridWidth then
  begin
    FGridWidth := Value;
    FModified := True
  end
end;

procedure TOptions.SetLivingCellColor(const Value : TColor);

begin
  if Value <> FLivingCellColor then
  begin
    FLivingCellColor := Value;
    FModified := True
  end
end;

end.
