unit UOptions;

interface

uses
  Graphics;

const
  RegistryKey = 'Software\Ian Lane\Life Screen Saver';

type
  TOptions = class
  private
    FAnimationDelay : Integer;
    FBackgroundColor : TColor;
    FDisplayedInformationColor : TColor;
    FDisplayGeneration : Boolean;
    FDisplayLivingCells : Boolean;
    FGridHeight : Cardinal;
    FGridLinesColor : TColor;
    FGridLinesStyle : TPenStyle;
    FGridLinesVisible : Boolean;
    FGridWidth : Cardinal;
    FLivingCellColor : TColor;
    FModified : Boolean;
    FMouseSensitivity : Integer;
    procedure SetAnimationDelay(const Value : Integer);
    procedure SetBackgroundColor(const Value : TColor);
    procedure SetDisplayGeneration(const Value : Boolean);
    procedure SetDisplayedInformationColor(const Value : TColor);
    procedure SetDisplayLivingCells(const Value : Boolean);
    procedure SetGridHeight(const Value : Cardinal);
    procedure SetGridLinesColor(const Value : TColor);
    procedure SetGridLinesStyle(const Value : TPenStyle);
    procedure SetGridLinesVisible(const Value : Boolean);
    procedure SetGridWidth(const Value : Cardinal);
    procedure SetLivingCellColor(const Value : TColor);
    procedure SetMouseSensitivity(const Value : Integer);
  public
    constructor Create;
    procedure LoadFromRegistry;
    procedure RestoreDisplayDefaults;
    procedure SaveToRegistry;
    destructor Destroy; override;
    property AnimationDelay : Integer read FAnimationDelay write SetAnimationDelay;
    property BackgroundColor : TColor read FBackgroundColor write SetBackgroundColor;
    property DisplayedInformationColor : TColor read FDisplayedInformationColor
                                                write SetDisplayedInformationColor;
    property DisplayGeneration : Boolean read FDisplayGeneration write SetDisplayGeneration;
    property DisplayLivingCells : Boolean read FDisplayLivingCells write SetDisplayLivingCells;
    property GridHeight : Cardinal read FGridHeight write SetGridHeight;
    property GridLinesColor : TColor read FGridLinesColor write SetGridLinesColor;
    property GridLinesStyle : TPenStyle read FGridLinesStyle write SetGridLinesStyle;
    property GridLinesVisible : Boolean read FGridLinesVisible write SetGridLinesVisible;
    property GridWidth : Cardinal read FGridWidth write SetGridWidth;
    property LivingCellColor : TColor read FLivingCellColor write SetLivingCellColor;
    property Modified : Boolean read FModified;
    property MouseSensitivity : Integer read FMouseSensitivity write SetMouseSensitivity;
  end;

var
  Options : TOptions;

implementation

uses
  Registry;

const
  DefaultAnimationDelay = 750;
  DefaultBackgroundColor = clBlack;
  DefaultDisplayGeneration = True;
  DefaultDisplayLivingCells = True;
  DefaultGridLinesColor = clWhite;
  DefaultGridLinesStyle = psDot;
  DefaultGridLinesVisible = False;
  DefaultGridHeight = 20;
  DefaultGridWidth = 20;
  DefaultLivingCellColor = clRed;
  DefaultMouseSensitivity = 20;
  DefaultDisplayedInformationColor = DefaultLivingCellColor;

constructor TOptions.Create;

begin
  Inherited Create;
  RestoreDisplayDefaults;
  LoadFromRegistry
end;

destructor TOptions.Destroy;

begin
  if Modified then
    SaveToRegistry;
  Inherited Destroy
end;

procedure TOptions.LoadFromRegistry;

var
  Registry : TRegistry;

  function ReadInteger(const Key : String; const Default : Integer) : Integer;

  begin
    try
      Result := Registry.ReadInteger(Key)
    except
      Result := Default
    end
  end;

  function ReadRestrictedInteger(const Key : String; const Min, Max, Default : Integer) : Integer;

  begin
    Result := ReadInteger(Key, Default);
    if Result < Min then
      Result := Min
    else
      if Result > Max then
        Result := Max
  end;

begin { TOptions.LoadFromRegistry }
  Registry := TRegistry.Create;
  try
    if Registry.OpenKey(RegistryKey + '\Grid Size', False) then
    try
      FGridHeight := ReadRestrictedInteger('Down', 10, 100, DefaultGridHeight);
      FGridWidth := ReadRestrictedInteger('Across', 10, 100, DefaultGridWidth)
    finally
      Registry.CloseKey
    end;
    if Registry.OpenKey(RegistryKey + '\Color', False) then
    try
      FBackgroundColor := ReadInteger('Background', DefaultBackgroundColor);
      FLivingCellColor := ReadInteger('Living Cells', DefaultLivingCellColor)
    finally
      Registry.CloseKey
    end;
    if Registry.OpenKey(RegistryKey + '\Grid Lines', False) then
    try
      FGridLinesColor := ReadInteger('Color', DefaultGridLinesColor);
      FGridLinesStyle := TPenStyle(ReadRestrictedInteger('Style', Ord(Low(TPenStyle)), Ord(psDashDotDot), Ord(DefaultGridLinesStyle)));
      try
        FGridLinesVisible := Registry.ReadBool('Show')
      except
        FGridLinesVisible := DefaultGridLinesVisible
      end
    finally
      Registry.CloseKey
    end;
    if Registry.OpenKey(RegistryKey + '\Information', False) then
    try
      try
        FDisplayGeneration := Registry.ReadBool('Display Generation')
      except
        FDisplayGeneration := DefaultDisplayGeneration
      end;
      try
        FDisplayLivingCells := Registry.ReadBool('Display Living Cells')
      except
        FDisplayLivingCells := DefaultDisplayLivingCells
      end;
      FDisplayedInformationColor := ReadInteger('Color', DefaultDisplayedInformationColor)
    finally
      Registry.CloseKey
    end;
    if Registry.OpenKey(RegistryKey, False) then
    try
      FAnimationDelay := ReadRestrictedInteger('Animation Delay', 0, 30000, DefaultAnimationDelay);
      FMouseSensitivity := ReadRestrictedInteger('Mouse Sensitivity', 0, 100, DefaultMouseSensitivity)
    finally
      Registry.CloseKey
    end;
    FModified := False
  finally
    Registry.Free
  end
end;

procedure TOptions.RestoreDisplayDefaults;

begin
  FAnimationDelay := DefaultAnimationDelay;
  FBackgroundColor := DefaultBackgroundColor;
  FDisplayedInformationColor := DefaultDisplayedInformationColor;
  FDisplayGeneration := DefaultDisplayGeneration;
  FDisplayLivingCells := DefaultDisplayLivingCells;
  FGridHeight := DefaultGridHeight;
  FGridLinesColor := DefaultGridLinesColor;
  FGridLinesStyle := DefaultGridLinesStyle;
  FGridLinesVisible := DefaultGridLinesVisible;
  FGridWidth := DefaultGridWidth;
  FLivingCellColor := DefaultLivingCellColor;
  FMouseSensitivity := DefaultMouseSensitivity
end;

procedure TOptions.SaveToRegistry;

var
  Registry : TRegistry;

begin
  Registry := TRegistry.Create;
  try
    if Registry.OpenKey(RegistryKey + '\Grid Size', True) then
    try
      Registry.WriteInteger('Down', GridHeight);
      Registry.WriteInteger('Across', GridWidth)
    finally
      Registry.CloseKey
    end;
    if Registry.OpenKey(RegistryKey + '\Color', True) then
    try
      Registry.WriteInteger('Background', BackgroundColor);
      Registry.WriteInteger('Living Cells', LivingCellColor)
    finally
      Registry.CloseKey
    end;
    if Registry.OpenKey(RegistryKey + '\Grid Lines', True) then
    try
      Registry.WriteInteger('Color', GridLinesColor);
      Registry.WriteInteger('Style', Ord(GridLinesStyle));
      Registry.WriteBool('Show', GridLinesVisible)
    finally
      Registry.CloseKey
    end;
    if Registry.OpenKey(RegistryKey + '\Information', True) then
    try
      Registry.WriteBool('Display Generation', DisplayGeneration);
      Registry.WriteBool('Display Living Cells', DisplayLivingCells);
      Registry.WriteInteger('Color', DisplayedInformationColor)
    finally
      Registry.CloseKey
    end;
    if Registry.OpenKey(RegistryKey, True) then
    try
      Registry.WriteInteger('Animation Delay', AnimationDelay);
      Registry.WriteInteger('Mouse Sensitivity', MouseSensitivity)
    finally
      Registry.CloseKey
    end
  finally
    Registry.Free
  end
end;

procedure TOptions.SetAnimationDelay(const Value : Integer);

begin
  if Value <> FAnimationDelay then
  begin
    FAnimationDelay := Value;
    FModified := True
  end
end;

procedure TOptions.SetBackgroundColor(const Value : TColor);

begin
  if Value <> FBackgroundColor then
  begin
    FBackgroundColor := Value;
    FModified := True
  end
end;

procedure TOptions.SetDisplayGeneration(const Value : Boolean);

begin
  if Value <> FDisplayGeneration then
  begin
    FDisplayGeneration := Value;
    FModified := True
  end
end;

procedure TOptions.SetDisplayedInformationColor(const Value : TColor);

begin
  if Value <> FDisplayedInformationColor then
  begin
    FDisplayedInformationColor := Value;
    FModified := True
  end
end;

procedure TOptions.SetDisplayLivingCells(const Value : Boolean);

begin
  if Value <> FDisplayLivingCells then
  begin
    FDisplayLivingCells := Value;
    FModified := True
  end
end;

procedure TOptions.SetGridHeight(const Value : Cardinal);

begin
  if Value <> FGridHeight then
  begin
    FGridHeight := Value;
    FModified := True
  end
end;

procedure TOptions.SetGridLinesColor(const Value : TColor);

begin
  if Value <> FGridLinesColor then
  begin
    FGridLinesColor := Value;
    FModified := True
  end
end;

procedure TOptions.SetGridLinesStyle(const Value : TPenStyle);

begin
  if Value <> FGridLinesStyle then
  begin
    FGridLinesStyle := Value;
    FModified := True
  end
end;

procedure TOptions.SetGridLinesVisible(const Value : Boolean);

begin
  if Value <> FGridLinesVisible then
  begin
    FGridLinesVisible := Value;
    FModified := True
  end
end;

procedure TOptions.SetGridWidth(const Value : Cardinal);

begin
  if Value <> FGridWidth then
  begin
    FGridWidth := Value;
    FModified := True
  end
end;

procedure TOptions.SetLivingCellColor(const Value : TColor);

begin
  if Value <> FLivingCellColor then
  begin
    FLivingCellColor := Value;
    FModified := True
  end
end;

procedure TOptions.SetMouseSensitivity(const Value : Integer);

begin
  if Value <> FMouseSensitivity then
  begin
    FMouseSensitivity := Value;
    FModified := True
  end
end;

initialization

Options := TOptions.Create;

finalization

Options.Free

end.
