{*******************************************************************************}
{                                                                               }
{ FileName     : ILfHist2.inc                                                   }
{                                                                               }
{ Author       : Ian Lane (Email: lanei@ideal.net.au)                           }
{                                                                               }
{ Synopsis     : Contains the implementation of the TLifeHistory object.        }
{                                                                               }
{ See also     : ILfHist1.inc, ILfGrid1, ILfGrid2                               }
{                                                                               }
{ Distribution : This object is free for public use and objects may be freely   }
{                descended from it as long as credit is given to the author.    }
{                                                                               }
{                          Copyright (c) 1998 Ian Lane                          }
{                                                                               }
{*******************************************************************************}

{ Add a new grid to the circular buffer }
function TLifeHistory.Add(const Grid : TLifeGrid) : TLifeGrid;

begin
  with FList do
    if Count < Capacity then
      FMostRecent := Add(TLifeGrid.CreateEmpty)
    else
      FMostRecent := (FMostRecent + 1) mod Count;
  Result := FList[FMostRecent];
  Result.Assign(Grid)
end;

{ Remove all the grids from the buffer }
procedure TLifeHistory.Clear;

var
  i : Integer;

begin
  for i := 0 to FList.Count - 1 do
    TLifeGrid(FList[i]).Free;
  FList.Count := 0
end;

{ Searches the circular buffer for a grid matching the contents of the specified grid. The
  search i performed starting with the most recently added grid and proceeding "backwards".
  Returns the index of the matching grid (0 <= Result <= Count - 1) on success and -1 on no
  match found }
function TLifeHistory.Contains(const Grid : TLifeGrid) : Integer;

begin
  Result := 0;
  while (Result < Count) and not Grid.Equal(Grids[Result]) do
    Inc(Result);
  if Result >= Count then
    Result := -1
end;

constructor TLifeHistory.Create(const NumberOfLevels : Cardinal);

begin
  Inherited Create;
  FList := TList.Create;
  FList.Capacity := NumberOfLevels
end;

destructor TLifeHistory.Destroy;

begin
  Clear;
  FList.Free;
  Inherited Destroy;
end;

function TLifeHistory.GetCount : Cardinal;

begin
  Result := FList.Count
end;

{ Return a grid based on its "distance" from the most recently added grid. So, 0 returns the
  most recently added grid; 1 returns the grid added immediately before the most recently added
  grid; and so forth }
function TLifeHistory.GetGrid(const Index : Cardinal) : TLifeGrid;

begin
  Result := FList[(FMostRecent + Count - Index) mod Count]
end;

{ Load all the grids in the circular buffer from the specified stream }
procedure TLifeHistory.LoadFromStream(const Stream : TStream);

var
  i : Integer;
  StreamCount : Integer;

begin
  Clear;
  Stream.ReadBuffer(StreamCount, SizeOf(StreamCount));
  for i := 0 to StreamCount - 1 do
  begin
    FList.Add(TLifeGrid.CreateEmpty);
    TLifeGrid(FList[i]).LoadFromStream(Stream)
  end;
  FMostRecent := Count - 1
end;

function TLifeHistory.GetMaxNumberOfLevels : Cardinal;

begin
  Result := FList.Capacity
end;

{ Save all the grids in the circular buffer to the specified stream }
procedure TLifeHistory.SaveToStream(const Stream : TStream);

var
  i : Integer;
  StreamCount : Integer;

begin
  StreamCount := Count;
  Stream.WriteBuffer(StreamCount, SizeOf(StreamCount));
  for i := 0 to Count - 1 do
    Grids[i].SaveToStream(Stream)
end;

procedure TLifeHistory.SetMaxNumberOfLevels(const Value : Cardinal);

begin
  Clear;
  FList.Capacity := Value
end;

