{*******************************************************************************}
{                                                                               }
{ FileName     : ILfGrid2.inc                                                   }
{                                                                               }
{ Author       : Ian Lane (Email: lanei@ideal.net.au)                           }
{                                                                               }
{ Synopsis     : Contains the implementation of the TLifeGrid object.           }
{                                                                               }
{ See also     : ILfGrid1.inc                                                   }
{                                                                               }
{ Distribution : This object is free for public use and objects may be freely   }
{                descended from it as long as credit is given to the author.    }
{                                                                               }
{                          Copyright (c) 1998 Ian Lane                          }
{                                                                               }
{*******************************************************************************}

{ Allocate memory for FGrid }
procedure TLifeGrid.AllocGrid;

begin
  FGrid := AllocMem(FGridWidth * FGridHeight)
end;

{ Assign the contents of another grid to this grid }
procedure TLifeGrid.Assign(const Source : TLifeGrid);

begin
  SetGridSize(Source.FGridWidth, Source.FGridHeight);
  Move(Source.FGrid^, FGrid^, FGridWidth * FGridHeight)
end;

{ Returns the offset in FGrid (which is only 1-dimensional) of the specified cell
  (which is on a 2-dimensional plane) }
function TLifeGrid.CellCoordToGridOffset(const X : TGridWidthRange;
                                         const Y : TGridHeightRange) : Cardinal;

begin
{$IFOPT R+}
  if X >= FGridWidth then
    raise ERangeError.Create('X parameter out of range');
  if Y >= FGridHeight then
    raise ERangeError.Create('Y parameter out of range');
{$ENDIF}
  Result := Y * FGridWidth + X
end;

{ Set the state of all the cells in FGrid to False (dead) }
procedure TLifeGrid.Clear;

begin
  FillChar(FGrid^, FGridWidth * FGridHeight, False)
end;

constructor TLifeGrid.Create(const GridWidth : TPermissableGridWidth;
                             const GridHeight : TPermissableGridHeight);

begin
  CreateEmpty;
  FGridHeight := GridHeight;
  FGridWidth := GridWidth;
  AllocGrid
end;

constructor TLifeGrid.CreateEmpty;

begin
  Inherited Create;
end;

destructor TLifeGrid.Destroy;

begin
  FreeGrid;
  Inherited Destroy
end;

{ Applies the Rules of Life to determine if the specified cell should live or die }
function TLifeGrid.DoesCellLive(const X : TGridWidthRange;
                                const Y : TGridHeightRange) : Boolean;

begin
{$IFOPT R+}
  if X >= FGridWidth then
    raise ERangeError.Create('X parameter out of range');
  if Y >= FGridHeight then
    raise ERangeError.Create('Y parameter out of range');
{$ENDIF}
  case NumberOfNeighbours(X, Y) of
    2    : Result := Cells[X, Y];
    3    : Result := True;
    else   Result := False
  end
end;

{ Test to see if this grid is equal to Source. }
function TLifeGrid.Equal(const Source : TLifeGrid) : Boolean;

begin
  if (Source.FGridWidth <> FGridWidth) or (Source.FGridHeight <> FGridHeight) then
    Result := False
  else
    Result := CompareMem(FGrid, Source.FGrid, FGridWidth * FGridHeight)
end;

{ Free the memory used by FGrid }
procedure TLifeGrid.FreeGrid;

begin
{$IFDEF VER80}
  FreeMem(FGrid, FGridWidth * FGridHeight)
{$ELSE}
  FreeMem(FGrid)
{$ENDIF}
end;

{ Get the state of the specified cell }
function TLifeGrid.GetCell(const X : TGridWidthRange; const Y : TGridHeightRange) : Boolean;

begin
  Result := FGrid^[CellCoordToGridOffset(X, Y)]
end;

{ Counts the number of "live" cells in FGrid }
function TLifeGrid.GetLiveCellCount : Integer;

var
  i : Integer;

begin
  Result := 0;
  for i := 0 to FGridWidth * FGridHeight - 1 do
    Inc(Result, Ord(FGrid^[i]))
end;

{ Load the state of all the cells from the specified stream. }
procedure TLifeGrid.LoadFromStream(const Stream : TStream);

var
  GridHeight : TPermissableGridHeight;
  GridWidth : TPermissableGridWidth;

begin
  with Stream do
  begin
    ReadBuffer(GridWidth, SizeOf(GridWidth));
    ReadBuffer(GridHeight, SizeOf(GridHeight));
    SetGridSize(GridWidth, GridHeight);
    ReadBuffer(FGrid^, FGridWidth * FGridHeight)
  end
end;

{ Count the number of live neighbours that the specified cell has. Cells on the edge of the
  grid are considered to have the cell on the opposite edge of the grid as a neighbour }
function TLifeGrid.NumberOfNeighbours(const X : TGridWidthRange;
                                      const Y : TGridHeightRange) : Integer;

var
  xMinus1, xPlus1, yMinus1, yPlus1  : Integer;

begin
{$IFOPT R+}
  if X >= GridWidth then
    raise ERangeError.Create('X parameter out of range');
  if Y >= GridHeight then
    raise ERangeError.Create('Y parameter out of range');
{$ENDIF}
  { Pre-calculate some values for the sake of efficiency }
  xMinus1 := (X + GridWidth - 1) mod GridWidth; { Equivalent to (X - 1) with a "wrap" }
  xPlus1 := (X + 1) mod GridWidth;
  yMinus1 := (Y + GridHeight - 1) mod GridHeight; { Equivalent to (Y - 1) with a "wrap" }
  yPlus1 := (Y + 1) mod GridHeight;
  { Count the number of live neighbours that this cell has }
  Result := 0;
  if Cells[xMinus1, yMinus1] then
    Inc(Result);
  if Cells[X, yMinus1] then
    Inc(Result);
  if Cells[xPlus1, yMinus1] then
    Inc(Result);
  if Cells[xMinus1, Y] then
    Inc(Result);
  if Cells[xPlus1, Y] then
    Inc(Result);
  if Cells[xMinus1, yPlus1] then
    Inc(Result);
  if Cells[X, yPlus1] then
    Inc(Result);
  if Cells[xPlus1, yPlus1] then
    Inc(Result)
end;

{ Save the state of all the cells to the specified stream }
procedure TLifeGrid.SaveToStream(const Stream : TStream);

begin
  with Stream do
  begin
    WriteBuffer(FGridWidth, SizeOf(FGridWidth));
    WriteBuffer(FGridHeight, SizeOf(FGridHeight));
    WriteBuffer(FGrid^, FGridWidth * FGridHeight)
  end
end;

{ Set the state of the specified cell }
procedure TLifeGrid.SetCell(const X : TGridWidthRange; const Y : TGridHeightRange;
                            const Value : Boolean);

begin
  FGrid^[CellCoordToGridOffset(X, Y)] := Value
end;

{ Resize FGrid to the specified dimensions. This will have the side-effect of "clearing"
  FGrid }
procedure TLifeGrid.SetGridSize(const NewGridWidth : TPermissableGridWidth;
                                const NewGridHeight : TPermissableGridHeight);

begin
  FreeGrid;
  FGridWidth := NewGridWidth;
  FGridHeight := NewGridHeight;
  AllocGrid
end;

