unit fmMain;

interface

uses
  Buttons, Classes, ComCtrls, Controls, Dialogs, ExtCtrls, Forms, Menus, Messages,
  StdCtrls, ULife;

type
  TfrmMain = class(TForm)
    mnuMain: TMainMenu;
    mniFile: TMenuItem;
    mniFileNew: TMenuItem;
    mniFileOpen: TMenuItem;
    mniFileSave: TMenuItem;
    mniFileSaveAs: TMenuItem;
    mniFileDivider: TMenuItem;
    mniFileExit: TMenuItem;
    mniOptions: TMenuItem;
    mniOptionsOptions: TMenuItem;
    mniHelp: TMenuItem;
    mniHelpAbout: TMenuItem;
    pnlToolBar: TPanel;
    bvlToolBarTopBorder: TBevel;
    btnStep: TSpeedButton;
    btnStart: TSpeedButton;
    bvlToolBarDivider: TBevel;
    btnSetCells: TSpeedButton;
    btnClear: TSpeedButton;
    btnRandomise: TSpeedButton;
    bvlToolBarBotttomBorder: TBevel;
    cmpLife: TLife;
    pnlInformation: TPanel;
    lblGeneration: TLabel;
    lblGenerationValue: TLabel;
    lblLivingCells: TLabel;
    lblLivingCellsValue: TLabel;
    barStatusLine: TStatusBar;
    tmrNextGeneration: TTimer;
    dlgOpen: TOpenDialog;
    dlgSave: TSaveDialog;
    procedure btnRandomiseClick(Sender : TObject);
    procedure btnSetCellsClick(Sender : TObject);
    procedure btnStartClick(Sender : TObject);
    procedure btnStepClick(Sender : TObject);
    procedure cmpLifeChange(Sender : TObject);
    procedure cmpLifeDoesCellLive(const X : TGridWidthRange; const Y : TGridHeightRange;
                                  const Grid : TLifeGrid; var Result : Boolean);
    procedure FormCreate(Sender : TObject);
    procedure Minimised(Sender : TObject);
    procedure mniFileExitClick(Sender : TObject);
    procedure mniFileNewClick(Sender : TObject);
    procedure mniFileOpenClick(Sender : TObject);
    procedure mniFileSaveClick(Sender : TObject);
    procedure mniFileSaveAsClick(Sender : TObject);
    procedure mniHelpAboutClick(Sender : TObject);
    procedure mniOptionsOptionsClick(Sender : TObject);
    procedure Restored(Sender : TObject);
    procedure ShowHint(Sender : TObject);
    procedure tmrNextGenerationTimer(Sender : TObject);
  private
    FFileName : String;
    FIsMinimised : Boolean;
    procedure PatternStabilised(const Periodicity : Cardinal);
    procedure ReadOptions;
    procedure SetFileName(const Value : String);
    procedure UpdateMenusAndButtons;
  protected
    property FileName : String read FFileName write SetFileName;
    procedure WMGetMinMaxInfo(var Message : TWMGetMinMaxInfo); message WM_GETMINMAXINFO;
  end;

var
  frmMain : TfrmMain;

implementation

uses
  fmAbout, fmSettingsDialog, SysUtils, UOptions, URules, Windows;

resourcestring
  PatternStabilisedTitle = 'Pattern Stabilised';
  RepeatingPattern = 'The pattern is repeating itself every %d generations!';
  StaticPattern = 'The pattern is static!';

{$R *.DFM}

procedure TfrmMain.btnRandomiseClick(Sender : TObject);

var
  x : Integer;
  y : Integer;

begin
  with cmpLife do
    for x := 0 to GridWidth - 1 do
      for y := 0 to GridHeight - 1 do
        if Random < 0.5 then
          Cells[x, y] := True
        else
          Cells[x, y] := False
end;

procedure TfrmMain.btnSetCellsClick(Sender : TObject);

begin
  with cmpLife do
  begin
    AcceptMouseClicks := btnSetCells.Down;
    ShowGridLines := AcceptMouseClicks
  end
end;

procedure TfrmMain.btnStartClick(Sender : TObject);

begin
  tmrNextGeneration.Enabled := btnStart.Down;
  UpdateMenusAndButtons
end;

procedure TfrmMain.btnStepClick(Sender : TObject);

var
  Periodicity : Cardinal;

begin
  Periodicity := cmpLife.NextGeneration;
  if Periodicity > 0 then
    PatternStabilised(Periodicity)
end;

procedure TfrmMain.cmpLifeChange(Sender : TObject);

begin
  lblGenerationValue.Caption := ' ' + IntToStr(cmpLife.Generation);
  lblLivingCellsValue.Caption := ' ' + IntToStr(cmpLife.LiveCellCount);
  if FIsMinimised then
    Application.Title := Caption + ' [Generation ' + IntToStr(cmpLife.Generation) + ']';
  UpdateMenusAndButtons
end;

procedure TfrmMain.cmpLifeDoesCellLive(const X : TGridWidthRange; const Y : TGridHeightRange;
                                       const Grid : TLifeGrid; var Result : Boolean);

begin
  if Grid[X, Y] then
    Result := Rules.LiveCells[Grid.NumberOfNeighbours(X, Y)]
  else
    Result := Rules.DeadCells[Grid.NumberOfNeighbours(X, Y)]
end;

procedure TfrmMain.FormCreate(Sender : TObject);

begin
  Application.OnHint := ShowHint;
  Application.OnMinimize := Minimised;
  Application.OnRestore := Restored;
  ReadOptions;
  UpdateMenusAndButtons;
  Randomize
end;

procedure TfrmMain.Minimised(Sender : TObject);

begin
  FIsMinimised := True
end;

procedure TfrmMain.mniFileExitClick(Sender : TObject);

begin
  Close
end;

procedure TfrmMain.mniFileNewClick(Sender : TObject);

begin
  cmpLife.ClearCells;
  if Sender = mniFileNew then
    FileName := ''
end;

procedure TfrmMain.mniFileOpenClick(Sender : TObject);

begin
  dlgOpen.FileName := FileName;
  if dlgOpen.Execute then
  begin
    FileName := dlgOpen.FileName;
    cmpLife.LoadFromFile(FileName);
    Options.GridHeight := cmpLife.GridHeight;
    Options.GridWidth := cmpLife.GridWidth
  end
end;

procedure TfrmMain.mniFileSaveClick(Sender : TObject);

begin
  if Length(FileName) > 0 then
    cmpLife.SaveToFile(FileName)
  else
    mniFileSaveAsClick(Sender)
end;

procedure TfrmMain.mniFileSaveAsClick(Sender : TObject);

begin
  dlgSave.FileName := FileName;
  if dlgSave.Execute then
  begin
    FileName := dlgSave.FileName;
    cmpLife.SaveToFile(FileName)
  end
end;

procedure TfrmMain.mniHelpAboutClick(Sender : TObject);

var
  frmAbout : TfrmAbout;

begin
  frmAbout := TfrmAbout.Create(Self);
  try
    frmAbout.ShowModal
  finally
    frmAbout.Free
  end
end;

procedure TfrmMain.mniOptionsOptionsClick(Sender : TObject);

var
  frmSettingsDialog : TfrmSettingsDialog;

begin
  frmSettingsDialog := TfrmSettingsDialog.Create(Self);
  try
    if frmSettingsDialog.ShowModal = mrOK then
    begin
      ReadOptions;
      if Rules.Modified then
        cmpLife.ResetGeneration
    end
  finally
    frmSettingsDialog.Free
  end
end;

procedure TfrmMain.PatternStabilised(const Periodicity : Cardinal);

var
  msg : String;
  
begin
  if Periodicity = 1 then
    msg := StaticPattern
  else
    msg := Format(RepeatingPattern, [Periodicity]);
  MessageBeep(MB_ICONASTERISK);
  Application.MessageBox(PChar(msg), PChar(PatternStabilisedTitle), MB_ICONINFORMATION or MB_OK)
end;

procedure TfrmMain.ReadOptions;

begin
  cmpLife.GridHeight := Options.GridHeight;
  cmpLife.GridWidth := Options.GridWidth;
  cmpLife.Color := Options.BackgroundColor;
  cmpLife.CellColor := Options.LivingCellColor;
  cmpLife.GridLineColor := cmpLife.CellColor;
  pnlInformation.Visible := Options.DisplayGeneration or Options.DisplayLivingCells;
  if pnlInformation.Visible then
  begin
    pnlInformation.Color := Options.BackgroundColor;
    lblGenerationValue.Visible := Options.DisplayGeneration;
    lblGeneration.Visible := Options.DisplayGeneration;
    if Options.DisplayGeneration then
    begin
      lblGeneration.Font.Color := Options.DisplayedInformationColor;
      lblGenerationValue.Font.Color := Options.DisplayedInformationColor
    end;
    lblLivingCells.Visible := Options.DisplayLivingCells;
    lblLivingCellsValue.Visible := Options.DisplayLivingCells;
    if Options.DisplayLivingCells then
    begin
      lblLivingCells.Font.Color := Options.DisplayedInformationColor;
      lblLivingCellsValue.Font.Color := Options.DisplayedInformationColor
    end
  end;
  tmrNextGeneration.Interval := Options.AnimationDelay
end;

procedure TfrmMain.Restored(Sender : TObject);

begin
  FIsMinimised := False;
  Application.Title := Caption
end;

procedure TfrmMain.SetFileName(const Value : String);

begin
  if Value <> FFileName then
  begin
    FFileName := Value;
    Caption := 'Life - ' + ExtractFileName(FFileName);
    Application.Title := Caption
  end
end;

procedure TfrmMain.ShowHint(Sender : TObject);

begin
  barStatusLine.SimpleText := Application.Hint
end;

procedure TfrmMain.tmrNextGenerationTimer(Sender : TObject);

var
  Periodicity : Cardinal;

begin
  Periodicity := cmpLife.NextGeneration;
  if Periodicity > 0 then
  begin
    tmrNextGeneration.Enabled := False;
    btnStart.Down := False;
    UpdateMenusAndButtons;
    PatternStabilised(Periodicity)
  end
end;

procedure TfrmMain.UpdateMenusAndButtons;

begin
  mniFileNew.Enabled := not btnStart.Down;
  mniFileOpen.Enabled := not btnStart.Down;
  mniFileSave.Enabled := not btnStart.Down;
  mniFileSaveAs.Enabled := not btnStart.Down;
  mniOptions.Enabled := not btnStart.Down and not btnSetCells.Down;
  mniHelp.Enabled := not btnStart.Down and not btnSetCells.Down;
  btnStep.Enabled := not btnStart.Down and not btnSetCells.Down;
  with btnStart do
  begin
    Enabled := not btnSetCells.Down;
    if Down then
    begin
      Caption := 'Stop';
      Hint := 'Stop stepping through the generations'
    end
    else
    begin
      Caption := 'Start';
      Hint := 'Automatically step through the generations'
    end
  end;
  btnSetCells.Enabled := not btnStart.Down;
  btnClear.Enabled := btnSetCells.Down and (cmpLife.LiveCellCount > 0);
  btnRandomise.Enabled := btnSetCells.Down;
  with cmpLife do
    if AcceptMouseClicks then
      Cursor := crHandPoint
    else
      Cursor := crDefault
end;

procedure TfrmMain.WMGetMinMaxInfo(var Message : TWMGetMinMaxInfo);

begin
  Inherited;
  with Message.MinMaxInfo^.ptMinTrackSize do
  begin
    x := 416;
    y := 200
  end
end;

end.
