unit URules;

interface

type
  TRuleArray = array[0..8] of Boolean;
  TRules = class
  private
    FDeadCells : TRuleArray;
    FLiveCells : TRuleArray;
    FModified : Boolean;
    function GetDeadCells(const Index : Integer) : Boolean;
    function GetLiveCells(const Index : Integer) : Boolean;
    procedure SetDeadCells(const Index : Integer; const Value : Boolean);
    procedure SetLiveCells(const Index : Integer; const Value : Boolean);
  public
    constructor Create;
    procedure LoadFromRegistry;
    procedure RestoreDefaults;
    procedure SaveToRegistry;
    property DeadCells[const Index : Integer] : Boolean read GetDeadCells write SetDeadCells;
    property LiveCells[const Index : Integer] : Boolean read GetLiveCells write SetLiveCells;
    property Modified : Boolean read FModified;
  end;

var
  Rules : TRules;

implementation

uses
  Registry, UOptions;

const
  DefaultDeadCells : TRuleArray = (False, False, False, True, False, False, False, False, False);
  DefaultLiveCells : TRuleArray = (False, False, True, True, False, False, False, False, False);

constructor TRules.Create;

begin
  Inherited Create;
  LoadFromRegistry
end;

function TRules.GetDeadCells(const Index : Integer) : Boolean;

begin
  Result := FDeadCells[Index]
end;

function TRules.GetLiveCells(const Index : Integer) : Boolean;

begin
  Result := FLiveCells[Index]
end;

procedure TRules.LoadFromRegistry;

var
  Registry : TRegistry;

begin
  Registry := TRegistry.Create;
  try
    if Registry.OpenKey(RegistryKey + '\Rules', False) then
    try
      try
        Registry.ReadBinaryData('Dead Cells', FDeadCells, SizeOf(FDeadCells));
      except
        FDeadCells := DefaultDeadCells
      end;
      try
        Registry.ReadBinaryData('Live Cells', FLiveCells, SizeOf(FLiveCells))
      except
        FLiveCells := DefaultLiveCells
      end
    finally
      Registry.CloseKey
    end
    else
      RestoreDefaults;
    FModified := False
  finally
    Registry.Free
  end
end;

procedure TRules.RestoreDefaults;

begin
  FDeadCells := DefaultDeadCells;
  FLiveCells := DefaultLiveCells;
  FModified := True
end;

procedure TRules.SaveToRegistry;

var
  Registry : TRegistry;

begin
  Registry := TRegistry.Create;
  try
    if Registry.OpenKey(RegistryKey + '\Rules', True) then
    try
      Registry.WriteBinaryData('Dead Cells', FDeadCells, SizeOf(FDeadCells));
      Registry.WriteBinaryData('Live Cells', FLiveCells, SizeOf(FLiveCells))
    finally
      Registry.CloseKey
    end
  finally
    Registry.Free
  end
end;

procedure TRules.SetDeadCells(const Index : Integer; const Value : Boolean);

begin
  if Value <> FDeadCells[Index] then
  begin
    FDeadCells[Index] := Value;
    FModified := True
  end
end;

procedure TRules.SetLiveCells(const Index : Integer; const Value : Boolean);

begin
  if Value <> FLiveCells[Index] then
  begin
    FLiveCells[Index] := Value;
    FModified := True
  end
end;

initialization

Rules := TRules.Create;

finalization

Rules.Free

end.
