{*******************************************************
          TWave class and TWavwPlayer Component

This unit contains the code for the TWave class and the
TWavePlayer component. TWave class allows you to
include WaveAudio data directly in your components and
therefore in your *.DFM and *.EXE files just like
graphics. TWavePlayer is a simple component using the
TWave class which holds WaveAudio data.

The component implements three published properties,
  Enabled -- if true then able to play audio
  Options -- audio play options
  Wave -- the WaveAudio data

Two public methods
  Play -- play the WaveAudio data
  Abort -- abort the WaveAudio data

Two events,
  BeforePlay -- execute code before playing WaveAudioData
  AfterPlay -- execute code after playing WaveAudioData

                Paul Warren
       HomeGrown Software Development
     (c) 1995 Langley British Columbia.
              (604) 530-9097
       e-mail:  hg_soft@uniserve.com
  Home page: http://haven.uniserve.com/~hg_soft

********************************************************}

unit Waveplay;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, MMSystem;

type
  TWave = class(TPersistent)
  private
    FModified: Boolean;
    FWaveData: TMemoryStream;
    FOnChange: TNotifyEvent;
    function GetEmpty: Boolean;
    procedure SetModified(Value: Boolean);
  protected
    procedure ReadData(Stream: TStream); virtual;
    procedure WriteData(Stream: TStream); virtual;
    procedure Changed(Sender: TObject);
    procedure DefineProperties(Filer: TFiler); override;
  public
    constructor Create; virtual;
    destructor Destroy;
    procedure Assign(Source: TPersistent); override;
    procedure LoadFromFile(const Filename: string); virtual;
    procedure SaveToFile(const Filename: string); virtual;
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToStream(Stream: TStream);
    property Empty: Boolean read GetEmpty;
    property WaveData: TMemoryStream read FWaveData;
    property Modified: Boolean read FModified write SetModified;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

  TWaveClass = class of TWave;

  { TWavePlayer }

  TWavePlayOpt = (wpoSync, wpoAsync, wpoNoDefault, wpoLoop, wpoNoStop);

  TOptionsSet = set of TWavePlayOpt;

  TWavePlayer = class(TComponent)
  private
    { Private declarations }
    FEnabled: boolean;
    FOptions: TOptionsSet;
    FFlags: word;
    FWave: TWave;
    FBeforePlay: TNotifyEvent;
    FAfterPlay: TNotifyEvent;
    procedure SetOptions(NewValue: TOptionsSet);
    procedure SetWave(AWave: TWave);
    procedure SetBeforePlay(Value: TNotifyEvent);
    procedure SetAfterPlay(Value: TNotifyEvent);
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Play;
    procedure Abort;
    property Flags: word read FFlags write FFlags;
  published
    { Published declarations }
    property Enabled: boolean read FEnabled write FEnabled;
    Property Options: TOptionsSet read FOptions write SetOptions default [wpoSync];
    property Wave: TWave read FWave write SetWave;
    property BeforePlay: TNotifyEvent read FBeforePlay write SetBeforePlay;
    property AfterPlay: TNotifyEvent read FAfterPlay write SetAfterPlay;
  end;

procedure Register;

implementation

{ TWave }

{ class instance constructor }
constructor TWave.Create;
begin
  inherited Create;
  { create WaveAudio data buffer }
  FWaveData := TMemoryStream.Create;
end;

{ destroy class instance }
destructor TWave.Destroy;
begin
  FWaveData.Free;
  inherited Destroy;
end;

{ returns false if buffer is nil }
function TWave.GetEmpty;
begin
  Result := FWaveData = nil;
end;

{ method to copy WavwAudio data when required }
procedure TWave.Assign(Source: TPersistent);
begin
  if (Source = nil) or (Source is TWave) then
  begin
    if Source <> nil then
      FWaveData := TWave(Source).FWaveData;
    Changed(Self);
    Exit;
  end;
  inherited Assign(Source);
end;

{ method to load WaveAudio data from a stream - used
  by the library and by the LoadFromFile method }
procedure TWave.LoadFromStream(Stream: TStream);
begin
  FWaveData.SetSize(Stream.Size);
  Stream.ReadBuffer(FWaveData.Memory^, Stream.Size);
  Changed(Self);
end;

{ method to save WaveAudio data to a stream - used
  by the library and by the SaveToFile method }
procedure TWave.SaveToStream(Stream: TStream);
begin
  Stream.WriteBuffer(FWaveData.Memory^, FWaveData.Size);
end;

{ method to indicate data has changed - triggers
  the OnChange method }
procedure TWave.Changed(Sender: TObject);
begin
  FModified := True;
  if Assigned(FOnChange) then FOnChange(Self);
end;

{ method to allow "fake" data to be read and
  written by the library }
procedure TWave.DefineProperties(Filer: TFiler);
begin
  Filer.DefineBinaryProperty('Data', ReadData, WriteData, not Empty);
end;

{ method to set modified flag }
procedure TWave.SetModified(Value: Boolean);
begin
  if Value then
    Changed(Self) else
    FModified := False;
end;

{ method to read data from *.wav file - calls
  LoadFromStream }
procedure TWave.LoadFromFile(const Filename: string);
var
  Stream: TStream;
begin
  Stream := TFileStream.Create(Filename, fmOpenRead);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;

{ method to write data to *.wav file - calls
  SaveToStream }
procedure TWave.SaveToFile(const Filename: string);
var
  Stream: TStream;
begin
  Stream := TFileStream.Create(Filename, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

{ method for library to read data from stream -
  calls LoadFromStream }
procedure TWave.ReadData(Stream: TStream);
begin
  LoadFromStream(Stream);
end;

{ method for library to write data to stream -
  calls SaveToStream }
procedure TWave.WriteData(Stream: TStream);
begin
  SaveToStream(Stream);
end;

{ TWavePlayer }

{ construct TWavePlayer }
constructor TWavePlayer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  { create class instance }
  FEnabled := true;
  FOptions := [wpoSync];
  FFlags := snd_Memory;
  FWave := TWave.Create;
end;

destructor TWavePlayer.Destroy;
begin
  { must make this call or crash and burn }
  sndPlaySound(nil, 0);
  { free class instance }
  FWave.Free;
  inherited Destroy;
end;

{ SetOptions method }
procedure TWavePlayer.SetOptions(NewValue: TOptionsSet);
begin
  if NewValue <> FOptions then
  begin
    FOptions := NewValue;
    { update Flags }
    Flags := snd_Memory; { always set snd_Memory flag }
    if wpoSync in FOptions then
      Flags := Flags or snd_Sync;
    if wpoASync in FOptions then
      Flags := Flags or snd_ASync;
    if wpoNoDefault in FOptions then
      Flags := Flags or snd_NoDefault;
    if wpoLoop in FOptions then
      Flags := Flags or snd_Loop;
    if wpoNoStop in FOptions then
      Flags := Flags or snd_NoStop;
  end;
end;

{ SetBeforePlay method }
procedure TWavePlayer.SetBeforePlay(Value: TNotifyEvent);
begin
  FBeforePlay := Value;
end;

{ SetAfterPlay method }
procedure TWavePlayer.SetAfterPlay(Value: TNotifyEvent);
begin
  FAfterPlay := Value;
end;

{ method to play WaveAudio data }
procedure TWavePlayer.Play;
begin
  if Assigned(FBeforePlay) then FBeforePlay(Self); { BeforePlay event }
  if Enabled then
  try
    Screen.Cursor := crHourGlass;
    sndPlaySound(FWave.FWaveData.Memory, Flags);
  finally
    Screen.Cursor := crDefault;
  end;
  { AfterPlay event -- will execute before play has finished if
    Options.wpoAsync := true }
  if Assigned(FAfterPlay) then FAfterPlay(Self);
end;

{ method to abort }
procedure TWavePlayer.Abort;
begin
  try
    Screen.Cursor := crHourGlass;
    sndPlaySound(nil, 0);
  finally
    Screen.Cursor := crDefault;
  end;
end;

{ method to copy WaveAudio data from property
  editor, or programmatically }
procedure TWavePlayer.SetWave(AWave: TWave);
begin
  { copy data from source }
  FWave.Assign(AWave);
end;

{ register component on 'Misc' page }
procedure Register;
begin
  RegisterComponents('Misc', [TWavePlayer]);
end;

end.
