unit Tsmcode;

{Description:  Memo component with added events, methods and properties}
{Author:       Richard Shotbolt (100327,2305@compuserve.com)}
{Date:         23 Jan 1996}

interface

uses
   SysUtils, WinTypes, WinProcs, Messages, Classes,
   Controls, StdCtrls;

type
   TNotifyEvent = procedure(Sender:TObject) of object;

   TSMemo = class(TMemo)
   private
       FOnVScroll: TNotifyEvent;
       FOnHScroll: TNotifyEvent;
       FCurrentLine: integer;
       FCurrentPosition: integer;
       FTopLine: integer;
       FLinesVisible: integer;
       curline: integer; {store for  current cursor line}
       curposn: integer; {store for current cursor position}
       curtop: integer;  {store for current top line number}
       maxlines: integer;  {store for max number of visible lines}
       procedure DoOnHScroll (var Msg: TWMHScroll); message WM_HSCROLL;
       procedure DoOnVScroll (var Msg: TWMVScroll); message WM_VSCROLL;
       function GetCurrentLine: integer;
       function GetCurrentPosition: integer;
       function GetTopLine: integer;
       procedure SetCurrentLine(Value: integer);
       procedure SetCurrentPosition(Value: integer);
       function GetLinesVisible: integer;
       function GetMaxLinesVisible: integer;
   protected
   public
       constructor Create(AOwner:TComponent); override;
       destructor Destroy; override;
       {properties}
       property CurrentLine: integer read GetCurrentLine write SetCurrentLine;
       property CurrentPosition: integer read GetCurrentPosition write SetCurrentPosition;
       property Topline: integer read GetTopLine;
       procedure GetMaxLines;
       property LinesVisible: integer read GetLinesVisible;
       property MaxLinesVisible: integer read GetMaxLinesVisible;
       {methods}
       procedure ScrollBy(X: integer; Y: integer);
   published
       property OnVScroll: TNotifyEvent read FOnVScroll write FOnVScroll;
       property OnHScroll: TNotifyEvent read FOnHScroll write FOnHScroll;
end;

procedure Register;

implementation

constructor TSMemo.Create(AOwner:TComponent);
begin
inherited Create(AOwner);
end;

procedure TSMemo.DoOnHScroll (var Msg: TWMHScroll);
{New event handler - HScroll}
begin
inherited;
if assigned (FOnHScroll) then
   FOnHScroll(Self);
end;

procedure TSMemo.DoOnVScroll (var Msg: TWMVScroll);
{New event handler - VScroll}
begin
inherited;
if assigned (FOnVScroll) then
   FOnVScroll(Self);
end;

function TSMemo.GetCurrentLine: integer;
{Get line number containing cursor}
begin
curline := SendMessage(Handle, EM_LINEFROMCHAR, SelStart, 0);
result := curline;
end;

function TSMemo.GetCurrentPosition: integer;
{Get character position of cursor within line}
begin
curposn := SelStart - SendMessage(Handle, EM_LINEINDEX,
   (SendMessage(Handle, EM_LINEFROMCHAR, SelStart, 0)), 0);
result := curposn;
end;

function TSMemo.GetTopLine: integer;
{Get number of topmost visible line}
begin
curtop := SendMessage(Handle, EM_GETFIRSTVISIBLELINE, 0, 0);
result := curtop;
end;

procedure TSMemo.GetMaxLines;
{Get number of visible lines in control}
var
   mtemp: TMemo;
   maxpossible: integer;
begin
maxlines := 0;
maxpossible := Height div (-Font.Height);
if maxpossible = 0 then
   {height too small}
   exit;
try
   begin
   {make a temporary TMemo wit some of TSMemo's properties}
   mtemp := TMemo.Create(Self);
   mtemp.Parent := Self;
   mtemp.Left := Left;
   mtemp.Top := Top;
   mtemp.Height := Height;
   mtemp.Width := Width;
   mtemp.ScrollBars := ScrollBars;
   mtemp.Font := Font;
   mtemp.Show;
   mtemp.SendToBack;
   {count lines before TSmemo scrolls}
   while (SendMessage(mtemp.Handle, EM_GETFIRSTVISIBLELINE, 0, 0) = 0) do
      begin
      if (maxlines > maxpossible) then
      {memo too small for font - return 0}
          begin
          maxlines := 0;
          break;
          end;
      mtemp.Lines.Add ('');
      Inc(maxlines);
      end;
   end;
finally
   mtemp.Free;
end;
end;

function TSMemo.GetMaxLinesVisible: integer;
begin
{get number of visible lines}
if Visible = true then
   begin
   if maxlines = 0 then
      {recalculate new value}
      GetMaxLines;
   result := maxlines;
   end
else
   result := 0;
end;

function TSMemo.GetLinesVisible: integer;
var
   c: integer;
   n: integer;
begin
n := GetMaxLinesVisible;
{truncate value to actual numbver of lines visible if necessary}
c := Lines.Count - SendMessage(Handle, EM_GETFIRSTVISIBLELINE, 0, 0);
if c < n then
   result := c
else
   result := n;
end;

procedure TSMemo.SetCurrentLine(Value: integer);
begin
curline := Value;
{Restrict range to available lines}
if curline < 0 then
   curline := 0;
if curline > Lines.Count - 1 then
   curline := Lines.Count - 1;
{Put cursor on start of selected line}
SelLength := 0;
SelStart := SendMessage(Handle, EM_LINEINDEX, curline, 0);
end;

procedure TSMemo.SetCurrentPosition(Value: integer);
begin
{Value must be within range}
curposn := Value;
if curposn < 0 then
   curposn := 0;
if (curposn > Length(Lines[curline])) then
   curposn := Length(Lines[curline]);
{Put cursor in selected position}
SelLength := 0;
SelStart := SendMessage(Handle, EM_LINEINDEX, curline, 0) + curposn;
end;

procedure TSMemo.ScrollBy(X: integer; Y: integer);
{Scroll by X characters and Y lines}
begin
SendMessage(Handle, EM_LINESCROLL, 0, (LongInt(X) shl 16) + Y);
end;

destructor TSMemo.Destroy;
begin
inherited Destroy;
end;

procedure Register;
begin
RegisterComponents('Samples',[TSMemo]);
end;

end.

