{ Projektstyrings komponent til Delphi 1.0x og 2.0x }
{ Copyrights by T S Eriksen, 1996 }
{ Email: TSEriksen@Cyberdude.com }
{ All rights reserved }
{$D-,L-,Y-}
unit Project;

interface

uses
  {$IFDEF WIN32} Windows, {$ENDIF}
  Classes;

type
  {$IFNDEF WIN32}
  DWORD = LongInt; { Define doubleword which is not done in Delphi 16bit }
  {$ENDIF}
  TVFileType= (vftProgram,vftLibrary,vftUnknown,vftDriver,vftFont,vftVXD,vftStaticLib);
  TFileType = (ftProgram,ftLibrary);
  TCharset  = (cs7bit_ASCII,csJapan_JIS_X_0208,csKorea_KSC_5601,csTaiwan_GB5,
               csUnicode,csLatin_2,csCyrillic,csMultilingual,csGreek,csTurkish,
               csHebrew,csArabic);
  TLanguage = (lanArabic,lanBulgarian,lanCatalan,lanTraditional_Chinese,lanCzech,
               lanDanish,lanGerman,lanGreek,lanUS_English,lanCastilian_Spanish,
               lanFinnish,lanFrench,lanHebrew,lanHungarian,lanIcelandic,
               lanItalian,lanJapanese,lanKorean,lanDutch,lanNorwegian_Bokmal,
               lanPolish,lanBrazilian_Portuguese,lanRhaeto_Romanic,lanRomanian,
               lanRussian,lanCroato_Serbian_latin,lanSlovak,lanAlbanian,
               lanSwedish,lanThai,lanTurkish,lanUrdu,lanBahasa,
               lanSimplified_Chinese,lanSwiss_German,lanUK_English,
               lanMexican_Spanish,lanBelgian_French,lanSwiss_Italian,
               lanBelgian_Dutch,lanNorwegian_Nynorsk,lanPortuguese,
               lanSerbo_Croatian,lanCanadian_French,lanSwiss_French,lanIgnorre);
  TStringFile=(sfComments,sfCompanyName,sfFileDescription,sfFileVersion,
               sfInternalName,sfLegalCopyright,sfLegalTrademarks,
               sfOriginalFilename,sfProductName,sfPrivateBuild,sfSpecialBuild,
               sfProductVersion);
  TFileFlag  =(ffDebug,ffPrerelease,ffPatched,ffPrivateBuild,ffInfoInferred,
	         ffSpecialBuild);
  TFileFlags = Set of TFileFlag;

type
  TProject = class(TComponent)
  private
    { user defined properties }
    NoBackUp        : Boolean;
    FResFile        : string; { Resourcefile name }
    FCompany        : string; { stored in Ini-file }
    FCopyright      : string; { stored in Ini-file }
    FComment        : string; { default : <comments about this file/product> }
    FDescription    : string; { default : <Description of file> }
    FSpecialBuild   : string; { default : <How this build differs from a standard build> }
    FPrivateBuild   : string; { default : <Private information> }
    FProductname    : string; { default : <name of product this file ships with> }
    FProductVersion : string; { default : <0.0.0> }
    FOriginalName   : string; { default : <name of .exe file > }
    FTrademarks     : string; { default : <Legal trademarks used in this build> }
    FLanguage       : TLanguage; { default : lanUK_English }
    FCharset        : TCharset;  { default : csMultilinqual }
    FFileType       : TFileType; { default : ffProgram }
    { Auto defined properties }
    { Internalname,           { TProject.name }
    { Fileversion ,           { TProject.version + TProject.build }
    { FileFlags }             { ffprerease when version < 1.0, special/private build }
    { property data buffers etc }
    FSmartBackup : boolean;   { Perform backup checking Archive bit }
    FAutoNaming  : Boolean;   { Styrer automatisk navngivning af project }
    FAutobackup  : boolean;   { manuel backup via bekrft eller automatisk }
    FBackupfiles : string;    { hvilke filer der skal kopieres }
    FBackupPath  : string;    { hvor de skal kopieres til (main-dir) }
    FProjectPath : string;    { hvor de skal kopieres fra }
    FBuild       : Integer;   { build nr, incr. ved component write state}
    FNotes       : TStrings;
    FLog         : TStrings;  { project noter / log   }
    FVersion     : string;    { Version number }
    { Private declarations - property read }
    function  GetVersion    :string;
    { Private declarations - property write }
    procedure SetResFile    (value:string);
    procedure SetLog        (value:TStrings);
    procedure SetNotes      (value:TStrings);
    procedure SetVersion    (value:string);
    procedure SetBackupPath (value:string);
    procedure SetProjectPath(value:string);
    procedure SetBackupFiles(value:string);
    procedure SetCompany    (value:string);
    procedure SetCopyright  (value:string);
  protected
    { Protected declarations }
    procedure   Loaded; override;
    procedure   ReadFromDelphiIni;
    procedure   WriteToDelphiIni;
    procedure   WriteVersionResource;
    procedure   WriteState(Writer:TWriter);    override;
  public
    constructor Create(AComponent:TComponent); override;
    procedure   Free;
    { Public declarations }
  published
    { Published declarations }
    property AutoNaming  : Boolean   read FAutoNaming  write FAutoNaming;
    property Build       : Integer   read FBuild       write FBuild default 0;
    property Log         : TStrings  read FLog         write SetLog;
    property Notes       : TStrings  read FNotes       write SetNotes;
    property Version     : string    read GetVersion   write SetVersion;
    { Backup properties }
    property AutoBackup  : Boolean   read FAutoBackup  write FAutobackup;
    property BackupFiles : string    read FBackupfiles write SetBackupFiles;
    property BackupPath  : string    read FBackUpPath  write SetBackUpPath;
    property ProjectPath : string    read FProjectPath write SetProjectPath;
    property SmartBackup : boolean   read FSmartBackup write FSmartBackup;
    { version resource properties }
    property Company       : string read FCompany        write SetCompany;
    property Copyright     : string read FCopyright      write SetCopyright;
    property Comment       : string read FComment        write FComment;
    property Description   : string read FDescription    write FDescription;
    property SpecialBuild  : string read FSpecialBuild   write FSpecialBuild;
    property PrivateBuild  : string read FPrivateBuild   write FPrivateBuild;
    property Productname   : string read FProductName    write FProductname;
    property ProductVersion: string read FProductVersion write FProductVersion;
    property OriginalName  : string read FOriginalName   write FOriginalName;
    property Trademarks    : string read FTrademarks     write FTrademarks;
    property Language      : TLanguage read FLanguage    write FLanguage;
    property Charset       : TCharset  read FCharset     write FCharset;
    property FileType      : TFileType read FFileType    write FFileType;
    property ResourceFile  : string    read FResFile     write SetResFile;
  end;

  TVersionInfo = class
  private
    { property data buffers }
    FFileType    : TFileType;
    FLanguage    : TLanguage;
    FCharSet     : TCharset;
    { internal data buffers }
    FStringData  : TStrings;
    FFileFlag    : TFileFlags;
  protected
    procedure StrToVer        (Value:string;var MS,LS:DWORD);
    function  DecodeLanguage  (language:TLanguage):word;
    function  DecodeCharset   (Charset:TCharset  ):word;
    function  DecodeStringFile(SFI:TStringFile   ):string;
    procedure DecodeFileFlags(Var Flags,Mask:DWORD);
    procedure BackUpFile      (FileName:string);
    procedure SaveFile        (FileName:string;Buffer:pointer;BufSize:Word);
  public
    constructor Create(AnOwner:TObject);
    procedure Free;
    procedure SaveToFile(FileName:string);
    procedure Clear;
    procedure SetString(SFI:TStringFile;Data:string);
    function  GetString(SFI:TStringFIle):string;
    property  Charset  : TCharSet   read FCharset  write FCharset ;
    property  Language : TLanguage  read FLanguage write FLanguage;
    property  FileType : TFileType  read FFileType write FFileType;
    property  FileFlags: TFileFlags read FFileFlag write FFileFlag;
  end;

type
  TBackup = class
  private
    FAnyError : boolean;
    FSource,
    FTarget,
    FPattern : string;
    FArchive,
    FSubDirs : boolean;
    FAttribs : Integer;
    procedure SetPattern(value:string);
    procedure SetTarget(value:string);
    procedure SetSource(value:string);
  protected
    function  FileCopy(Input,output:string;Size:LongInt):boolean; { copy single file }
    function  GetTargetPath(SourcePath:string):string;
    procedure CopyMatchingFiles(Path:string);
    procedure ScanFolders(Path:string);
    procedure CopyFiles(Path,Pattern:string);
    procedure CopyArchiveFiles(Path,Pattern:string);
  public
    { Init/Exit procedures }
    constructor Create;
    procedure Free;
    { Toolbox procedures }
    function  QualifyPath(path:string):string;
    procedure ForceDirectories(Dir: string);
    { Execution procedure }
    function  Execute:boolean;
    { public properties }
    property  Source  : string  read FSource  write SetSource;
    property  Target  : string  read FTarget  write SetTarget;
    property  Pattern : string  read FPattern write SetPattern;
    property  Archive : boolean read FArchive write FArchive;
  end;


{$IFDEF WIN32}
type
  TUserReg = class(TObject)
  private
    { Private declarations }
    KeyOpen : Boolean;
    KeyHandle: HKEY;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create;
    procedure Free;
    procedure Open(key:string);
    procedure Close;
    procedure WriteString (Name,Value:string);
    procedure WriteInteger(Name:string;Value:integer);
    function  ReadString  (Name,Default:string):string;
    function  ReadInteger (Name:string;Default:integer):integer;
  end;
{$ENDIF}

procedure Register;

implementation

Uses
  {$IFNDEF WIN32} IniFiles, WinProcs, WinTypes, {$ENDIF}
  Forms, SysUtils, FileCtrl, Dialogs;

{ Global constants }
const
  {$IFDEF WIN32 }
  DelphiProject = '*.dfm;*.pas;*.dpr;*.dof;*.dsk;*.res';
  {$ELSE}
  DelphiProject = '*.dfm;*.pas;*.dpr;*.dsk;*.dsm;*.opt;*.res';
  {$ENDIF}

  Warning = 'Please note: The backup-feature has been changed.'#13+
            'Backups will now include subfolders under ProjectPath'#13+
            'and the BackupPath is now the backup-root.'#13#13+
            'Please see the helpfile for more info.';

{ private procedures }
function QualifyPath(path:string):string;
{ Expands path and adds driveletters etc. then }
{ adds a trailing backslash to a string if needed }
begin
  Path:=ExpandFileName(Path);
  If Path[length(Path)]<>'\' then Path:=Path+'\';
  QualifyPath:=Path;
end;


function CreatePath(Path:string):string;
{ Create path if it dosn't exist }
var
  pTemp : Array[0..99] of char;
begin
  Path:=ExpandFileName(Path);
  CreatePath:=Path;
  If not DirectoryExists(Path) then begin
    StrPCopy(pTemp,'Path "'+LowerCase(Path)+'" does not exist.'+#13+'Create it?');
    Case application.Messagebox(pTemp,'Create path',
         Mb_YesNo or Mb_IconQuestion) of
      Id_Yes : begin
        {$I-}                        { Sl RunTime error check fra }
        ForceDirectories(Path);
        {$I+}                        { Sl RunTime error check til }
        If IOResult<>0 then begin
          GetDir(0,Path);
          CreatePath:=Path;
          raise exception.create('Could not create path');
        end;
      end; { Id_Yes }
      Id_No : Begin
        GetDir(0,Path);
        CreatePath:=Path;
      end;
    end;
  end; { Dir Exists }
end;

{ protected procedures }

{$IFDEF WIN32}

procedure TProject.ReadFromDelphiIni;
var
  Reg : TUserReg;
  D,M,Y : Word;
begin
  DecodeDate(Date,Y,M,D);
  Reg:=TUserReg.Create;
  With Reg do try
    Open('SOFTWARE\BORLAND\DELPHI\2.0\Project Version');
    FCompany    :=ReadString('Company','<Name of your company>');
    FCopyright  :=ReadString('Copyrights','Copyrights  T S Eriksen, '+IntToStr(Y));
    FLanguage   :=TLanguage(ReadInteger('Language',Word(lanUs_English)));
    FCharset    :=TCharSet (ReadInteger('Charset',Word(csMultilingual)));
    FBackupPath :=ReadString('BackupPath',GetCurrentDir);
    FSmartBackup:=Boolean(ReadInteger('SmartBackup',0));
    If (ReadInteger('Warning',0)=0) then begin
      application.messagebox(Warning,'Warning',Mb_Ok or Mb_IconInformation);
      WriteInteger('Warning',1);
      FAutoBackup:=FALSE;
    end;
    close;
  finally
    Reg.Free;
  end;
end;

procedure TProject.WriteToDelphiIni;
var
  Reg : TUserReg;
  D,M,Y : Word;
begin
  DecodeDate(Date,Y,M,D);
  Reg:=TUserReg.Create;
  With Reg do try
    Open('SOFTWARE\BORLAND\DELPHI\2.0\Project Version');
    WriteString  ('BackupPath',FBackupPath);
    WriteString  ('Company',FCompany);
    WriteString  ('Copyrights',FCopyright);
    WriteInteger ('Language',Word(FLanguage));
    WriteInteger ('CharSet',Word(FCharset));
    WriteInteger ('SmartBackup',Word(FSmartBackup));
    close;
  finally
    Reg.Free;
  end;
end;

{$ELSE}

procedure TProject.ReadFromDelphiIni;
const
  PV = 'Project Version';
var
  Ini : TIniFile;
  D,M,Y : Word;
begin
  Try
    DecodeDate(Date,Y,M,D);
    Ini:=TIniFile.Create('DELPHI.INI');
    With Ini do begin
      FCompany    :=ReadString(PV,'Company','<Name of your company>');
      FCopyright  :=ReadString(PV,'Copyrights','Copyrights  T S Eriksen, '+IntToStr(Y));
      FLanguage   :=TLanguage(ReadInteger(PV,'Language',Word(lanUs_English)));
      FCharSet    :=TCharSet (ReadInteger(PV,'CharSet', Word(csMultilingual)));
      FSmartBackup:=Boolean(ReadInteger(PV,'SmartBackup',0));
      {$IFDEF WIN32}
      FBackupPath :=ReadString(PV,'BackupPath',GetCurrentDir);
      {$ELSE}
      GetDir(0,FBackupPath);
      FBackupPath :=ReadString(PV,'BackupPath',FBackupPath);
      {$ENDIF}
      If ReadInteger(PV,'Warning',0)=0 then begin
        application.messagebox(Warning,'Warning',Mb_Ok or Mb_IconInformation);
        WriteInteger(PV,'Warning',1);
        FAutoBackup:=FALSE;
      end;
    end;
  Finally
    Ini.Free;
  end;
end;

procedure TProject.WriteToDelphiIni;
const
  PV = 'Project Version';
var
  Ini : TIniFile;
begin
  Try
    Ini:=TIniFile.Create('DELPHI.INI');
    With Ini do begin
      WriteString (PV,'BackupPath',FBackupPath);
      WriteString (PV,'Company',FCompany);
      WriteString (PV,'Copyrights',FCopyright);
      WriteInteger(PV,'Language',Word(FLanguage));
      WriteInteger(PV,'CharSet', Word(FCharset));
      WriteInteger(PV,'SmartBackup',Word(FSmartBackup));
    end;
  Finally
    Ini.Free;
  end;
end;
{$ENDIF}

procedure TProject.WriteState(Writer:TWriter);
begin
  { When something on a form has changed all components (on it) will
    be saved when next compiled - before that save, we'll just
    increase the Build number }
  if csDesigning in ComponentState then begin
    Inc(FBuild);                        { increase build number }
    NoBackUp:=FALSE;
    WriteToDelphiIni;                   { store generel settings in Ini file}
    WriteVersionResource;               { write version resource }
  end;
  Inherited WriteState(Writer);
end;

procedure TProject.Loaded;
var
  s : string;
  a : integer;
begin
  inherited Loaded;
  If (csDesigning in ComponentState) then begin
    { Denne blok udfres kun ved
      1) Load af projekt
      2) Visning af form/modul hvor TProject er lagret  }
    Inc(FBuild);                     { increase build number }
  end else begin
    { Denne blok udfrers kun ved
      1) Start af applikation }
    If (AutoNaming) then begin
      application.Title:=Name+' ver. '+Version+'.'+
                         IntToStr(Build);
      s:=application.title;
      For a:=1 to length(s) do if s[a]='_' then s[a]:=' ';
      application.title:=s;
    end;
  end;
end;

procedure TProject.WriteVersionResource;
function UseStr(value:string):boolean;
begin
  UseStr:=((Value>'') and (Value[1]<>#60));
end;
var
  Dc          : Char;
  Ok          : Boolean;
  tmp         : array[0..299] of char;
  s           : string;
  VersionInfo : TVersionInfo;
begin
  VersionInfo:=TVersionInfo.Create(SELF);
  VersionInfo.FileFlags:=[];	    { Siden TProject benytter samme interne var navne }
  VersionInfo.Language:=FLanguage ; { som TVersionInfo, skal 4 linier st uden for et }
  VersionInfo.Charset :=FCharset  ; { WITH statement }
  VersionInfo.FileType:=FFileType ;
  With VersionInfo do try
    If UseStr(FComment) then SetString(sfComments,FComment);
    If UseStr(FCompany) then SetString(sfCompanyName,FCompany);
    If UseStr(FCopyright) then SetString(sfLegalCopyright,FCopyright);
    If UseStr(FDescription) then SetString(sfFileDescription,FDescription);
    If UseStr(FProductname) then SetString(sfProductname,FProductname);
    If UseStr(FProductversion) then SetString(sfProductversion,FProductversion);
    If UseStr(FOriginalname) then SetString(sfOriginalFilename,FOriginalname);
    If UseStr(FTrademarks) then SetString(sfLegalTrademarks,FTrademarks);
    If UseStr(FSpecialbuild) then begin
      SetString(sfSpecialBuild,FSpecialBuild);
      FileFlags:=FileFlags+[ffSpecialBuild];
    end;
    If UseStr(FPrivatebuild) then begin
      SetString(sfPrivateBuild,FPrivateBuild);
      FileFlags:=FileFlags+[ffPrivateBuild];
    end;
    s:=name;
    While Pos('_',s)>0 do s[Pos('_',s)]:=' ';
    SetString(sfInternalname,s);
    SetString(sfFileVersion,FVersion+'.'+IntToStr(Build));
    DC:=DecimalSeparator;
    DecimalSeparator:='.';
    Try
      try
        { if major version is less than 1 - it's a pre-release }
        If StrToFloat(FVersion)<1 then
          FileFlags:=FileFlags+[ffPrerelease];
      except
        { if we can't resolve the version - it's a pre-release as well }
        FileFlags:=FileFlags+[ffPrerelease];
      end;
    finally
      DecimalSeparator:=DC;
    end;
    OK:=TRUE;
    Try
      SaveToFile(Qualifypath(ProjectPath)+ResourceFile);
    except
      Ok:=FALSE;
      FLog.add(DateTimeToStr(Now)+': Error occured writing versionresource'+
                ' - build '+IntToStr(FBuild)+'.');
      raise exception.create('Error occured writing versionresource');
    end;
    If ok then
      FLog.add(DateTimeToStr(Now)+': Versionresource written'+
                 ' - build '+IntToStr(FBuild)+'.');
  finally
    VersionInfo.Free;
  end;
end;

{ property functions (read) }

function  TProject.GetVersion:string;
begin
  GetVersion:=FVersion;
end;

{ property procedures (write) }

procedure  TProject.SetBackupFiles(value:string);
begin
  if csDesigning in ComponentState then begin
    If Value='' then value:='*.*';
    FBackupFiles:=LowerCase(Value);
  end;
end;

procedure  TProject.SetBackupPath(value:string);
begin
  if csDesigning in ComponentState then
    FBackupPath:=LowerCase(CreatePath(Value));
end;

procedure  TProject.SetProjectPath(value:string);
begin
  if csDesigning in ComponentState then
    FProjectPath:=LowerCase(CreatePath(Value));
end;

procedure  TProject.SetVersion(Value:string);

  function MakeFolderName(Value:string):string;
  { Create name of backup directory from the version number }
  var
    s : string;
  begin
    s:=QualifyPath(FProjectPath);
    system.delete(s,length(s),1);
    MakeFolderName:=ExtractFileName(s)+'\Ver'+Value+'\';
  end;

  function QualifyVerValue(Value:string):string;
  var
    a : integer;
  begin
    a:=1;
    While a<=length(value) do begin
      if not (value[a] in ['0'..'9','.']) then begin
        system.delete(value,a,1);
        dec(a);
      end;
      inc(a);
    end;
    if value='' then value:='0.01';
    result:=value;
  end;

  function CreateBackup:boolean;
  var
    Backup : TBackup;
    s      : string;
  begin
    Backup:=TBackup.Create;
    With Backup do try
      { Create backup path }
      s:=QualifyPath(FBackupPath)+MakeFolderName(FVersion);
      ForceDirectories(s);
      Target :=s;
      Source :=QualifyPath(FProjectPath);
      Pattern:=FBackupfiles;
      Archive:=FSmartBackup;
      If Execute then begin
        Result:=TRUE;
        case Archive of
          false : Log.add(DateTimeToStr(Now)+': Backup successfull');
          true  : Log.add(DateTimeToStr(Now)+': Changed files backed up successfully');
        end;
      end else begin
        Result:=FALSE;
        Log.add(DateTimeToStr(Now)+': Error occured in backup');
        raise exception.create('WARNING: Error occured in backup');
      end;
    finally
      Free;
    end;
  end;

Var
  pTemp : Array[0..99] of char;
begin
  Value:=QualifyVerValue(Value);
  If (csDesigning in componentstate) and (not noBackUp) then begin
    If value<>Fversion then case FAutoBackup of
      FALSE : begin
        StrPCopy(pTemp,'Create backup of version '+GetVersion+'?');
        Case application.Messagebox(pTemp,'Version number changed',
                         Mb_YesNoCancel or Mb_IconQuestion) of
          Id_No : Begin
            Build:=0;
          end;
          Id_Yes : try
            CreateBackup;
          finally
            Build:=0;
          end;
          else Value:=FVersion; { Cancel pressed }
        end; { Case MsgBox  }
      end;
      TRUE : try
        CreateBackup;
      finally
        Build:=0;
      end;
    end; { Case Autobackup          }
  end; { designing - any backup? }
  FVersion:=Value;
  If not NoBackup then
    FLog.add(DateTimeToStr(Now)+': Version '+GetVersion);
  NoBackup:=FALSE; { nu m backups gerne ske }
end;

procedure  TProject.SetNotes(Value:TStrings);
begin
  FNotes.Assign(Value);
end;

procedure  TProject.SetLog(Value:TStrings);
begin
  FLog.Assign(Value);
end;

procedure TProject.SetCompany(Value:string);
begin
  FCompany:=Value;
end;

procedure TProject.SetCopyright(Value:string);
begin
  FCopyright:=Value;
end;

procedure TProject.SetResFile(value:string);
begin
  FResFile:=ChangeFileExt(Value,'.res');
end;

{ Init & Exit }

procedure TProject.Free;
begin
  FLog.Free;
  FNotes.Free;
  Inherited Free;
end;

constructor TProject.Create(AComponent:TComponent);
begin
  Inherited Create(AComponent);
  FNotes:=TStringList.Create;
  FLog  :=TStringList.Create;
  { Only do this if designing - if runtime, it's not nesseary }
  If (csDesigning in ComponentState) then begin
    FLog.add(DateTimeToStr(Now)+': Project started.');
    FBackupfiles:=DelphiProject;
    FVersion:='0.01';
    GetDir(0,FProjectPath);
    FProjectPath:=LowerCase(FProjectPath);
      ReadFromDelphiIni;
    FComment       :='<Additional diagnostic info>';
    FDescription   :='<Description of file>';
    FSpecialBuild  :='<How this build differs from a standard build>';
    FPrivateBuild  :='<Private information>';
    FProductname   :='<name of product this file ships with>';
    FProductVersion:='<0.0.0.0>';
    FOriginalName  :='<name of .exe file>';
    FTrademarks    :='<Legal trademarks used in this build>';
    FFileType      :=ftProgram;
    NoBackup       :=TRUE;
    FResFile       :='Version.res';
  end;
end;

{ **************************************************************************** }

{ Globale typer og konstanter der dkker bde 16 og 32 bits ressourcefiler }

type
  Vs_FixedFileInfo = record
    dwSignature        : DWORD;   { $FEEFO4BD }
    dwStrucVersion     : DWORD;   { $00010000  (Version structure) }
    dwFileVersionMS    : DWORD;   { ????,???? }
    dwFileVersionLS    : DWORD;   { ????,???? }
    dwProductVersionMS : DWORD;   { ????,???? }
    dwProductVersionLS : DWORD;   { ????,???? }
    dwFileFlagsMask    : DWORD;   { 0000,0000 }
    dwFileFlags        : DWORD;   { 0000,0000 }
    dwFileOS           : DWORD;   { VOS_NT_WINDOWS32 (4), VOS_WINDOWS16 (1) }
    dwFileType         : DWORD;   { VFT_APP, VFT_DLL }
    dwFileSubtype      : DWORD;   { must be zero for App or Dll  }
    dwFileDateMS       : DWORD;   { DateTime for File - allways set to zero }
    dwFileDateLS       : DWORD;   { when using Microsoft or Borland RC      }
  end;

{$IFDEF WIN32} { Dette er VersionInfo 32 bit }

{ Type erklring for VS_VERSION_INFO er opdelt i 3 dele, 1 og 3 er ikke variabel,
  mens del 2 er! Del 2 opbrydes i en rkke mindre ikke-variable dele
  som logikken i objektet skal kunne skifte mellem. }

  type
  FileHeader = record
    UnKnown : Array[0..15] of word;
      { Header ndres ikke for resourceantal/type etc - Muligvis blot
        bitmnster der viser at det er en 32bits resourcefil ! Vrdier:
        0000 0000 2000 0000 FFFF 0000 FFFF 0000 0000 0000
        0000 0000 0000 0000 0000 0000 0000 0000 0000 0000 }
    Resource : record { Beskrivelse af resource i fil }
      SizeOfRes    : DWORD; { same as wLength }
      Space1       : DWORD; { always 00002000 }
      ResTypeStart : DWORD; { always 1000FFFF - VERSIONINFO }
      IdentifierMS : DWORD; { always 0100FFFF stter navn til et numerisk ettal }
      IdentifierLS : DWORD; { always 00000000 }
      ResTypeStop  : DWORD; { always 00003000 - VERSIONINFO }
      Padding1     : DWORD; { always 00000000 }
      Padding2     : DWORD; { always 00000000 }
      { her starter 32 bits resource VS_VERSION_INFO_1 }
    end;
  end;

  VS_VERSION_INFO_1 = record
    wLength      : word;        { size of version resource }
    wValueLength : word;        { size of VS_FIXEDFILEINFO }
    bText        : word;        { always 0 - binary resource }
    szKey   : array[0..15] of WideChar;   { "VS_VERSION_INFO" }
    padding : array[0.. 1] of byte;       { 32bit boundary padding }
    Value   : VS_FIXEDFILEINFO;
    Child   : record
      wLength      : word;     { size of block in bytes }
      wValueLength : word;     { always 0 }
      bText        : word;     { always 0 - binary value }
      szKey   : array[0..14] of WideChar; { "StringFileInfo" }
      Value   : record
        wLength      : word; { size of block in bytes }
        wValueLength : word; { size of blocks from VS_VERSION_INFO_2 }
        bText        : word; { always 0 - binary value }
        szKey  : array[0..7] of WideChar; { Eks. "040904E4" }
      end;  { value }
    end; { Child  }
  end; { VS_VERSION_INFO }

  VS_VERSION_INFO_2 = array[sfComments..sfProductVersion] of record
    CopySize : Word;     { Number of bytes to copy }
    case byte of
      0 : ( wLength     : word;   { number of bytes in block }
            wValueLength: word;   { number of chars in value }
            bText       : word;   { always 1 - text value }
            VarChar     : Array[3..299] of WideChar  ); { contains szKey, padding & Value }
      1 : ( ByteArray   : Array[0..599] of byte); { superimposed array to copy from }
  end;

  VS_VERSION_INFO_3 = record
    wLength     : word;   { number of bytes in block }
    wValueLength: word;   { always 0 }
    bText       : word;   { always 0 - binary value }
    szKey   : array[0..12] of WideChar;   { "VarFileInfo" }
    Child   : record
      wLength     : word; { number of bytes in block = $24 }
      wValueLength: word; { size of Langauge & Charset = 4 }
      bText       : word; { always 0 - binary value }
      szKey    : array[0..12] of WideChar; { "Translation" }
      Language : Word;
      CharSet  : Word;
    end; { child }
  end; { VS_VERSION_INFO_3 }

procedure TVersionInfo.SaveToFile(FileName:string);
type
  TBuffer= Array[0..SizeOf(VS_VERSION_INFO_1)+
                    SizeOf(VS_VERSION_INFO_2)+
                    SizeOf(VS_VERSION_INFO_3)] of Byte;
  pBuffer= ^TBuffer;
var
  VVI1   : VS_VERSION_INFO_1;
  VVI2   : VS_VERSION_INFO_2;
  VVI3   : VS_VERSION_INFO_3;
  RFH    : FileHeader;
  Buffer : pBuffer;
  a,b,T  : integer;
begin
  { 32bit VS_VERSION_INFO_1 }
  Filename:=ChangeFileExt(Filename,'.res');
  With VVI1 do begin
    wLength      := SizeOf(VS_VERSION_INFO_1)+
                    SizeOf(VS_VERSION_INFO_3);
    wValueLength := SizeOf(VS_FIXEDFILEINFO);
    bText        := 0;
    StringToWideChar('VS_VERSION_INFO',szKey,16);
    for a:=0 to 1 do Padding[a]:=0;
    With Value do begin
      dwSignature     :=$FEEF04BD;
      dwStrucVersion  :=$00010000; { VersionInfo }
      DecodeFileFlags(dwFileFlags,dwFileFlagsMask);
      dwFileSubtype   :=$00000000;
      dwFileOS        :=$00040004; { VOS_NT_Windows32 }
      dwFileDateMS    :=$00000000;
      dwFileDateLS    :=$00000000;
    { Default variable values }
      Case FFileType of
        ftProgram : dwFileType:=1; { VFT_APP }
        ftLibrary : dwFileType:=2; { VFT_DLL }
      end;
      StrToVer(FStringData.Strings[Byte(sfFileVersion)],
               dwFileVersionMS,dwFileVersionLS);
      StrToVer(FStringData.Strings[Byte(sfProductVersion)],
               dwProductVersionMS,dwProductVersionLS);
    end;
    With Child do begin { VS_VERSION_INFO.CHILD }
      wLength      :=SizeOf(Child)+2;
      wValueLength :=0;
      bText        :=0;
      StringToWideChar('StringFileInfo',szKey,16);
      With Value do begin
        wLength      := SizeOf(Value)+2;
        wValueLength := 0;
        bText        := 0;
        StringToWideChar(IntToHex(DecodeLanguage(FLanguage),4)+
                         IntToHex(DecodeCharset(FCharset),4),szKey,9);
      end;  { value }
    end; { Child  }

  end;
  { 32bit VS_VERSION_INFO 2}
  For a:=Byte(sfComments) to Byte(sfProductVersion) do
   with VVI2[TStringFile(a)] do
   if FStringData.Strings[a]>'' then begin
    b:=3; { der er 3 words fr denne parameter }
    t:=0;
    StringToWideChar(DecodeStringFile(TStringFile(a)),@VarChar[b],255);
    b:=b+Length(DecodeStringFile(TStringFile(a)));
    Repeat
      VarChar[b]:=#0;
      Inc(b);
    until (((b*2)*8) mod 32 = 0);
    StringToWideChar(FStringData.Strings[a],@VarChar[b],255);
    wValueLength:=Length(FStringData.Strings[a])+1;
    b:=b+wValueLength;
    Repeat
      VarChar[b]:=#0;
      Inc(b);
    until (((b*2)*8) mod 32 = 0);
    wLength :=b*2;
    bText   :=1;
    CopySize:=wLength;
    { Calculate nested block sizes (3 levels) }
    VVI1.wLength:=VVI1.wLength+CopySize;
    VVI1.Child.wLength:=VVI1.Child.wLength+CopySize;
    VVI1.Child.Value.wLength:=VVI1.Child.Value.wLength+CopySize;
  end else begin
    CopySize:=0;
  end;
  { 32bit VS_VERSION_INFO 3}
  With VVI3 do begin
    { Denne blok er frdig .. }
    wLength     := SizeOf(VS_VERSION_INFO_3);
    wValueLength:= 0;
    bText       := 0;
    StringToWideChar('VarFileInfo',szKey,12);
    With Child do begin
      wLength     := 36;
      wValueLength:= 4;         { size of Langauge & Charset = 4 }
      bText       := 0;
      StringToWideChar('Translation',szKey,12);
      Language:=DecodeLanguage(FLanguage);
      Charset :=DecodeCharset (FCharset );
    end; { child }
  end; { VS_VERSION_INFO_3 }

  { 32bit Resource file header }
  With RFH do begin
    For a:=0 to 15 do UnKnown[a]:=$0000;
    Unknown[2]:=$0020;
    Unknown[4]:=$FFFF;
    Unknown[6]:=$FFFF;
    With resource do begin
      SizeOfRes    := VVI1.wLength;
      Space1       := $00000020;
      ResTypeStart := $0010FFFF; { VERSIONINFO - neutral major }
      IdentifierMS := $0001FFFF; { Res 1 - neutral minor }
      IdentifierLS := $00000000;
      ResTypeStop  := $00000030; { VERSIONINFO }
      Padding1     := $00000000;
      Padding2     := $00000000;
    end;
  end;

  { create temporary buffer }
  New(Buffer);
  try
    { move resourcefile header and first part of the versionresource to
      temp buffer }
    Move(RFH,Buffer^[0],SizeOf(RFH));   b:=SizeOf(RFH);
    Move(VVI1,Buffer^[b],SizeOf(VVI1)); b:=b+SizeOf(VVI1);
    { Copy the variable string block to buffer }
    For a:=Byte(sfComments) to Byte(sfProductVersion) do
     with VVI2[TStringFile(a)] do if CopySize>0 then begin
       Move(ByteArray,Buffer^[b],CopySize);
       b:=b+CopySize;
    end;
    { Copy last (constant) block }
    Move(VVI3,Buffer^[b],SizeOf(VVI3)); b:=b+SizeOf(VVI3);
    { save resource file }
    SaveFile(FileName,Buffer,b);
  finally
    Dispose(Buffer);
  end;
end;

{$ELSE} { Dette er VersionInfo 16 bit }

type
  FileHeader = record
    { Fileheader er tom p 16bits resourcefiler }
    Resource  :  record { Beskrivelse af resource i fil }
      ResTypeStart : Word; { always FF10 - VersionInfo }
      IdentifierMS : Word; { always 00FF - stter navn til et numerisk ettal }
      IdentifierLS : Word; { always 0100  }
      ResTypeStop  : Word; { always 3000 - VERSIONINFO }
      SizeOfRes    : Word; { same as cbBlock }
      Padding      : Word; { always 0000 }
      { her starter 16 bits resource VS_VERSION_INFO_1 }
    end;
  end;

{ Type erklring for VS_VERSION_INFO er opdelt i 3 dele, 1 og 3 er ikke variabel,
  mens del 2 er! Del 2 opbrydes i en rkke mindre ikke-variable dele
  som logikken i objektet skal kunne skifte mellem. }

  VS_VERSION_INFO_1 = record { frste del af VS_VERSION_INFO - ikke variabel }
    cbBlock  : Word;                 {  16bit - Size of complete block incl. nested blocks }
    cbValue  : Word;                 {  32bit - Size of "abValue" block - SizeOf(TVsFixedFileInfo) }
    szKey    : array[0..15] of char; { 160bit - always "VS_VERSION_INFO" }
  { padding1 : array[0..-1] of byte; { 160bit - align szKey on 32bit boundary }
    abValue  : Vs_FixedFileInfo;     { 576bit }
    Child1   : record
      cbBlock  : Word; { Size of complete block incl. nested blocks }
      cbValue  : Word; { Size of abValue block }
      szKey    : array[0..14] of char; { always "StringFileInfo" }
      padding1 : byte;                 { align szKey on 32bit boundary }
      Value : record
        cbBlock : Word; { Size of complete block incl. nested blocks }
        cbValue : Word; { Size of abValue block }
        szKey   : Array[0..8] of Char; { eks. "040904E4" }
        Padding : Array[0..2] of Byte;
      end; { value }
    end; { child1 }
  end; { part 1 }

  (*
  Den logiske opbygning af den variable blok - bemrk at arrays lngde er
  variabel ! }
  VS_VERSION_INFO_2 : record
    Children : array[0..9] of record
      cbBlock : Word;                  { Size of complete block          }
      cbValue : Word;                  { Size of abValue block           }
      szKey   : array[0..#] of char;   { "Comment","Companyname" etc     }
      padding : array[0..#] of byte;   { align szKey on 32bit boundary   }
      abValue : array[0..#] of char;
      padding2: array[0..#] of byte;   { align abValue on 32bit boundary }
    end;
  end;
  *)

  { Denne blok bruges i stedet, og der kopieres blot CopySize bytes fra
    Byte array over i fil-bufferen ... }

  VS_VERSION_INFO_2 = array[sfComments..sfProductVersion] of record
    CopySize : Word;     { Number of bytes to copy }
    CalcSize : Word;     { Size of block minus pad-bytes }
    case byte of
      0 : ( cbBlock : Word; { size of complete block }
            cbValue : Word; { size of abValue ( part of VarChar member ) }
            VarChar : Array[0..295] of char  );
      1 : ( ByteArray : Array[0..299] of byte);
  end;

  VS_VERSION_INFO_3 = record { Sidste del af VS_VERSION_INFO - ikke variabel }
    Child2  : record
      cbBlock : Word;                   { Size of complete block (Value=36) }
      cbValue : Word;                   { Size of abValue block  (Value=20) }
      szKey   : array[0..11] of char;   { always "VarFileInfo"  }
  {   padding : array[0..x] of byte;    {align szKey on 32bit boundary }
      children : record
        cbBlock : Word;                 { Size of complete block  (Value=20) }
        cbValue : Word;                 { Size of abValue block   (Value= 4) }
        szKey   : array[0..11] of char; { always "Translation" }
  {     padding : array[0..x] of byte; {align szKey on 32bit boundary }
        Language: Word; { LangID,CharSet }
        Charset : Word;
      end; { children }
    end;   { child 2 }
  end;     { part 3 }

{ Her fyldes de tre records med data og skrives s til disk som en
  16bits resourcefil }
procedure TVersionInfo.SaveToFile(FileName:string);
type
  TBuffer= Array[0..SizeOf(VS_VERSION_INFO_1)+
                    SizeOf(VS_VERSION_INFO_2)+
                    SizeOf(VS_VERSION_INFO_3)] of Byte;
  pBuffer= ^TBuffer;
var
  VVI1   : VS_VERSION_INFO_1;
  VVI2   : VS_VERSION_INFO_2;
  VVI3   : VS_VERSION_INFO_3;
  RFH    : FileHeader;
  Buffer : pBuffer;
  a,b    : integer;
begin
  BackUpFile(FileName);
  { Fill block VS_VERSION_INFO_1 }
  { 16bit VS_VERSION_INFO_1 }
  With VVI1 do begin
    cbBlock :=SizeOf(VS_VERSION_INFO_1)+
              SizeOf(VS_VERSION_INFO_3);
    cbValue :=SizeOf(abValue);
    StrCopy(szKey,'VS_VERSION_INFO');
    { 16bit VS_FIXEDFILEINFO : }
    With abValue do begin
      dwSignature     :=$FEEF04BD;
      dwStrucVersion  :=$00010000; { VersionInfo }
      DecodeFileFlags(dwFileFlags,dwFileFlagsMask);
      dwFileSubtype   :=$00000000;
      dwFileOS        :=$00000001; { VOS_Windows16 }
      dwFileDateMS    :=$00000000;
      dwFileDateLS    :=$00000000;
    { Default variable values }
      Case FFileType of
        ftProgram : dwFileType:=1; { VFT_APP }
        ftLibrary : dwFileType:=2; { VFT_DLL }
      end;
      StrToVer(FStringData.Strings[Byte(sfFileVersion)],
               dwFileVersionMS,dwFileVersionLS);
      StrToVer(FStringData.Strings[Byte(sfProductVersion)],
               dwProductVersionMS,dwProductVersionLS);
    end;
    With Child1 do begin
      cbBlock := SizeOf(Child1);
      cbValue := 0;
      StrCopy(szKey,'StringFileInfo');
      Padding1:=0;
      With Value do begin
        cbBlock := SizeOf(Value);
        cbValue := 0 ;
        StrPCopy(szKey,IntToHex(DecodeLanguage(FLanguage),4)+
                       IntToHex(DecodeCharset(FCharset),4));
        For a:=0 to 2 do Padding[a]:=0;
      end;
    end;
  end;
  { 16bit VS_VERSION_INFO_2 }
  For a:=Byte(sfComments) to Byte(sfProductVersion) do
   with VVI2[TStringFile(a)] do
   if FStringData.Strings[a]>'' then begin
    b:=0;
    StrPCopy(@VarChar[b],DecodeStringFile(TStringFile(a)));
    b:=Length(DecodeStringFile(TStringFile(a)));
    Repeat
      VarChar[b]:=#0;
      Inc(b);
    until ((b*8)) mod 32 = 0;
    StrPCopy(@VarChar[b],FStringData.Strings[a]);
    cbValue:=Length(FStringData.Strings[a]);
    b:=b+cbValue;
    Repeat
      VarChar[b]:=#0;
      Inc(b);
    until ((b*8)) mod 32 = 0;
    cbBlock :=4+b;
    CopySize:=4+b;
    CalcSize:=length(DecodeStringFile(TStringFile(a)))+
              Length(FStringData.Strings[a]);
    { Calculate nested block sizes }
    VVI1.cbBlock:=VVI1.cbBlock+CopySize;
    VVI1.Child1.cbBlock:=VVI1.Child1.cbBlock+CopySize;
    VVI1.Child1.Value.cbBlock:=VVI1.Child1.Value.cbBlock+CopySize;
  end else begin
    { No data - empty block }
    CopySize:=0;
    CalcSize:=0;
    For b:=0 to 299 do ByteArray[b]:=0;
  end;
  { 16bit VS_VERSION_INFO_3 }
  With VVI3.Child2 do begin
    { Denne blok er frdig }
    cbBlock:=SizeOf(VS_VERSION_INFO_3);
    cbValue:=0;
    StrCopy(szKey,'VarFileInfo');
    With Children do begin
      cbBlock:=SizeOf(Children);
      cbValue:=4;
      strCopy(szKey,'Translation');
      Language:=DecodeLanguage(FLanguage);
      Charset :=DecodeCharset (FCharset );
    end;
  end;
  { 16bit Resource file header - block identifying first resource }
  With RFH.Resource do begin
    ResTypeStart:=$10FF;
    IdentifierMS:=$FF00;
    IdentifierLS:=$0001;
    ResTypeStop :=$0030;
    SizeOfRes   :=VVI1.cbBlock;
    Padding     :=$0000;
  end;
  { Create temporary write buffer }
  GetMem(Buffer,SizeOf(TBuffer));
  Try
    { fill buffer with binary image of file }
    Move(RFH,Buffer^[0],SizeOf(RFH));   b:=SizeOf(RFH);
    Move(VVI1,Buffer^[b],SizeOf(VVI1)); b:=b+SizeOf(VVI1);
    { Copy the variable string block to buffer }
    For a:=Byte(sfComments) to Byte(sfProductVersion) do
     with VVI2[TStringFile(a)] do if CopySize>0 then begin
       Move(ByteArray,Buffer^[b],CopySize); b:=b+CopySize;
    end;
    { Copy constant block }
    Move(VVI3,Buffer^[b],SizeOf(VVI3)); b:=b+SizeOf(VVI3);
    { save 16bit resource file }
    SaveFile(FileName,Buffer,b);
  finally
    FreeMem(Buffer,SizeOf(TBuffer));
  end;

end;

{$ENDIF}

{ Internal protected procedures }
{$IFNDEF WIN32}

Const
  VS_FF_DEBUG = 1;
  VS_FF_PRERELEASE = 2;
  VS_FF_PATCHED = 4;
  VS_FF_PRIVATEBUILD = 8;
  VS_FF_INFOINFERRED = $10;
  VS_FF_SPECIALBUILD = $20;
{$ENDIF}

procedure TVersionInfo.DecodeFileFlags(Var Flags,Mask:DWORD);
begin
  Flags:=0;
  if ffDebug        in FFileFlag then Flags:=Flags or VS_FF_Debug;
  if ffPrerelease   in FFileFlag then Flags:=Flags or VS_FF_Prerelease;
  if ffPatched      in FFileFlag then Flags:=Flags or VS_FF_Patched;
  if ffPrivateBuild in FFileFlag then Flags:=Flags or VS_FF_PrivateBuild;
  if ffInfoInferred in FFileFlag then Flags:=Flags or VS_FF_InfoInferred;
  if ffSpecialbuild in FFileFlag then Flags:=Flags or VS_FF_SpecialBuild;
  Mask:=Flags;
end;

procedure TVersionInfo.BackUpFile(FileName:string);
{ Skal udbygges - backup rutinen krver at resourcefilen altid har
  extension .Res }
{ create backup of existing .res file }
begin
  { remove previous backup if it exists }
  If FileExists(FileName+'.~re') then begin
    FileSetAttr(FileName+'~re',0); { remove all attribs }
    SysUtils.DeleteFile(FileName+'.~re');
  end;
  { rename old res file if it exists }
  If FileExists(FileName+'.res') then begin
    FileSetAttr(FileName+'.res',0); { remove all attribs }
    Renamefile(FileName+'.res',FileName+'.~re');
  end;
end;

procedure TVersionInfo.SaveFile(FileName:string;Buffer:pointer;BufSize:Word);
{ write contents of buffer to disk }
var
  FoH : THandle;
  Err : boolean;
begin
  {$I-}
  FoH:=FileCreate(FileName);
  Err:=(Foh<0);
  If Foh>0 then begin
    Err:=(FileWrite(FoH,Buffer^,BufSize)<0);
    FileClose(FoH);
  end;
  If Err then
    raise exception.create('Error writing file');
  {$I+}
end;

procedure TVersionInfo.StrToVer(Value:string;var MS,LS:DWORD);
{ return a 4 word version number from a string version number }
var
  OP : record
    case byte of
      0 : ( Words : Array[0..3] of Word ) ;
      1 : ( DWords: Array[0..1] of DWORD) ;
  end;
  a  : byte;
begin
  a:=1;
  While a<length(Value) do if not (Value[a] in ['0'..'9','.']) then
    System.delete(Value,a,1)  { if not number - delete it }
  else
    Inc(a);
  Value:=Value+'.';
  For a:=3 downto 0 do try
    OP.Words[a]:=StrToInt(System.Copy(Value,1,Pos('.',Value)-1));
    System.Delete(Value,1,Pos('.',Value));
  except
    OP.Words[a]:=0;
  end;
  LS:=OP.DWORDS[0];
  MS:=OP.DWORDS[1];
end;

function  TVersionInfo.DecodeLanguage(language:TLanguage):Word;
{ return language code from user-def. language type }
var
  value : Word;
begin
  case Language of
    lanArabic           : Value:=$0401;  lanBulgarian           : Value:=$0402;
    lanCatalan          : Value:=$0403;  lanTraditional_Chinese : Value:=$0404;
    lanCzech            : Value:=$0405;  lanDanish              : Value:=$0406;
    lanGerman           : Value:=$0407;  lanGreek               : Value:=$0408;
    lanUS_English       : Value:=$0409;  lanCastilian_Spanish   : Value:=$040A;
    lanFinnish          : Value:=$040B;  lanFrench              : Value:=$040C;
    lanHebrew           : Value:=$040D;  lanHungarian           : Value:=$040E;
    lanIcelandic        : Value:=$040F;  lanItalian             : Value:=$0410;
    lanJapanese         : Value:=$0411;  lanKorean              : Value:=$0412;
    lanDutch            : Value:=$0413;  lanNorwegian_Bokmal    : Value:=$0414;
    lanPolish           : Value:=$0415;  lanBrazilian_Portuguese: Value:=$0416;
    lanRhaeto_Romanic   : Value:=$0417;  lanRomanian            : Value:=$0418;
    lanRussian          : Value:=$0419;  lanCroato_Serbian_latin: Value:=$041A;
    lanSlovak           : Value:=$041B;  lanAlbanian            : Value:=$041C;
    lanSwedish          : Value:=$041D;  lanThai                : Value:=$041E;
    lanTurkish          : Value:=$041F;  lanUrdu                : Value:=$0420;
    lanBahasa           : Value:=$0421;  lanSimplified_Chinese  : Value:=$0804;
    lanSwiss_German     : Value:=$0807;  lanUK_English          : Value:=$0809;
    lanMexican_Spanish  : Value:=$080A;  lanBelgian_French      : Value:=$080C;
    lanSwiss_Italian    : Value:=$0810;  lanBelgian_Dutch       : Value:=$0813;
    lanNorwegian_Nynorsk: Value:=$0814;  lanPortuguese          : Value:=$0816;
    lanSerbo_Croatian   : Value:=$081A;  lanCanadian_French     : Value:=$0c0C;
    lanSwiss_French     : Value:=$100C;  else                     Value:=$0000;
  end;
  Decodelanguage:=Value;
end;

function  TVersionInfo.DecodeCharset (Charset:TCharset):Word;
{ return Charset code from user-def. type }
var
  value : Word;
begin
  Case CharSet of
    cs7bit_ASCII       : Value:=   0;
    csJapan_JIS_X_0208 : Value:= 932;
    csKorea_KSC_5601   : Value:= 949;
    csTaiwan_GB5       : Value:= 950;
    csUnicode          : Value:=1200;
    csLatin_2          : Value:=1250;
    csCyrillic         : Value:=1251;
    csMultilingual     : Value:=1252;
    csGreek            : Value:=1253;
    csTurkish          : Value:=1254;
    csHebrew           : Value:=1255;
    csArabic           : Value:=1256;
  end;
  DecodeCharset:=Value;
end;

function  TVersionInfo.DecodeStringFile(SFI:TStringFile):string;
{ return StringFileInfo string from user-def. type }
var
  value : string;
begin
  Case SFI of
    sfComments        : Value:='Comments';
    sfCompanyName     : Value:='CompanyName';
    sfFileDescription : Value:='FileDescription';
    sfFileVersion     : Value:='FileVersion';
    sfInternalName    : Value:='InternalName';
    sfLegalCopyright  : Value:='LegalCopyright';
    sfLegalTrademarks : Value:='LegalTrademarks';
    sfOriginalFilename: Value:='OriginalFilename';
    sfProductName     : Value:='ProductName';
    sfSpecialBuild    : Value:='SpecialBuild';
    sfPrivateBuild    : Value:='PrivateBuild';
    sfProductVersion  : Value:='ProductVersion';
  end;
  DecodeStringFile:=Value;
end;

{ Init and exit procedures }

Constructor TVersionInfo.Create(AnOwner:TObject);
begin
  Inherited Create;
  FStringData :=TStringList.Create;
  Clear;
end;

procedure TVersionInfo.Free;
begin
  Clear;
  FStringData.Free;
  Inherited Free;
end;

{ Methods }

procedure TVersionInfo.Clear;
{ clear temporary storage of data }
var
  a : integer;
begin
  FStringData .Clear;
  FFileType   :=ftProgram;
  FLanguage   :=lanUK_English;
  FCharSet    :=csMultilingual;
  FFileFlag   :=[];
  For a:=Byte(sfComments) to Byte(sfProductVersion) do
    FStringData.Add('');
end;

procedure TVersionInfo.SetString(SFI:TStringFile;Data:string);
{ Store StringFileInfo string data in temporary storage
  NOTE: At least ONE StringFileData must be included in every
        resourcefile }
begin
  FStringData.Strings[Byte(SFI)]:=Data;
end;

function  TVersionInfo.GetString(SFI:TStringFIle):string;
begin
  GetString:=FStringData.Strings[Byte(SFI)];
end;


{ * TBackup definition ******************************************************* }

procedure TBackup.SetTarget(value:string);
begin
  If DirectoryExists(value) then
    FTarget:=QualifyPath(value)
  else
    raise exception.create('Target path does not exist');
end;

procedure TBackup.SetSource(value:string);
begin
  If DirectoryExists(value) then begin
    FSource:=QualifyPath(value);
  end else
    raise exception.create('Source path does not exist');
end;

procedure TBackup.SetPattern(value:string);
begin
  if value='' then value:='*.*;';
  if value[length(Value)]<>';' then value:=value+';';
  FPattern:=value;
end;

procedure TBackup.ForceDirectories(Dir: string);
{ Recursive algoritm which creates subdirs   }
{ copied from Delphi 2.0, copyrights Borland }
begin
  if Dir[Length(Dir)] = '\' then
    delete(dir,length(dir),1);
  if (Length(Dir) < 3) or DirectoryExists(Dir) then Exit;
  ForceDirectories(ExtractFilePath(Dir));
  MkDir(Dir);
end;

function TBackup.FileCopy(Input,output:string;Size:LongInt):boolean;
{ copies "size" bytes from file "input" to file "output", "input/output" must
  be full qualified filenames in the form "drive\path\name". }
var
  pTemp      : array[0..99] of char;
  InputFile  : THandle;
  OutPutFile : THandle;
  Buffer     : Pointer;
  Count      : LongInt;
  Drive      : byte;
  FileAttr   : LongInt;
begin
  FileCopy:=TRUE;
  Drive:=(Byte(Output[1]) and not 32)-64;       { Find output drev }
  If DiskFree(Drive)<size then begin
    FileCopy:=FALSE;
    raise exception.create('Drive '+Char(Drive+64)+': is full, '+
          'cannot copy '+ExtractFileName(Input));
  end else try
    GetMem (buffer,$FFF0);                    { Create 64K-1 buffer in memory }
    {$I-}                                     { Disable normal errorhandling }
    try
      FileAttr:=FileGetAttr(input);
      If FileAttr<>-1 then begin
        { remove ReadOnly on new file }
        FileAttr:=(FileAttr and not (faArchive  or faReadOnly));
        FileSetAttr(Output,FileAttr);
      end;
      InputFile:=FileOpen(Input,fmOpenRead);    { open input file }
      OutputFile:=FileCreate(Output);           { create output file }
      While size>0 do begin
        case size>$FFF0 of
          true : Count:=$FFF0;                  { max 64K-16 bytes, or if  }
          false: Count:=Size;                   { possible - copy whole file }
        end;
        FileRead (Inputfile ,Buffer^,Count);    { read file }
        FileWrite(Outputfile,Buffer^,Count);    { write new file }
        size:=size-count;
      end;
    finally
      FileClose(Inputfile);                     { close input file }
      FileClose(Outputfile);                    { close output file }
    end;
    {$I+}                                     { Re-enable normal errorhandling }
    If IOresult<>0 then begin
      FileCopy:=FALSE;
      raise exception.create('Error occured during backup of "'+
               LowerCase(ExtractFileName(Input))+'".'+#13+
               'Backup may not be complete')
    end else
      { Remove Archive bit on Input-file }
      FileSetAttr(Input,FileAttr);
  finally
    FreeMem(buffer,Size);                     { free buffer }
    Application.ProcessMessages;              { Allow Windows handling... }
  end;
end;

function TBackup.GetTargetPath(SourcePath:string):string;
{ Pre : Source and Target must be qualified before using this
        function.
  post: Result is quarentied to exist and be qualified. }
begin
  system.delete(SourcePath,1,length(Source));
  ForceDirectories(Target+SourcePath);
  Result:=QualifyPath(Target+SourcePath);
end;

function TBackup.QualifyPath(path:string):string;
{ Expands path and adds driveletters etc. then }
{ adds a trailing backslash to a string if needed }
begin
  Path:=ExpandFileName(Path);
  If Path[length(Path)]<>'\' then Path:=Path+'\';
  QualifyPath:=Path;
end;

procedure TBackup.ScanFolders(Path:string);
var
  SR : TSearchRec;
begin
  try
    CopyMatchingFiles(Path);
    If FindFirst(Path+'*.*',faDirectory,SR)=0 then repeat
      If ((SR.Attr and faDirectory)=faDirectory) and
        (SR.Name[1]<>'.') then begin
          ScanFolders(Path+SR.Name+'\');
        end;
    until (FindNext(SR)<>0);
  finally
    FindClose(SR);
  end;
end;

procedure TBackup.CopyMatchingFiles(Path:string);
var
  s,p : string;
begin
  p:=pattern;
  While Pos(' ',p)>0 do system.delete(p,Pos(' ',p),1);
  case Archive of
    true : While pos(';',p)>0 do begin
      s:=system.copy(P,1,Pos(';',p)-1);
      system.delete(p,1,pos(';',p));
      CopyArchiveFiles(Path,s);
    end;
    false: While pos(';',p)>0 do begin
      s:=system.copy(P,1,Pos(';',p)-1);
      system.delete(p,1,pos(';',p));
      CopyFiles(Path,s);
    end;
  end; { case }
end;

procedure TBackup.CopyFiles  (Path,Pattern:string);
var
  SR : TSearchRec;
begin
  try
    If FindFirst(Path+Pattern,FAttribs,SR)=0 then repeat
      FAnyError:=(not FileCopy(Path+SR.Name,GetTargetPath(Path)+SR.Name,SR.Size))
                 or FAnyError;
    until (FindNext(SR)<>0);
  finally
    FindClose(SR);
  end;
end;

procedure TBackup.CopyArchiveFiles(Path,Pattern:string);
var
  SR : TSearchRec;
begin
  try
    If FindFirst(Path+Pattern,FAttribs,SR)=0 then repeat
      If (SR.Attr and faArchive) = faArchive then
        FAnyError:=(not FileCopy(Path+SR.Name,GetTargetPath(Path)+SR.Name,SR.Size))
                   or FAnyError;
    until (FindNext(SR)<>0);
  finally
    FindClose(SR);
  end;
end;

function TBackup.Execute:boolean;
begin
  FAnyError:=FALSE;
  ScanFolders(Source);
  Execute:=NOT FAnyError;
end;

constructor TBackup.Create;
begin
  inherited Create;
  FAttribs:=(faAnyFile and not (faVolumeId or faDirectory));
  FSource:='';
  FTarget:='';
  FArchive:=FALSE;
end;

procedure TBackup.Free;
begin
  inherited Free;
end;

{ NT/95 Registry Interface *************************************************** }
{$IFDEF WIN32}
constructor TUserReg.Create;
begin
  inherited Create;
  KeyOpen:=FALSE;
end;

procedure TUserReg.Free;
begin
  if KeyOpen then Close;
  Inherited Free;
end;

procedure TUserReg.Open(key:string);
var
  SubKey : array[0..99] of char;
  Disposition : DWORD;
begin
  If KeyOpen then Close;
  If Pos('\',Key)=1 then delete(key,1,1);
  StrPCopy(SubKey,Key);
  KeyOpen:=RegCreateKeyEx(
       HKEY_CURRENT_USER,
       SubKey,
       0,
       NIL,
       REG_OPTION_NON_VOLATILE,
       KEY_ALL_ACCESS,
       NIL,
       KeyHandle, { HKEY }
       @Disposition)=ERROR_SUCCESS;
  If not KeyOpen then
    raise exception.create('Cannot connect to registry');
end;

procedure TUserReg.Close;
begin
  If KeyOpen then begin
    KeyOpen:=(RegCloseKey(KeyHandle)<>ERROR_SUCCESS);
    If KeyOpen then
      raise exception.create('Error disconnecting from the Registry');
  end;
end;

procedure TUserReg.WriteString (Name,Value:string);
var
  SubKey : Array[0..299] of char;
  Temp   : Array[0..299] of char;
  DataType : DWORD;
  ASuccess : Boolean;
  BufSize  : DWORD;
begin
  StrPCopy(SubKey,Name);
  StrPCopy(Temp,Value);
  BufSize:=Length(Temp);
  DataType := REG_SZ;
  ASuccess:=RegSetValueEx(KeyHandle, SubKey, 0, REG_SZ, @Temp,
    BufSize)=ERROR_SUCCESS;
  if not asuccess then
    raise exception.create('Error writing string to Registry');
end;

procedure TUserReg.WriteInteger(Name:string;Value:integer);
var
  SubKey : Array[0..299] of char;
  Temp   : Array[0..20] of char;
  DataType : DWORD;
  ASuccess : Boolean;
  BufSize  : DWORD;
begin
  StrPCopy(SubKey,Name);
  StrPCopy(Temp,IntToStr(Value));
  BufSize:=Length(Temp);
  DataType := REG_SZ;
  ASuccess:=RegSetValueEx(KeyHandle, SubKey, 0, REG_SZ, @Temp,
    BufSize)=ERROR_SUCCESS;
  if not asuccess then
    raise exception.create('Error writing integer to Registry');
end;

function  TUserReg.ReadString  (Name,Default:string):string;
var
  SubKey : Array[0..299] of char;
  DataType : DWORD;
  ASuccess : Boolean;
  BufSize : DWORD;
begin
  StrPCopy(SubKey,Name);
  BufSize:=299;
  DataType := REG_SZ;
  ASuccess:=RegQueryValueEx(KeyHandle, SubKey, nil, @DataType, @SubKey,
    @BufSize)=error_success;
  if asuccess then
    ReadString:=StrPas(SubKey)
  else
    ReadString:=DEFAULT;
end;

function  TUserReg.ReadInteger (Name:string;Default:integer):integer;
var
  SubKey : Array[0..299] of char;
  DataType : DWORD;
  ASuccess : Boolean;
  BufSize : DWORD;
begin
  StrPCopy(SubKey,Name);
  BufSize:=299;
  DataType := REG_DWORD;
  ASuccess:=RegQueryValueEx(KeyHandle, SubKey, nil, @DataType, @SubKey,
    @BufSize)=error_success;
  if asuccess then
    ReadInteger:=StrToInt(StrPas(SubKey))
  else
    ReadInteger:=DEFAULT;
end;
{$ENDIF}

{ Registration procedure ***************************************************** }

procedure Register;
begin
  RegisterComponents('Samples', [TProject]);
end;

end.

{ Revision story:

version 2.2
   017	  Changed TProject.Create to check for design time so no calls will be made
	  to Delphi .ini / Delphi registry - keys at runtime
   016    Check of Version name format changed to removal of all non-numeric chars
          (except from '.') 
   015    Added TBackup object - multiciple directory backup with ArchiveBit check
   014    Split of Log and Notes into two properties. Log msg will be reversed
          ordered - newest msg at top.
   013    Added TUserReg object as a workaround the problems with Borlands
          Registry component.
version 2.1 - uploaded to Delphi Super Page 5. januar 1997
   012    Bug fixed - Error calculating size of 32bit Versionresource
   011    GetString method added to the TVersionInfo object
   010    ResourceFile property added - user may now define own filename
   009    Code cleanup - Messagebox changed to "raise exception.create" for
          all error-msg.
          Code cleanup - Property name shows underscore "_" as space " " in
          application.title and versionresource string "Internal name"
   010    Code cleanup - FOnceOnly refined (no backup question when
          loading project)
          Extension-removal removed from Version Resource Writer
          (no "error writting ver-res" when path contains a '.')
   008    Code cleanup - Version number now checked as float, stored as string
          (Ver 0.01 now possible)
version 2.0
   007    Version Resource Writer added
version 1.1
   006	  Autoname added
version 1.0
   005    bug fixed - Access violation when switching between components.
                      (caused by "GetActiveWindow" !)
   004    bug fixed - FBuild not reset when using autobackup
   003    Bug fixed - FBuild was reset with every project load
   002    Bug fixed - TProject was Delphi 2.0 dependent
   001    failed if more than one dir-level should be created
          backup err msg on first copy if createbackup dir failed
   000    unknown
}




