unit Dbnavnew;
{$D-}
{
    Copyright (c) 1995 by Maggie Owens.

    Please send bug reports, suggestions & comments to mmowens@panix.com.
    Updates (if any) will also be avialable for download from:
      http://www.panix.com/~mmowens/

    Revision history:
      This is version 1.1, finished 7/25/96.
      Corrects the Vertical Orientation bug.
      Version 1.0 first released to the public June 10, 1996.

    If you would like to included on the update notification mailing list,
    please send email to mmowens@panix.com with "subscribe navigator" as
    the subject.
    I would also appreciate it if you send me your name if you use this
    navigator, so I can maintain a usage count.

    Revision history:
    This component was originally developed in early '95 by modifying a copy of
    Delphi 1.0's DBNavigator to correct the resource leak and add functionality:
    lookuphep, locate and locatenext buttons; direct access to the enabled /
    disabled state of the buttons, and the ability to override the state of any
    given button. This version of the navigator is in use in numerous production
    applications.
    The set bookmark, goto bookmark and clear bookmark buttons were added in
    early '96 for my Borland conference paper. The newer version navigator has
    been thoroughly tested and debugged, but has not yet been included in any
    production applications.

    Code that differs from the Delphi 1.0 DBNavigator has been marked with the
    comment "New".

    Special thanks to:
      Ed Jordan, for helping me solve the problem of more than 16 elements in
        the set of published properties, by providing sample code for the
        TmoNavBtnSwitches object.
      Bill Morgenthein, for the set bookmark, goto bookmark and clear bookmark
        bitmaps (My drawing abilities rate -1 on a scale of 1 to 10).
      Everyone at the Burgiss group, for using the navigator.
      My mother, Terry Winter Owens (http://www.panix.com/~twowens/)
        for her tireless help in editing my conference papers.
      Lizards, in general, for eating the bugs.

    DISCLAIMER: I have provided these components and source code to the public
    free of charge. You accept these components AS IS without any representation
    or warranty of any kind, including but not limited to the warranty of
    merchantability or fitness for a particular purpose. You may not sell these
    components.
}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, Menus, StdCtrls, DB, DBTables, Mask, Buttons;

const
  InitRepeatPause = 400;  { pause before repeat timer (ms) }
  RepeatPause     = 100;  { pause before hint window displays (ms)}
  SpaceSize       =  5;   { size of space between special buttons }
  DefaultBookmarkMessage =
    'The bookmark you are trying to use belongs to a different data set than the one you are on.'
     + #13+
    'Continue anyway?';   { Message to display if ShowBookmarkMessage is true
                           and BookmarkMessage is blank.}

type
  TmoNavBtn = class;
  TNavDataLink = class;

  TNavGlyph = (ngEnabled, ngDisabled);
  {Note the new buttons below: nbLookupHelp (Lookup Help), nbLocate (Locate
  record), nbLocateNext (Locate next record), nbBkSet (Set Bookmark), nbBkGoto
  (Go to bookmark), nbBkClear (Clear Bookmark), nbPriorSet (move back 10
  record) and nbNextSet (move forward 10 records). nbLookupHelp, nbLocate and
  nbLocateNext are initially disabled. These three buttons require
  event-handling procedures in order to do anything.}
  TmoNavBtns = (nbLookupHelp, nbLocate, nbLocateNext, nbBkSet, nbBkGoto,
    nbBkClear, nbFirst, nbPriorSet, nbPrior, nbNext, nbNextSet, nbLast,
    nbInsert, nbDelete, nbEdit, nbPost, nbCancel, nbRefresh);
  TmoBtnSet = set of TmoNavBtns;
  TmoNavButtonStyle = set of (nsAllowTimer, nsFocusRect);
  ENavClick = procedure (Sender: TObject; Button: TmoNavBtns) of object;
  TmoNavOrientation = (noHorizontal, noVertical); {New}

  TmoGetBtn = function: TmoBtnSet of object; {New}
  TmoSetBtn = procedure(Value: TmoBtnSet) of object; {New}

  TmoNavBtnSwitches = class(TPersistent) {New}
  private
    GetBtn: TmoGetBtn;
    SetBtn: TmoSetBtn;
    function GetSwitch(Index: Integer): Boolean;
    procedure SetSwitch(Index: Integer; Value: Boolean);
  public
    constructor Create(AGetFunc: TmoGetBtn; ASetProc: TmoSetBtn);
  published
    property nbLookupHelp: Boolean index 0 read GetSwitch write SetSwitch default False;
    property nbLocate: Boolean index 1 read GetSwitch write SetSwitch default False;
    property nbLocateNext: Boolean index 2 read GetSwitch write SetSwitch default False;
    property nbBkSet: Boolean index 3 read GetSwitch write SetSwitch default True;
    property nbBkGoto: Boolean index 4 read GetSwitch write SetSwitch default True;
    property nbBkClear: Boolean index 5 read GetSwitch write SetSwitch default True;
    property nbFirst: Boolean index 6 read GetSwitch write SetSwitch default True;
    property nbPriorSet: Boolean index 7 read GetSwitch write SetSwitch default True;
    property nbPrior: Boolean index 8 read GetSwitch write SetSwitch default True;
    property nbNext: Boolean index 9 read GetSwitch write SetSwitch default True;
    property nbNextSet: Boolean index 10 read GetSwitch write SetSwitch default True;
    property nbLast: Boolean index 11 read GetSwitch write SetSwitch default True;
    property nbInsert: Boolean index 12 read GetSwitch write SetSwitch default True;
    property nbDelete: Boolean index 13 read GetSwitch write SetSwitch default True;
    property nbEdit: Boolean index 14 read GetSwitch write SetSwitch default True;
    property nbPost: Boolean index 15 read GetSwitch write SetSwitch default True;
    property nbCancel: Boolean index 16 read GetSwitch write SetSwitch default True;
    property nbRefresh: Boolean index 17 read GetSwitch write SetSwitch default True;
  end; {TmoNavBtnSwitches}

  TmoDBNav = class(TCustomPanel)
  private
    FVisibleSwitches: TmoNavBtnSwitches; {New - Used to set the visiblebtns
      property at design time. This is necessary because you can't publish a set
      property that has more than 16 elements.}
    FOrientation: TmoNavOrientation; {New - horizontal or vertical. Extra code
      is in the procedure AdjustSize.}
    FBookmark: TBookmark; {New}
    FShowBookmarkMessage: Boolean; {New - indicates whether or not a message
      should be displayed when the user clicks Goto Bookmark while on a dataset
      other than that on which the bookmark was set.}
    FBookmarkMessage: String; {New - this is the message that is displayed if
      FShowBookmarkMessage is true and the user clicks the Goto Bookmark button
      while on a different dataset than that on which the bookmark was set.}
    FBookmarkDs: TDataSet; {New - the dataset for which the bookmark was set.}
    FDataLink: TNavDataLink;
    FEnabledButtons: TmoBtnSet; {New - specifies which of the visible buttons
      are enabled.}
    FUserEnabledButtons: TmoBtnSet; {Specifies which of the buttons have been
      explicitly enabled by the user.}
    FVisibleButtons: TmoBtnSet;
    FHints: TStrings;
    ButtonWidth: Integer;
    ButtonHeight: Integer;
    MinBtnSize: TPoint;
    FOnNavClick: ENavClick;
    FOnLookupHelp: ENavClick; {New - procedure to be executed when the
      lookuphelp button is clicked. Nothing will happen if there is no procedure
      for this event.}
    FOnLocate: ENavClick; {New - procedure to be executed when the
      locate button is clicked. Nothing will happen if there is no procedure
      for this event.}
    FOnLocateNext: ENavClick; {New - procedure to be executed when the
      locatenext button is clicked. Nothing will happen if there is no procedure
      for this event.}
    FOnDataChanged: TNotifyEvent; {New - procedure to be executed when the
      DataChanged event of the navigator occurs.}
    FOnEditingChanged: TNotifyEvent; {New - procedure to be executed when the
      EditingChanged event of the navigator occurs.}
    FocusedButton: TmoNavBtns;
    FConfirmCancel: Boolean; {New - displays a confirmation dialog when the user
      tries to cancel a record, much like ConfirmDelete.}
    FConfirmDelete: Boolean;
    FMaxSetMove: Integer; {New - max # of records to move on NextSet/PriorSet.
      Checked in SetMoveByAmount.}
    FSetMove: Integer; {# of records to move on NextSet/PriorSet}
    function GetVisibleButtons: TmoBtnSet; {New - Function to be called by
      TmoNavBtnSwitches to determine which buttons are enabled.}
    procedure SetVisibleButtons(Value: TmoBtnSet); {New - function called by
      TmoNavBtnSwitches to change the set of visible buttons.}
    function GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);
    procedure InitButtons;
    procedure InitHints;
    procedure Click(Sender: TObject);
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetEnabled(Value: TmoBtnSet);
    procedure SetMaxSetMove(Value: Integer); {New - used to set FMaxSetMove.}
    procedure SetMoveByAmount(Value: Integer); {New - used to set SetMove.}
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure SetHints(Value: TStrings);
    procedure SetOrientation(Value: TmoNavOrientation); {New - used to change
      the orientation from horizontal to vertical or vice-versa.}
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure UpdateEnabled; {New - used to update the set of enabled buttons
      when they are enabled or disabled.}
  protected
    Buttons: array[TmoNavBtns] of TmoNavBtn;
    procedure DataChanged;
    procedure EditingChanged;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property EnabledButtons : TmoBtnSet read FEnabledButtons; {New - runtime
      readonly property that returns the set of enabled buttons.}
    property VisibleButtons: TmoBtnSet read FVisibleButtons write FVisibleButtons;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure BtnClick(Index: TmoNavBtns);
    procedure SetButtonState(Index: TmoNavBtns; bEnable: Boolean); {New - used
      to explicitly enable or disable a particular button, overriding its
      default state.}
  published
    property VisibleBtns: TmoNavBtnSwitches read FVisibleSwitches write FVisibleSwitches;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property ShowBookmarkMessage: Boolean read FShowBookmarkMessage
      write FShowBookmarkMessage default True; {New}
    property BookmarkMessage: String read FBookmarkMessage write FBookmarkMessage; {New}
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Ctl3D;
    property Hints: TStrings read FHints write SetHints;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ConfirmCancel: Boolean read FConfirmCancel write FConfirmCancel default False; {New}
    property ConfirmDelete: Boolean read FConfirmDelete write FConfirmDelete default True; {New}
    property Orientation: TmoNavOrientation read FOrientation write SetOrientation default noHorizontal; {New}
    property MaxSetMove: Integer read FMaxSetMove write SetMaxSetMove default 100; {New}
    property SetMove: Integer read FSetMove write SetMoveByAmount default 10; {New}
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick: ENavClick read FOnNavClick write FOnNavClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
    {New - proc to execute when LookupHelp is clicked.}
    property OnLookupHelp: ENavClick read FOnLookupHelp write FOnLookupHelp;
    {New - proc to execute when locate is clicked.}
    property OnLocate: ENavClick read FOnLocate write FOnLocate;
    {New - proc to execute when LocateNext is clicked.}
    property OnLocateNext: ENavClick read FOnLocateNext write FOnLocateNext;
    {New - proc to execute when OnDataChanged occurs.}
    property OnDataChanged: TNotifyEvent read FOnDataChanged write FOnDataChanged;
    {New - proc to execute when OnEditingChanged occurs.}
    property OnEditingChanged: TNotifyEvent read FOnEditingChanged write
      FOnEditingChanged;
  end; {TmoDBNav}

  { TmoNavBtn }
  TmoNavBtn = class(TSpeedButton)
  private
    FIndex: TmoNavBtns;
    FNavStyle: TmoNavButtonStyle;
    FRepeatTimer: TTimer;
    procedure TimerExpired(Sender: TObject);
  protected
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
  public
    destructor Destroy; override;
    property NavStyle: TmoNavButtonStyle read FNavStyle write FNavStyle;
    property Index : TmoNavBtns read FIndex write FIndex;
  end; {TmoNavBtn}

  { TNavDataLink }
  TNavDataLink = class(TDataLink)
  private
    FNavigator: TmoDBNav;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
  public
    constructor Create(ANav: TmoDBNav);
    destructor Destroy; override;
  end; {TNavDataLink}


procedure Register;

implementation
{$IFDEF VER80}
  {$R dbnav16.RES}
{$ENDIF}
{$IFDEF VER90}
  {$R DBNAV32.RES}
{$ENDIF}

uses
  DBIErrs, DBITypes, Clipbrd, DBConsts;

const
  {New: constants for hint identifiers for the new buttons.}
  SLookupHelp = 50000;
  SLocate = 50001;
  SLocateNext = 50002;
  SBookmarkSet = 50003;
  SBookmarkClear = 50004;
  SBookmarkGoto = 50005;
  SPriorSet = 50006;
  SNextSet = 50007;
  SCancelChangesQuestion = 50008;
  BtnStateName: array[TNavGlyph] of PChar = ('EN', 'DI');
  BtnTypeName: array[TmoNavBtns] of PChar = ('LOOKUPHELP', 'LOCATE',
    'LOCATENEXT', 'BKSET', 'BKGOTO', 'BKCLEAR', 'FIRST', 'PRIORSET', 'PRIOR',
    'NEXT', 'NEXTSET', 'LAST', 'INSERT', 'DELETE', 'EDIT', 'POST', 'CANCEL',
    'REFRESH'); {New}
  BtnHintId: array[TmoNavBtns] of Word = (SLookupHelp, SLocate, SLocateNext,
    SBookmarkSet, SBookmarkGoto, SBookmarkClear, SFirstRecord, SPriorSet,
    SPriorRecord, SNextRecord, SNextSet, SLastRecord, SInsertRecord,
    SDeleteRecord, SEditRecord, SPostEdit, SCancelEdit, SRefreshRecord); {New}

procedure Register;
begin
  RegisterComponents('Samples', [TmoDBNav])
end; {Resister}

{ TmoNavBtnSwitches }

constructor TmoNavBtnSwitches.Create(AGetFunc: TmoGetBtn; ASetProc: TmoSetBtn);
begin
  inherited Create;
  GetBtn := AGetFunc;
  SetBtn := ASetProc;
  nbLookupHelp := False;
  nbLocate := False;
  nbLocateNext := False;
  nbBkSet := True;
  nbBkGoto := True;
  nbBkClear := True;
  nbFirst := True;
  nbPriorSet := True;
  nbPrior := True;
  nbNext := True;
  nbNextSet := True;
  nbLast := True;
  nbInsert := True;
  nbDelete := True;
  nbEdit := True;
  nbPost := True;
  nbCancel := True;
  nbRefresh := True
end; {Create}

function TmoNavBtnSwitches.GetSwitch(Index: Integer): Boolean;
begin
  Result := TmoNavBtns(Index) in GetBtn
end; {GetSwitch}

procedure TmoNavBtnSwitches.SetSwitch(Index: Integer; Value: Boolean);
begin
  if Value then
    SetBtn(GetBtn + [TmoNavBtns(Index)])
  else
    SetBtn(GetBtn - [TmoNavBtns(Index)])
end; {SetSwitch}

{ TmoDBNav }

constructor TmoDBNav.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] +
    [csFramed, csOpaque];
  FDataLink := TNavDataLink.Create(Self);
  FVisibleButtons := [nbBkSet, nbBkGoto, nbBkClear, nbFirst, nbPriorSet,
    nbPrior, nbNext, nbNextSet, nbLast, nbInsert, nbDelete, nbEdit, nbPost,
    nbCancel, nbRefresh]; {New}
  FEnabledButtons := FVisibleButtons; {New}
  FUserEnabledButtons := FVisibleButtons; {New}
  FShowBookmarkMessage := True; {New}
  FHints := TStringList.Create;
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Width := 22*15; {New}
  Height := 25;
  ButtonWidth := 0;
  ButtonHeight := 0;
  FocusedButton := nbBkSet; {New}
  FConfirmCancel := False; {New}
  FConfirmDelete := True;
  FMaxSetMove := 100; {New}
  FSetMove := 10; {New}
  FOrientation := noHorizontal; {New}
  FBookmark := nil; {New}
  FBookmarkDs := nil; {New}
  FVisibleSwitches := TmoNavBtnSwitches.Create(GetVisibleButtons, SetVisibleButtons) {New}
end; {Create}

destructor TmoDBNav.Destroy;
begin
  {New - free the bookmark and its dataset if necessary.}
  if FBookmark <> nil then
    Dispose(FBookmark);
  if FBookmarkDs <> nil then
    FBookmarkDs := nil;
  FDataLink.Free;
  FDataLink := nil;
  {New - the code below is missing from the native Delphi 1.0 TDBNavigator, and
  is responsible for resource leakage.}
  FHints.Free;
  FVisibleSwitches.Free; {New}
  inherited Destroy
end; {Destroy}

function TmoDBNav.GetVisibleButtons: TmoBtnSet; {New}
begin
  Result := FVisibleButtons
end; {GetVisibleButtons}

procedure TmoDBNav.SetVisibleButtons(Value: TmoBtnSet); {New}
var
  I: TmoNavBtns;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate
end; {SetVisibleButtons}

procedure TmoDBNav.SetOrientation(Value: TmoNavOrientation); {New}
var
  W, H: Integer;
begin
  if FOrientation <> Value then begin
    FOrientation := Value;
    W := Width;
    H := Height;
    {Height & width are reversed in the call below to make the size switch with
    the orientation. Otherwise the buttons would be tall & skinny or short and
    fat when switching the orientation.}
    {If statement added below 7/25 to fix vertical orientation bug.}
    if ((W >= H) and (Value = noVertical)) or ((W <= H) and
      (Value = noHorizontal)) then
      inherited SetBounds(Left, Top, H, W)
  end
end; {SetOrientation}

procedure TmoDBNav.InitButtons;
var
  I: TmoNavBtns;
  Btn: TmoNavBtn;
  X: Integer;
  ResName: array[0..40] of Char;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do begin
    Btn := TmoNavBtn.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := (I in FEnabledButtons) and (I in FVisibleButtons) and
      (I in FUserEnabledButtons); {New}
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);

    Btn.Glyph.Handle := LoadBitmap(HInstance,
        StrFmt(ResName, 'MODBN_%s', [BtnTypeName[I]])); {New}

    Btn.NumGlyphs := 2;
    Btn.OnClick := Click;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X
  end;
  InitHints;
  Buttons[nbPrior].NavStyle    := Buttons[nbPrior].NavStyle + [nsAllowTimer];
  Buttons[nbNext].NavStyle     := Buttons[nbNext].NavStyle + [nsAllowTimer];
  Buttons[nbPriorSet].NavStyle := Buttons[nbPriorSet].NavStyle + [nsAllowTimer]; {New}
  Buttons[nbNextSet].NavStyle  := Buttons[nbNextSet].NavStyle + [nsAllowTimer] {New}
end; {InitButtons}

procedure TmoDBNav.InitHints;
var
  I: Integer;
  J: TmoNavBtns;
begin
  for J := Low(Buttons) to High(Buttons) do
    Buttons[J].Hint := LoadStr (BtnHintId[J]);
  J := Low(Buttons);
  for I := 0 to (FHints.Count - 1) do begin
    if FHints.Strings[I] <> '' then Buttons[J].Hint := FHints.Strings[I];
    if J = High(Buttons) then Exit;
    Inc(J)
  end
end; {InitHints}

procedure TmoDBNav.SetHints(Value: TStrings);
begin
  FHints.Assign(Value);
  InitHints
end; {SetHints}

procedure TmoDBNav.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil
end; {Notification}

procedure TmoDBNav.SetEnabled(Value: TmoBtnSet);
var
  i : TmoNavBtns;
begin
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Enabled := (I in FVisibleButtons) and (I in FUserEnabledButtons); {New}
  Invalidate
end; {SetEnabled}

procedure TmoDBNav.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW, MinH: Integer;
  I: TmoNavBtns;
  LastBtn: TmoNavBtns;
  Space, Temp, Remain: Integer;
  X, Y: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[nbLookuphelp] = nil then Exit;

  Count := 0;
  LastBtn := High(Buttons);
  for I := Low(Buttons) to High(Buttons) do begin
    if Buttons[I].Visible then begin
      Inc(Count);
      LastBtn := I
    end
  end;
  if Count = 0 then Inc(Count);

  if FOrientation = noHorizontal then begin
    MinW := Count * (MinBtnSize.X - 1) + 1;
    if W < MinW then
      W := MinW;
    if H < MinBtnSize.Y then
      H := MinBtnSize.Y;
    ButtonWidth := ((W - 1) div Count) + 1;
    Temp := Count * (ButtonWidth - 1) + 1;
    if Align = alNone then
      W := Temp;
    X := 0;
    Remain := W - Temp;
    Temp := Count div 2;
    for I := Low(Buttons) to High(Buttons) do begin
      if Buttons[I].Visible then begin
        Space := 0;
        if Remain <> 0 then begin
          Dec (Temp, Remain);
          if Temp < 0 then begin
            Inc (Temp, Count);
            Space := 1
          end
        end;
        Buttons[I].SetBounds (X, 0, ButtonWidth + Space, Height);
        Inc (X, ButtonWidth - 1 + Space);
        LastBtn := I
      end else
        Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
    end;
  end else begin {New - Vertical} 
    MinH := Count * (MinBtnSize.Y - 1) + 1;
    if H < MinH then
      H := MinH;
    if W < MinBtnSize.X then
      W := MinBtnSize.X;
    ButtonHeight := ((H - 1) div Count) + 1;
    Temp := Count * (ButtonHeight - 1) + 1;
    if Align = alNone then
      H := Temp;
    Y := 0;
    Remain := H - Temp;
    Temp := Count div 2;
    for I := Low(Buttons) to High(Buttons) do begin
      if Buttons[I].Visible then begin
        Space := 0;
        if Remain <> 0 then begin
          Dec (Temp, Remain);
          if Temp < 0 then begin
            Inc (Temp, Count);
            Space := 1
          end
        end;
        Buttons[I].SetBounds (0, Y, Width, ButtonHeight + Space);
        Inc (Y, ButtonHeight - 1 + Space);
        LastBtn := I
      end else
        Buttons[I].SetBounds (0, Height + 1, ButtonHeight, Width)
    end
  end {Vertical}
end; {AdjustSize}

procedure TmoDBNav.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H)
end; {SetBounds}

procedure TmoDBNav.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0
end; {WMSize}

procedure TmoDBNav.Click(Sender: TObject);
begin
  BtnClick (TmoNavBtn (Sender).Index)
end; {Click}

procedure TmoDBNav.BtnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TmoNavBtns;
  Form: TForm;
begin
  OldFocus := FocusedButton;
  FocusedButton := TmoNavBtn (Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate
  end
end; {BtnMouseDown}

procedure TmoDBNav.SetMaxSetMove(Value: Integer);
begin
  if Value <= 2 then
    raise Exception.create('MaxSetMove must be greater than 2');
  if Value < FSetMove then
    {Reduce FSetMove so that it doesn't violate the new boundary.}
    FSetMove := Value;
  FMaxSetMove := Value
end; {SetMaxSetMove}

procedure TmoDBNav.SetMoveByAmount(Value: Integer);
begin
  if (Value <= 1) or (Value > FMaxSetMove) then
    {Value may not be 1, 0, negative or greater than specified amount.}
    Raise Exception.Create('SetMove must be between 1 and ' +
      IntToStr(FMaxSetMove));
  FSetMove := Value
end; {SetMoveByAmount}

procedure TmoDBNav.BtnClick(Index: TmoNavBtns);
var
  ds: TDataSet;
  st: String;
begin
  if (DataSource <> nil) and (DataSource.State <> dsInactive) then begin
    ds := DataSource.DataSet;
    with DataSource.DataSet do begin
      case Index of
        nbLookupHelp: if not (csDesigning in ComponentState) and
          Assigned(FOnLookupHelp) then FOnLookupHelp(Self, Index);
        nbLocate: if not (csDesigning in ComponentState) and
          Assigned(FOnLocate) then FOnLocate(Self, Index);
        nbLocateNext: if not (csDesigning in ComponentState) and
          Assigned(FOnLocateNext) then FOnLocateNext(Self, Index);
        nbPriorSet: MoveBy(0-FSetMove);
        nbPrior: Prior;
        nbNext: Next;
        nbNextSet: MoveBy(FSetMove);
        nbFirst: First;
        nbLast: Last;
        nbInsert: Insert;
        nbEdit: Edit;
        nbCancel:
          if not FConfirmCancel or (MessageDlg (LoadStr(SCancelChangesQuestion),
            mtConfirmation, mbOkCancel, 0) <> idCancel) then
            Cancel;
        nbPost: Post;
        nbRefresh: Refresh;
        nbDelete:
          begin
            if not FConfirmDelete or (MessageDlg (LoadStr(SDeleteRecordQuestion),
              mtConfirmation, mbOKCancel, 0) <> idCancel) then
              Delete
          end;
        nbBkSet:
          begin
            if (FBookmark <> nil) and (FBookmarkDs <> nil) then
              {Free existing bookmark first}
              FBookmarkDs.FreeBookmark(FBookmark);
            FBookmarkDs := ds;
            FBookmark := GetBookmark;
            Buttons[nbBkGoto].Enabled := (nbBkGoto in FUserEnabledButtons);
            Buttons[nbBkClear].Enabled := (nbBkClear in FUserEnabledButtons)
          end;
        nbBkGoto:
          begin
            if (FBookmark <> nil) and (FBookmarkDs <> nil) then begin
              if (FBookmarkDs <> ds) and FShowBookmarkMessage then begin
                if FBookmarkMessage = '' then
                  st := DefaultBookmarkMessage
                else
                  st := FBookmarkMessage;
                if MessageDlg(st, mtConfirmation, [mbOk, mbCancel], 0) = mrOk then
                  FBookmarkDs.GotoBookmark(FBookmark)
              end else {Either we're on the same dataset or FShowBookmarkMessage
                is false}
                FBookmarkDs.GotoBookmark(FBookmark)
            end
          end;
        nbBkClear:
          begin
            if (FBookmark <> nil) and (FBookmarkDs <> nil) then begin
              FBookmarkDs.FreeBookmark(FBookmark);
              FBookmarkDs := nil
            end;
            Buttons[nbBkGoto].Enabled := False;
            Buttons[nbBkClear].Enabled := False
          end;
      end
    end
  end;
  if not (csDesigning in ComponentState) and Assigned(FOnNavClick) then
    FOnNavClick(Self, Index)
end; {BtnNavClick}

procedure TmoDBNav.SetButtonState(Index: TmoNavBtns; bEnable: Boolean);
begin
  Buttons[Index].Enabled := bEnable;
  if bEnable then begin
    Include(FUserEnabledButtons, Index);
    Include(FEnabledButtons, Index)
  end else begin
    Exclude(FUserEnabledButtons, Index);
    Exclude(FEnabledButtons, Index)
  end
end; {SetButtonState}

procedure TmoDBNav.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate
end; {WMSetFocus}

procedure TmoDBNav.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate
end; {WMKillFocus}

procedure TmoDBNav.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TmoNavBtns;
  OldFocus: TmoNavBtns;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate
        end
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate
        end
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click
      end
  end
end; {KeyDown}

procedure TmoDBNav.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS
end; {WMGetDlgCode}

procedure TmoDBNav.DataChanged;
var
  UpEnable, DnEnable: Boolean;
  I : TmoNavBtns;
begin
  UpEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.BOF;
  DnEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.EOF;
  Buttons[nbBkSet].Enabled := nbBkSet in FUserEnabledButtons;
  Buttons[nbBkGoto].Enabled := (FBookmark <> nil) and (nbBkGoto in FUserEnabledButtons);
  Buttons[nbBkClear].Enabled := (FBookmark <> nil) and (nbBkClear in FUserEnabledButtons);
  Buttons[nbFirst].Enabled := UpEnable and (nbFirst in FUserEnabledButtons);
  Buttons[nbPriorSet].Enabled := UpEnable and (nbPriorSet in FUserEnabledButtons);
  Buttons[nbPrior].Enabled := UpEnable and (nbPrior in FUserEnabledButtons);
  Buttons[nbNext].Enabled := DnEnable and (nbNext in FUserEnabledButtons);
  Buttons[nbNextSet].Enabled := DnEnable and (nbNextSet in FUserEnabledButtons);
  Buttons[nbLast].Enabled := DnEnable and (nbLast in FUserEnabledButtons);
  Buttons[nbDelete].Enabled := Enabled and FDataLink.Active and
    FDataLink.DataSet.CanModify and
    not (FDataLink.DataSet.BOF and FDataLink.DataSet.EOF) and
    (nbDelete in FUserEnabledButtons);

  Buttons[nbLocate].Enabled := (not (FDataLink.DataSet.BOF) or
    not (FDataLink.DataSet.EOF)) and (nbLocate in FUserEnabledButtons);
  Buttons[nbLocateNext].Enabled := Buttons[nbLocate].Enabled;
  UpdateEnabled;
  if not (csDesigning in ComponentState) and Assigned(FOnDataChanged) then
    FOnDataChanged(Self)
end; {DataChanged}

procedure TmoDBNav.UpdateEnabled;
var
  I : TmoNavBtns;
begin
  for I := Low(Buttons) to High(Buttons) do
    if Buttons[I].Enabled then
      Include(FEnabledButtons, I)
    else
      Exclude(FEnabledButtons, I)
end; {UpdateEnabled}

procedure TmoDBNav.EditingChanged;
var
  CanModify: Boolean;
  I: TmoNavBtns;
begin
  CanModify := Enabled and FDataLink.Active and FDataLink.DataSet.CanModify;
  Buttons[nbInsert].Enabled := CanModify and (nbInsert in FUserEnabledButtons);
  Buttons[nbEdit].Enabled := CanModify and not FDataLink.Editing and (nbEdit in
    FUserEnabledButtons);
  Buttons[nbPost].Enabled := CanModify and FDataLink.Editing and (nbPost in
    FUserEnabledButtons);
  Buttons[nbCancel].Enabled := CanModify and FDataLink.Editing and (nbCancel in
    FUserEnabledButtons);
  Buttons[nbRefresh].Enabled := (not (FDataLink.DataSet is TQuery)) and
    (nbRefresh in FUserEnabledButtons);
  Buttons[nbLookupHelp].Enabled := CanModify and
    (nbLookupHelp in FUserEnabledButtons);
  UpdateEnabled;
  if not (csDesigning in ComponentState) and Assigned(FOnEditingChanged) then
    FOnEditingChanged(Self)
end; {EditingChanged}

procedure TmoDBNav.ActiveChanged;
var
  I: TmoNavBtns;
begin
  if not (Enabled and FDataLink.Active) then
    for I := Low(Buttons) to High(Buttons) do
      Buttons[I].Enabled := False
  else begin
    DataChanged;
    EditingChanged
  end
end; {ActiveChanged}

procedure TmoDBNav.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged
end; {CMEnabledChanged}

procedure TmoDBNav.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if not (csLoading in ComponentState) then
    ActiveChanged
end; {SetDataSource}

function TmoDBNav.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource
end; {GetDataSource}

procedure TmoDBNav.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  InitHints;
  ActiveChanged
end; {Loaded}

{TmoNavBtn}

destructor TmoNavBtn.Destroy;
begin
  if FRepeatTimer <> nil then
    FRepeatTimer.Free;
  inherited Destroy
end; {Destroy}

procedure TmoNavBtn.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown (Button, Shift, X, Y);
  if nsAllowTimer in FNavStyle then begin
    if FRepeatTimer = nil then
      FRepeatTimer := TTimer.Create(Self);

    FRepeatTimer.OnTimer := TimerExpired;
    FRepeatTimer.Interval := InitRepeatPause;
    FRepeatTimer.Enabled  := True
  end
end; {MouseDown}

procedure TmoNavBtn.MouseUp(Button: TMouseButton; Shift: TShiftState; 
  X, Y: Integer);
begin
  inherited MouseUp (Button, Shift, X, Y);
  if FRepeatTimer <> nil then
    FRepeatTimer.Enabled  := False
end; {MouseUp}

procedure TmoNavBtn.TimerExpired(Sender: TObject);
begin
  FRepeatTimer.Interval := RepeatPause;
  if (FState = bsDown) and MouseCapture then begin
    try
      Click;
    except
      FRepeatTimer.Enabled := False;
      raise
    end
  end
end; {TimerExpired}

procedure TmoNavBtn.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
    (FIndex = TmoDBNav (Parent).FocusedButton) then begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R)
  end
end; {Paint}

{ TNavDataLink }

constructor TNavDataLink.Create(ANav: TmoDBNav);
begin
  inherited Create;
  FNavigator := ANav
end; {Create}

destructor TNavDataLink.Destroy;
begin
  FNavigator := nil;
  inherited Destroy
end; {Destroy}

procedure TNavDataLink.EditingChanged;
begin
  if FNavigator <> nil then FNavigator.EditingChanged
end; {EditingChanged}

procedure TNavDataLink.DataSetChanged;
begin
  if FNavigator <> nil then FNavigator.DataChanged
end; {DataSetChanged}

procedure TNavDataLink.ActiveChanged;
begin
  if FNavigator <> nil then FNavigator.ActiveChanged
end; {ActiveChanged}

end.
