unit Ubag;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, IniFiles;

type
  EBagError = class(Exception);

  TBag = class(TComponent)
  private
    FCount: integer;
    NList,VList: TStringList;
    FIniFile: string;
    FiniSect: string;
    FFont: TFont;
    function GetValue(AName: string): string;
    procedure SetValue(AName,AValue: string);
    procedure CheckIniParams;
    function BagStrToInt(AString: string): longint;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Clear;
    procedure DelValue(AName: string);
    procedure ReadIni;
    procedure WriteIni;
    procedure DeleteIni;
    function Count:integer;
    function GetBoolean(AName: string; ADef: Boolean): Boolean;
    procedure SetBoolean(AName: string; AValue: Boolean);
    function GetByIndex(Index:integer; var AName,AValue:string): Boolean;
    function GetColor(AName: string; ADef: TColor): TColor;
    procedure SetColor(AName: string; AValue: TColor);
    function GetFont(AName: string; ADef: TFont): TFont;
    procedure SetFont(AName: string; AValue: TFont);
    function GetInteger(AName: string; ADef: integer): integer;
    procedure SetInteger(AName: string; AValue: integer);
    function GetLongInt(AName: string; ADef: LongInt): LongInt;
    procedure SetLongInt(AName: string; AValue: LongInt);
    function GetString(AName,ADef: string): string;
    procedure SetString(AName,AValue: string);
    procedure GetFormPlace(AName: string; AForm: TForm);
    procedure SetFormPlace(AName: string; AForm: TForm);
  published
    property IniFile: string read FIniFile write FiniFile;
    property IniSect: string read FIniSect write FiniSect;
  end;

procedure Register;

implementation

const
  seps: array[1..5] of char = (',','<','>',';',':');

constructor TBag.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  NList := TStringList.Create;
  VList := TStringList.Create;
  FFont := TFont.Create;
  FCount := 0;
  Finifile := '';
  Finisect := '';
end;

destructor TBag.Destroy;
begin
  FFont.Free;
  NList.Free;
  VList.Free;
  inherited Destroy;
end;

function TBag.GetValue(AName: string): string;
var
  ix: integer;
  nam: string;
begin
  if AName = '' then raise EBagError.Create('No item specified');
  Result := '';
  if FCount = 0 then exit;
  nam := Uppercase(AName);
  ix := NList.IndexOf(nam);
  if ix >= 0 then Result := VList[ix];
end;

procedure TBag.SetValue(AName,AValue: string);
var
  ix: integer;
  nam: string;
begin
  if AName = '' then raise EBagError.Create('No item specified');
  nam := Uppercase(AName);
  ix := NList.IndexOf(nam);
  if ix < 0 then begin
    ix := NList.Add(nam);
    ix := VList.Add(AValue);
    Inc(FCount);
  end
  else VList[ix] := AValue;
end;

procedure TBag.CheckIniParams;
begin
  if FInifile = '' then raise EBagError.Create('No INI file specified');
  if FInisect = '' then raise EBagError.Create('No INI file section specified');
end;


function TBag.BagStrToInt(AString: string): longint;
begin
  try
    Result := StrToInt(AString);
  except
    ON EConvertError DO Result := 0;
  end;
end;

procedure TBag.Clear;
begin
  NList.Clear;
  VList.Clear;
  FCount := 0;
end;

procedure TBag.DelValue(AName: string);
var
  ix: integer;
  nam: string;
begin
  if AName = '' then raise EBagError.Create('No item specified');
  nam := Uppercase(AName);
  ix := NList.IndexOf(nam);
  if ix >= 0 then begin
    NList.Delete(ix);
    VList.Delete(ix);
    Dec(FCount);
  end;
end;

function TBag.Count: integer;
begin
  Result := FCount;
end;

procedure TBag.ReadIni;
var
  list1: TStringList;
  ix,p: integer;
  s1,s2: string;
begin
  CheckIniParams;
  list1 := TStringList.Create;
  with TIniFile.Create(Finifile) do try
    ReadSectionValues(Finisect,list1);
    for ix:=0 to list1.count-1 do begin
      s1 := Uppercase(list1[ix]);
      p := Pos('=',s1);
      if p > 1 then begin
        s2 := Copy(s1,p+1,255);
        s1 := Copy(s1,1,p-1);
        SetValue(s1,s2);
      end;
    end;
  finally
    list1.Free;
    Free;
  end;
  FCount := NList.Count;
end;

procedure TBag.WriteIni;
var
  ix: integer;
begin
  CheckIniParams;
  if FCount = 0 then raise EBagError.Create('No items to write to INI file');
  with TIniFile.Create(Finifile) do try
    for ix:=0 to FCount-1 do
      WriteString(Finisect,NList[ix],VList[ix]);
  finally
    Free;
  end;
end;

procedure TBag.DeleteIni;
begin
  CheckIniParams;
  with TIniFile.Create(Finifile) do try
    EraseSection(Finisect);
  finally
    Free;
  end;
end;

function TBag.GetBoolean(AName: string; ADef: Boolean): Boolean;
var s: string[2];
begin
  s := GetValue(AName);
  if s <> '' then Result := (s = '1') else Result := ADef;
end;

procedure TBag.SetBoolean(AName: string; AValue: Boolean);
var s: string[2];
begin
  if AValue then s := '1' else s := '0';
  SetValue(AName,s);
end;

function TBag.GetByIndex(Index:integer; var AName,AValue:string): Boolean;
var ix: integer;
begin
  Result := False;
  if (Index < 1) or (Index > FCount) then exit;
  ix := Index - 1;
  AName := NList[ix];
  AValue := VList[ix];
  Result := True;
end;

function TBag.GetColor(AName: string; ADef: TColor): TColor;
var s: string[99];
begin
  s := GetValue(AName);
  if s='' then Result := ADef else Result := BagStrToInt(s);
end;

procedure TBag.SetColor(AName: string; AValue: TColor);
begin
  SetValue(AName,IntToStr(AValue));
end;

function TBag.GetFont(AName: string; ADef: TFont): TFont;
var
  s1: string;
  p1: array[1..5] of byte;
  i1,fs: longint;
  bOK: boolean;
begin
  FFont.Assign(ADef);
  s1 := GetValue(AName);
  if s1 <> '' then begin
    bOK := True;
    for i1 := 1 to 5 do begin
      p1[i1] := Pos(seps[i1],s1);
      if (i1 > 1) and (p1[i1] <= p1[i1-1]) then bOK := False;
    end;
    if (not bOK) or (p1[1] < 2) or (p1[5] = Length(s1)) then
      raise EBagError.Create('Invalid font in INI file');

    with FFont do begin
      Name := Copy(s1,p1[5]+1,99);
      Color := BagStrToInt(Copy(s1,p1[4]+1,p1[5]-p1[4]-1));
      Height := BagStrToInt(Copy(s1,p1[3]+1,p1[4]-p1[3]-1));
      Size := BagStrToInt(Copy(s1,p1[2]+1,p1[3]-p1[2]-1));
      Pitch := TFontPitch(BagStrToInt(Copy(s1,p1[1]+1,p1[2]-p1[1]-1)));
      fs := BagStrToInt(Copy(s1,1,p1[1]-1));
      Style := [];
      if fs > 0 then begin
        if (fs and 1) > 0 then Style := Style + [fsBold];
        if (fs and 2) > 0 then Style := Style + [fsItalic];
        if (fs and 4) > 0 then Style := Style + [fsUnderline];
        if (fs and 8) > 0 then Style := Style + [fsStrikeOut];
      end;
    end;
  end;
  Result := FFont;
end;

procedure TBag.SetFont(AName: string; AValue: TFont);
var
  s1: string;
  fs: longint;
begin
  fs := 0;
  with AValue do begin
    if fsBold in Style then inc(fs);
    if fsItalic in Style then inc(fs,2);
    if fsUnderline in Style then inc(fs,4);
    if fsStrikeOut in Style then inc(fs,8);
    s1 := IntToStr(fs) + seps[1]
        + IntToStr(integer(Pitch)) + seps[2]
        + IntToStr(Size) + seps[3]
        + IntToStr(Height) + seps[4]
        + IntToStr(Color) + seps[5]
        + Name;
  end;
  SetValue(AName,s1);
end;

function TBag.GetInteger(AName: string; ADef: integer): integer;
var s: string[99];
begin
  s := GetValue(AName);
  if s='' then Result := ADef else Result := BagStrToInt(s);
end;

procedure TBag.SetInteger(AName: string; AValue: integer);
begin
  SetValue(AName,IntToStr(AValue));
end;

function TBag.GetLongInt(AName: string; ADef: LongInt): LongInt;
var s: string[99];
begin
  s := GetValue(AName);
  if s='' then Result := ADef else Result := BagStrToInt(s);
end;

procedure TBag.SetLongInt(AName: string; AValue: LongInt);
begin
  SetValue(AName,IntToStr(AValue));
end;

function TBag.GetString(AName,ADef: string): string;
begin
  Result := GetValue(AName);
  if Result='' then Result := ADef;
end;

procedure TBag.SetString(AName,AValue: string);
begin
  SetValue(AName,AValue);
end;


procedure TBag.GetFormPlace(AName: string; AForm: TForm);
var
  s: string[99];
  p,ix: byte;
  Place: TWindowPlacement;
  ar:  array[0..9] of LongInt;
begin
  s := GetValue(AName)+',';
  for ix := 0 to 9 do begin
    p := Pos(',',s);
    if p < 2 then exit;  {invalid data found for placement}
    ar[ix] := StrToInt(Copy(s,1,p-1));
    s := Copy(s,p+1,99);
  end;

  with Place do begin
    Length :=SizeOf(TWindowPlacement);
    Flags :=ar[0];
    ShowCmd := ar[1];
    ptMinPosition.X := ar[2];
    ptMinPosition.Y := ar[3];
    ptMaxPosition.X := ar[4];
    ptMaxPosition.Y := ar[5];
    rcNormalPosition.Left := ar[6];
    rcNormalPosition.Top := ar[7];
    rcNormalPosition.Right := ar[8];
    rcNormalPosition.Bottom := ar[9];
    if rcNormalPosition.Right > rcNormalPosition.Left then
      SetWindowPlacement(AForm.Handle,@Place)
   end;
 end;

function AppendS(const s:string; const num:longint): string;
begin
  Result := s+','+IntToStr(num);
end;

procedure TBag.SetFormPlace(AName: string; AForm: TForm);
var
  s: string[99];
  Place : TWindowPlacement;
begin
  Place.length :=SizeOf(TWindowPlacement);
  if not GetWindowPlacement(AForm.Handle,@Place) then exit;
  with Place do begin
    s := IntToStr(Flags);
    s := AppendS(s,ShowCmd);
    s := AppendS(s,ptMinPosition.X);
    s := AppendS(s,ptMinPosition.Y);
    s := AppendS(s,ptMaxPosition.X);
    s := AppendS(s,ptMaxPosition.Y);
    s := AppendS(s,rcNormalPosition.Left);
    s := AppendS(s,rcNormalPosition.Top);
    s := AppendS(s,rcNormalPosition.Right);
    s := AppendS(s,rcNormalPosition.Bottom);
  end;
  SetString(AName,s);
end;

procedure Register;
begin
  RegisterComponents('3K', [TBag]);
end;

end.