{-------------------------------------------------------------------------}
{UNIT:    SpinTime.Pas (v1.0) - A spin edit component for adding and      }
{                               subtracting time increments.              }
{TARGET:  Delphi 1 only.                                                  }
{AUTHOR:  George L. Roberts                                               }
{EMAIL:   robertsg@nettally.com                                           }
{HISTORY: February 1998 - Changed name from SpinTime to gSpinTime and     }
{                         added to Suite16.
{-------------------------------------------------------------------------}
{WARNING! This software is provided as is.  No warranty is given by the   }
{         author, expressed or implied.  Use this software at your own    }
{         risk.  The author assumes no responsibility for any damage from }
{         the use of this software.                                       }
{NOTE:    This software is freeware which means that it is free for use   }
{         and distribution.  Please do not remove/mask the about property.}
{-------------------------------------------------------------------------}
unit SpinTime;

interface

uses
  Classes, Controls, Dialogs, DsgnIntf, Graphics, IniFiles, Mask, Messages,
  Menus, Spin, StdCtrls, Suite16, SysUtils, WinProcs, WinTypes;

const
  __OBJNAME:  String  = 'gSpinTime';
  __OBJVER:   String  = 'v1.1';

type
  TTimeFormat    = (tfTwelveHour, tfTwentyFourHour);
  TAboutSpinTime = class(TgPropertyEditor)
    procedure Edit; override;
  end;

  TgSpinTime 	= class(TCustomMaskEdit)              				{ TgSpinTime }
  private
    FAboutBox:      TAboutSpinTime;
    FButton:        TSpinButton;
    FCanvas:        TCanvas;
    FEditorEnabled: Boolean;
    FFormatString:  String;
    FMinValue:      LongInt;
    FMaxValue:      LongInt;
    FMinIncrement:  Integer;
    FSeperator:     String[1];
    FTimeFormat:    TTimeFormat;
    procedure SetTime;
    procedure SetFormat( Format: TTimeFormat );
    function  GetMinHeight: Integer;
    procedure SetEditRect;
  protected
    procedure DoExit; override;
    procedure UpClick (Sender: TObject); virtual;
    procedure DownClick (Sender: TObject); virtual;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure CMEnter(var Message: TCMGotFocus); message CM_ENTER;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMPaste(var Message: TWMPaste);   message WM_PASTE;
    procedure WMCut(var Message: TWMCut);   message WM_CUT;
  public
    FTime:      TDateTime;
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    property    Button: TSpinButton read FButton;
    property    EditMask;                            {Moved from published}
    property    MaxLength;
  published
    property About: TAboutSpinTime read FAboutBox write FAboutBox;
    property AutoSelect;
    property AutoSize;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property EditorEnabled: Boolean read FEditorEnabled write FEditorEnabled default True;
    property Enabled;
    property Font;
    property MinuteIncrement: Integer read FMinIncrement write FMinIncrement;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property TimeFormat: TTimeFormat read FTimeFormat write SetFormat;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

  procedure register;

implementation

{$R SPINTIME}

constructor TgSpinTime.Create(AOwner: TComponent);
var iniWin: TIniFile;
begin
  inherited Create(AOwner);
  FButton               := TSpinButton.Create (Self);
  FButton.Width         := 15;                  		 {Initialize variables}
	FButton.Height        := 17;
  FButton.Visible       := True;
  FButton.Parent        := Self;
  FButton.FocusControl  := Self;
  FButton.OnUpClick     := UpClick;
  FButton.OnDownClick   := DownClick;
  ControlStyle          := ControlStyle - [csSetCaption];
  FEditorEnabled        := True;
  FTime                 := 0;

  { Get international time seperator }
  iniWin := TIniFile.Create( 'WIN.INI' );
  FSeperator := iniWin.ReadString( 'intl', 'sTime', ':' );
  iniWin.Free;

  SetFormat( FTimeFormat );
end;

destructor TgSpinTime.Destroy;
begin
  FButton := nil;
  inherited Destroy;
end;

{-------------------------------------------------------------------------}
{PROCEDURE:	Set international time format.                                }
{-------------------------------------------------------------------------}
procedure TgSpinTime.SetFormat( Format: TTimeFormat );
begin
  if (Format = tfTwelveHour) then
  begin
    FTimeFormat           := tfTwelveHour;
    EditMask              := '!90:00 <LL;1;_';
    FFormatString         := 'hh'+FSeperator+'nn am/pm';
  end
  else
  begin
    FTimeFormat           := tfTwentyFourHour;
    EditMask              := '!90:00;1;_';
    FFormatString         := 'hh'+FSeperator+'nn';
  end;
end;

{-------------------------------------------------------------------------}
{PROCEDURE:	Set time on loss of focus event.                              }
{-------------------------------------------------------------------------}
procedure TgSpinTime.DoExit;
begin
  inherited DoExit;
  SetTime;
end;

{-------------------------------------------------------------------------}
{PROCEDURE:	Attempt to store new time.                           					}
{-------------------------------------------------------------------------}
procedure TgSpinTime.SetTime;
var wrdHr, wrdMin: 	Word;
    strTest:       	String;
begin
  try
    wrdHr   := StrToInt( Copy( Text, 1, 2 ) );
    wrdMin  := StrToInt( Copy( Text, 4, 2 ) );
    if (FTimeFormat = tfTwelveHour) then      {Adjust hours to 24hr clock}
    begin
      if (UpperCase( Copy( Text, 7, 2 ) ) = 'PM')       and
         (wrdHr <> 12)                                  then
        wrdHr := wrdHr + 12
      else if (UpperCase( Copy( Text, 7, 2 ) ) = 'AM')  and
         (wrdHr = 12)                                   then
         wrdHr := 0;
    end;
    FTime 	:= EncodeTime( wrdHr, wrdMin, 0, 0 );
    strTest := FormatDateTime( FFormatString, ftime );
  except
    if (Length( Text ) <> 0) then
    begin
      MessageDlg( 'Please input an acceptable time.', mtWarning, [mbOk], 0 );
      SetFocus;
    end
    else
      FTime := 0;
  end;
end;

{-------------------------------------------------------------------------}
{PROCEDURE:	Increment time by MinuteIncrement.														}
{-------------------------------------------------------------------------}
procedure TgSpinTime.UpClick (Sender: TObject);
var wrdHr, wrdMin, wrdBogus: Word;
begin
  SetTime;
  if ReadOnly then MessageBeep(0)
  else
  begin
    DecodeTime( FTime, wrdHr, wrdMin, wrdBogus, wrdBogus );
    if (FMinIncrement <> 0) then
    begin
      wrdHr   := wrdHr  + (FMinIncrement div 60);
      wrdMin  := wrdMin + (FMinIncrement mod 60);
      while (wrdMin > 59) do
      begin
        Inc( wrdHr );
        wrdMin := wrdMin - 60;
      end;
      while (wrdHr > 23) do
        wrdHr := wrdHr - 24;
    end;
    FTime := EncodeTime( wrdHr, wrdMin, 0, 0 );
  end;
  Text := FormatDateTime( FFormatString, FTime );
end;

{-------------------------------------------------------------------------}
{PROCEDURE:	Decrement time by MinuteIncrement.														}
{-------------------------------------------------------------------------}
procedure TgSpinTime.DownClick (Sender: TObject);
var intHr, intMin, intBogus: Integer;
begin
  SetTime;
  if ReadOnly then MessageBeep(0)
  else
  begin
    DecodeTime( FTime, Word( intHr ), Word( intMin ), Word( intBogus ), Word( intBogus ) );
    if (FMinIncrement <> 0) then
    begin
      intHr   := intHr  - (FMinIncrement div 60);
      intMin  := intMin - (FMinIncrement mod 60);
      while (intMin < 0) do
      begin
        Dec( intHr );
        intMin  := 60 + intMin;
      end;
      while (intHr < 0) do
        intHr   := 24 + intHr;
    end;
    FTime := EncodeTime( intHr, intMin, 0, 0 );
  end;
  Text := FormatDateTime( FFormatString, FTime );
end;

{-------------------------------------------------------------------------}
{PROCEDURE:	If up/down arrow increment/decrement time.                    }
{-------------------------------------------------------------------------}
procedure TgSpinTime.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if      Key = VK_UP   then UpClick (Self)
  else if Key = VK_DOWN then DownClick (Self);
  inherited KeyDown(Key, Shift);
end;


{NOTE: Delphi standard edit procedures.-----------------------------------}
procedure TgSpinTime.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
{  Params.Style := Params.Style and not WS_BORDER;  }
  Params.Style := Params.Style or ES_MULTILINE or WS_CLIPCHILDREN;
end;
procedure TgSpinTime.CreateWnd;
var
  Loc: TRect;
begin
  inherited CreateWnd;
  SetEditRect;
end;
procedure TgSpinTime.SetEditRect;
var
  Loc: TRect;
begin
  SendMessage(Handle, EM_GETRECT, 0, LongInt(@Loc));
  Loc.Bottom := ClientHeight + 1;  {+1 is workaround for windows paint bug}
  Loc.Right := ClientWidth - FButton.Width - 2;
  Loc.Top := 0;
  Loc.Left := 0;
  SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@Loc));
  SendMessage(Handle, EM_GETRECT, 0, LongInt(@Loc));  {debug}
end;
procedure TgSpinTime.WMSize(var Message: TWMSize);
var
  Loc: TRect;
  MinHeight: Integer;
begin
  inherited;
  MinHeight := GetMinHeight;
    { text edit bug: if size to less than minheight, then edit ctrl does
      not display the text }
  if Height < MinHeight then
    Height := MinHeight
  else if FButton <> nil then
  begin
    FButton.SetBounds (Width - FButton.Width, 0, FButton.Width, Height);
    SetEditRect;
  end;
end;
function TgSpinTime.GetMinHeight: Integer;
var
  DC:       HDC;
  SaveFont: HFont;
  I:        Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  DC := GetDC(0);
  GetTextMetrics(DC, SysMetrics);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  I := SysMetrics.tmHeight;
  if I > Metrics.tmHeight then I := Metrics.tmHeight;
  Result := Metrics.tmHeight + I div 4 + GetSystemMetrics(SM_CYBORDER) * 4 + 2;
end;
procedure TgSpinTime.WMPaste(var Message: TWMPaste);
begin
  if not FEditorEnabled or ReadOnly then Exit;
  inherited;
end;
procedure TgSpinTime.WMCut(var Message: TWMPaste);
begin
  if not FEditorEnabled or ReadOnly then Exit;
  inherited;
end;
procedure TgSpinTime.CMEnter(var Message: TCMGotFocus);
begin
  if AutoSelect and not (csLButtonDown in ControlState) then
    SelectAll;
  inherited;
end;
{NOTE: Delphi standard edit procedures.-----------------------------------}

procedure TAboutSpinTime.Edit;
begin
  DisplayAbout( __OBJNAME, __OBJVER, __ME, __ADDRESS );
end;

procedure Register;
begin
  RegisterComponents( 'Suite16', [TgSpinTime] );
  RegisterPropertyEditor( TypeInfo(TAboutSpinTime), TgSpinTime, 'ABOUT', TAboutSpinTime );
end;

end.

