{-------------------------------------------------------------------------}
{UNIT:    OraDb.Pas - Database component written specifically for         }
{                     accessing Oracle databases.  Features include login }
{                     screen, change password screen, and default driver  }
{                     specifications.  Login screen automaticly sets user }
{											password.																						}
{TARGET:  Delphi 1 only.                                                  }
{AUTHOR:  George L. Roberts                                               }
{EMAIL:   robertsg@nettally.com                                           }
{-------------------------------------------------------------------------}
{WARNING! This software is provided as is.  No warranty is given by the   }
{         author, expressed or implied.  Use this software at your own    }
{         risk.  The author assumes no responsibility for any damage from }
{         the use of this software.                                       }
{NOTE:    This software is freeware which means that it is free for use   }
{         and distribution.  Please do not remove/mask the about property.}
{-------------------------------------------------------------------------}
unit OraDb;

interface

uses
	Classes, Db, DbTables, Dialogs, DsgnIntf, PassDlg, Suite16;

const
  __OBJNAME:  String = 'gOraDb';
  __OBJVER:   String = 'v1.0';

type
  TAboutOraDb = class(TgPropertyEditor)
    procedure Edit; override;
  end;

  TgOraDb = class( TComponent ) 																{ TgOraDb }
  private
    FAboutBox:  TAboutOraDb;
    FDatabase:  TDatabase;
    FParams:    TStrings;
    FDbName,
    FUser,
    FPassword,
    FInstance:  String;
    FConnected: Boolean;
    function    SetActiveState( Active: Boolean ): Boolean;
    procedure   SetDbName( Value: String );
    procedure   SetInstance( Value: String );
		procedure   SetParams( Value: TStrings );
    function    CheckForConnect: Boolean;
  public
    constructor Create( AOwner: TComponent ); override;
    destructor  Destroy; override;
    function    Login( strDefaultUser: String ): Boolean;
    function    ChangePassword: Boolean;
    procedure   Open;
    procedure   Close;
  published
    property    About: TAboutOraDb  read FAboutBox write FAboutBox;
    property    Connected: Boolean  read FConnected write SetActiveState;
    property    DbName:    String   read FDbName    write SetDbName;
    property    Instance:  String   read FInstance  write SetInstance;
    property    Password:  String   read FPassword  write FPassword;
    property    Params:    TStrings read FParams    write SetParams;
    property    User:      String   read FUser      write FUser;
  end;

  procedure Register;

implementation

{$R ORADB}

constructor TgOraDb.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FDatabase       := TDatabase.Create( Self );
  FParams         := TStringList.Create;

  FDatabase.DriverName    := 'ORACLE';
  FDatabase.LoginPrompt   := False;
  FParams.Add( 'SQLPASSTHRU MODE=SHARED NOAUTOCOMMIT'   );
  FParams.Add( 'OPEN MODE=READ/WRITE'                   );
  FParams.Add( 'SCHEMA CACHE SIZE=8'                    );
  FParams.Add( 'SCHEMA CACHE TIME=-1'                   );
  FParams.Add( 'LOCAL SHARE=FALSE'                      );
  FParams.Add( 'NET PROTOCOL=TNS'                       );
  FParams.Add( 'LANGDRIVER='                            );
  FParams.Add( 'SQLQRYMODE='                            );
  if (FDbName   = '') then
  begin
    FDbName                 := 'OraServer';
    FDatabase.DatabaseName  := 'OraServer';
  end;

end;

destructor TgOraDb.Destroy;
begin
  FDatabase.Free;
  FParams.Free;
  inherited Destroy;
end;

procedure TgOraDb.Open;
begin
  SetActiveState( True );
end;
procedure TgOraDb.Close;
begin
  SetActiveState( False );
end;
procedure TgOraDb.SetDbName( Value: String );
begin
  if not CheckForConnect then FDbName := Value;
end;
procedure TgOraDb.SetInstance( Value: String );
begin
  if not CheckForConnect then FInstance := Value;
end;
procedure TgOraDb.SetParams( Value: TStrings );
begin
	FParams.Assign( Value );
end;
function TgOraDb.CheckForConnect: Boolean;
begin
  result := FConnected;
  if FConnected then
    MessageDlg( 'Please disconnect from database before changing this setting.',
      mtWarning, [mbOk], 0 );
end;

{-------------------------------------------------------------------------}
{FUNCTION:	Connect/Disconnect database.                                  }
{PARAMETER:	Active - requested connection state.													}
{RETURNS:		Successful/Unsuccessful connection status.                    }
{-------------------------------------------------------------------------}
function TgOraDb.SetActiveState( Active: Boolean ): Boolean;
begin
    result := False;

    if  (FUser <> '') and (FPassword <> '') then
    begin
      FDatabase.Close;

      if Active then
      begin
        FDatabase.Params.Clear;
        FDatabase.Params.Assign( FParams );
        FDatabase.Params.Add( 'USER NAME='+FUser        );
        FDatabase.Params.Add( 'PASSWORD='+FPassword     );
        FDatabase.Params.Add( 'SERVER NAME='+FInstance  );
        FDatabase.DatabaseName  := FDbName;
        if (not (csReading in ComponentState)) then
          FDatabase.Open;
      end;
      FConnected := Active;
    end
    else
    begin
      if (not (csLoading in ComponentState)) then
        MessageDlg( 'User name and password are required to login to Oracle databases.',
          mtWarning, [mbOk], 0 );
    end;
    result := FDatabase.Connected;
end;

{-------------------------------------------------------------------------}
{FUNCTION:	Display login screen and attempt database connection.         }
{PARAMETER:	strDefaultUser - Default user name to appear in login screen. }
{RETURNS:		Successful/Unsuccessful connection status.                    }
{-------------------------------------------------------------------------}
function TgOraDb.Login( strDefaultUser: String ): Boolean;
var dlgLogin: TgPassDlg;
begin
  result 								:= False;
  dlgLogin              := TgPassDlg.Create( Self );
  dlgLogin.PasswEncrypt := dmSecond;
  dlgLogin.PasswordChar := '*';

  result := dlgLogin.Execute( strDefaultUser );
  if result then
  begin
    FUser     := dlgLogin.User;
    FPassword := dlgLogin.Password;
    result    := SetActiveState( True );
  end;

  dlgLogin.Free;
end;

{-------------------------------------------------------------------------}
{FUNCTION:	Get new Oracle password & execute DML to change user password.}
{RETURNS:		Successful/Unsuccessful password change.                      }
{-------------------------------------------------------------------------}
function TgOraDb.ChangePassword: Boolean;
var dlgLogin:         TgPassDlg;
    qryPassChg:       TQuery;
begin

  result := False;
  if FConnected then
  begin
    dlgLogin              := TgPassDlg.Create( Self );
    dlgLogin.FormCaption  := 'Change Oracle Password';
    dlgLogin.Label1       := 'New Password:';
    dlgLogin.Label2       := 'Confirm:';
    dlgLogin.PasswordChar := '*';
    dlgLogin.PasswEncrypt := dmBoth;

    if dlgLogin.Execute( '' )                                   and
      (Length( dlgLogin.User )+Length( dlgLogin.Password ) > 0) and
      (dlgLogin.User = dlgLogin.Password)                       then
    begin
      result                  := True;
      Password                := dlgLogin.Password;
      qryPassChg              := TQuery.Create( Self );
      qryPassChg.DatabaseName := FDbName;
      qryPassChg.SQL.Add( 'ALTER USER '+FUser );
      qryPassChg.SQL.Add( 'IDENTIFIED BY '+dlgLogin.Password );
      qryPassChg.ExecSQL;
      qryPassChg.Free;
    end                                      {End new password accepted.}
    else
      MessageDlg( 'Password not accepted.', mtWarning, [mbOk], 0 );
  end
  else
    MessageDlg( 'Database not currently connected.', mtWarning, [mbOk], 0 );

end;

procedure TAboutOraDb.Edit;
begin
  DisplayAbout( __OBJNAME, __OBJVER, __ME, __ADDRESS );
end;

procedure Register;
begin
  RegisterComponents( 'Suite16', [TgOraDb] );
  RegisterPropertyEditor( TypeInfo(TAboutOraDb), TgOraDb, 'ABOUT', TAboutOraDb );
end;

end.
