{-------------------------------------------------------------------------}
{UNIT:    Calendar.Pas - Adds navigation bar to Delphi calendar sample.   }
{TARGET:  Delphi 1 only.                                                  }
{AUTHOR:  George L. Roberts                                               }
{EMAIL:   robertsg@nettally.com                                           }
{-------------------------------------------------------------------------}
{WARNING! This software is provided as is.  No warranty is given by the   }
{         author, expressed or implied.  Use this software at your own    }
{         risk.  The author assumes no responsibility for any damage from }
{         the use of this software.                                       }
{NOTE:    This software is freeware which means that it is free for use   }
{         and distribution.  Please do not remove/mask the about property.}
{-------------------------------------------------------------------------}
unit Cal;

interface

uses
  Buttons, Classes, Controls, DsgnIntf, ExtCtrls, Graphics, Grids,
  Forms, Messages, StdCtrls, SysUtils, Suite16, WinProcs, WinTypes;

const
  __OBJNAME:  String = 'gCalendar';
  __OBJVER:   String = 'v1.0';

type
  TDayOfWeek = 0..6;
  TBtnImage = (biFBck, biBck, biToday, biFFwd, biFwd);
  TAboutCalendar = class(TgPropertyEditor)
    procedure Edit; override;
  end;

  TgCalendar = class(TCustomGrid)
  private
    FAboutBox:        TAboutCalendar;
    FFBckBtn,  FBckBtn,
    FTodayBtn, FFwdBtn,
    FFFwdBtn:         TSpeedButton;
    FDate:            TDateTime;
    FMonthOffset:     Integer;
    FOnChange:        TNotifyEvent;
    FReadOnly:        Boolean;
    FStartOfWeek:     TDayOfWeek;
    FUpdating:        Boolean;
    FUseCurrentDate:  Boolean;
    function  GetCellText(ACol, ARow: Integer): string;
    function  GetDateElement(Index: Integer): Integer;
    procedure SeTgCalDate(Value: TDateTime);
    procedure SetDateElement(Index: Integer; Value: Integer);
    procedure SetStartOfWeek(Value: TDayOfWeek);
    function  StoreCalendarDate: Boolean;
    procedure SetButtonImage( var sbRaw: TSpeedButton; ImgSpec: TBtnImage );
    procedure AdjustButtonSize;
  protected
    procedure Change; dynamic;
    procedure ChangeMonth(Delta: Integer);
    procedure FBckClick (Sender: TObject); virtual;
    procedure BckClick (Sender: TObject); virtual;
    procedure TodayClick (Sender: TObject); virtual;
    procedure FwdClick (Sender: TObject); virtual;
    procedure FFwdClick (Sender: TObject); virtual;
    procedure Click; override;
    function  DaysPerMonth(AYear, AMonth: Integer): Integer; virtual;
    function  DaysThisMonth: Integer; virtual;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    function  IsLeapYear(AYear: Integer): Boolean; virtual;
    function  SelectCell(ACol, ARow: Longint): Boolean; override;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    property CalendarDate: TDateTime  read FDate write SeTgCalDate stored StoreCalendarDate;
    property CellText[ACol, ARow: Integer]: string read GetCellText;
    procedure NextMonth;
    procedure NextYear;
    procedure PrevMonth;
    procedure PrevYear;
    procedure UpdateCalendar; virtual;
  published
    property About: TAboutCalendar read FAboutBox      write FAboutBox;
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property Day: Integer index 3  read GetDateElement write SetDateElement stored False;
    property Enabled;
    property Font;
    property GridLineWidth;
    property Month: Integer index 2  read GetDateElement write SetDateElement stored False;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property ReadOnly: Boolean read FReadOnly write FReadOnly default False;
    property ShowHint;
    property StartOfWeek: TDayOfWeek read FStartOfWeek write SetStartOfWeek;
    property TabOrder;
    property TabStop;
    property UseCurrentDate: Boolean read FUseCurrentDate write FUseCurrentDate default True;
    property Visible;
    property Year: Integer index 1  read GetDateElement write SetDateElement stored False;
    property OnClick;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
  end;

  procedure register;

implementation

{$R CAL}

constructor TgCalendar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFBckBtn    := TSpeedButton.Create( Self );
  FBckBtn     := TSpeedButton.Create( Self );
  FTodayBtn   := TSpeedButton.Create( Self );
  FFwdBtn     := TSpeedButton.Create( Self );
  FFFwdBtn    := TSpeedButton.Create( Self );

  FFBckBtn.Parent     := Self;
  FFBckBtn.Visible    := True;
  FFBckBtn.OnClick    := FBckClick;
  FFBckBtn.NumGlyphs  := 1;
  SetButtonImage( FFBckBtn, biFBck );

  FBckBtn.Parent      := Self;
  FBckBtn.Visible     := True;
  FBckBtn.OnClick     := BckClick;
  FBckBtn.NumGlyphs   := 1;
  SetButtonImage( FBckBtn, biBck );

  FTodayBtn.Parent    := Self;
  FTodayBtn.Visible   := True;
  FTodayBtn.OnClick   := TodayClick;
  FTodayBtn.NumGlyphs := 1;
  SetButtonImage( FTodayBtn, biToday );

  FFwdBtn.Parent      := Self;
  FFwdBtn.Visible     := True;
  FFwdBtn.OnClick     := FwdClick;
  FFwdBtn.NumGlyphs   := 1;
  SetButtonImage( FFwdBtn, biFwd );

  FFFwdBtn.Parent     := Self;
  FFFwdBtn.Visible    := True;
  FFFwdBtn.OnClick    := FFwdClick;
  FFFwdBtn.NumGlyphs  := 1;
  SetButtonImage( FFFwdBtn, biFFwd );

  Width           := 150;                  { Initialize calendar defaults }
  Height          := 110;
  FUseCurrentDate := True;
  FixedCols       := 0;
  FixedRows       := 1;
  ColCount        := 7;
  RowCount        := 8;
  ScrollBars      := ssNone;
  Options         := Options - [goRangeSelect] + [goDrawFocusSelected];
  FDate           := Date;
  UpdateCalendar;
  AdjustButtonSize;
end;

destructor TgCalendar.Destroy;
begin
  FFBckBtn.Free;
  FBckBtn.Free;
  FTodayBtn.Free;
  FFwdBtn.Free;
  FFFwdBtn.Free;
  inherited Destroy;
end;

procedure TgCalendar.FBckClick (Sender: TObject);
begin
  PrevYear;
end;
procedure TgCalendar.BckClick (Sender: TObject);
begin
  PrevMonth;
end;
procedure TgCalendar.TodayClick (Sender: TObject);
begin
  CalendarDate := Now;
end;
procedure TgCalendar.FwdClick (Sender: TObject);
begin
  NextMonth;
end;
procedure TgCalendar.FFwdClick (Sender: TObject);
begin
  NextYear;
end;

procedure TgCalendar.Change;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TgCalendar.Click;
var
  TheCellText: string;
begin
  inherited Click;
  TheCellText := CellText[Col, Row];
  if TheCellText <> '' then Day := StrToInt(TheCellText);
end;

function TgCalendar.IsLeapYear(AYear: Integer): Boolean;
begin
  Result := (AYear mod 4 = 0) and ((AYear mod 100 <> 0) or (AYear mod 400 = 0));
end;

function TgCalendar.DaysPerMonth(AYear, AMonth: Integer): Integer;
const
  DaysInMonth: array[1..12] of Integer = (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
begin
  Result := DaysInMonth[AMonth];
  if (AMonth = 2) and IsLeapYear(AYear) then Inc(Result); { leap-year Feb is special }
end;

function TgCalendar.DaysThisMonth: Integer;
begin
  Result := DaysPerMonth(Year, Month);
end;

procedure TgCalendar.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
var TheText:  string;
    OldColor: TColor;
begin
{  OldColor := Canvas.Brush.Color;
  if ARow = (RowCount-1) then
  begin
    Canvas.Brush.Color := clBtnFace;
    Canvas.FillRect( ARect );
  end;}

  TheText := CellText[ACol, ARow];
  with ARect, Canvas do
    TextRect(ARect, Left + (Right - Left - TextWidth(TheText)) div 2,
      Top + (Bottom - Top - TextHeight(TheText)) div 2, TheText);

  if (FFBckBtn.Height <> DefaultRowHeight) then AdjustButtonSize;
end;

function TgCalendar.GetCellText(ACol, ARow: Integer): string;
var
  DayNum: Integer;
begin
  if ARow = 0 then  { day names at tops of columns }
    Result := ShortDayNames[(StartOfWeek + ACol) mod 7 + 1]
  else
  begin
    DayNum := FMonthOffset + ACol + (ARow - 1) * 7;
    if (DayNum < 1) or (DayNum > DaysThisMonth) then Result := ''
    else Result := IntToStr(DayNum);
  end;
end;

function TgCalendar.SelectCell(ACol, ARow: Longint): Boolean;
begin
  if ((not FUpdating) and FReadOnly) or (CellText[ACol, ARow] = '') then
    Result := False
  else Result := inherited SelectCell(ACol, ARow);
end;

procedure TgCalendar.SeTgCalDate(Value: TDateTime);
begin
  FDate := Value;
  UpdateCalendar;
  Change;
end;

function TgCalendar.StoreCalendarDate: Boolean;
begin
  Result := not FUseCurrentDate;
end;

function TgCalendar.GetDateElement(Index: Integer): Integer;
var
  AYear, AMonth, ADay: Word;
begin
  DecodeDate(FDate, AYear, AMonth, ADay);
  case Index of
    1: Result := AYear;
    2: Result := AMonth;
    3: Result := ADay;
    else Result := -1;
  end;
end;

procedure TgCalendar.SetDateElement(Index: Integer; Value: Integer);
var
  AYear, AMonth, ADay: Word;
begin
  if Value > 0 then
  begin
    DecodeDate(FDate, AYear, AMonth, ADay);
    case Index of
      1: if AYear <> Value then AYear := Value else Exit;
      2: if (Value <= 12) and (Value <> AMonth) then AMonth := Value else Exit;
      3: if (Value <= DaysThisMonth) and (Value <> ADay) then ADay := Value else Exit;
      else Exit;
    end;
    FDate := EncodeDate(AYear, AMonth, ADay);
    FUseCurrentDate := False;
    UpdateCalendar;
    Change;
  end;
end;

procedure TgCalendar.SetStartOfWeek(Value: TDayOfWeek);
begin
  if Value <> FStartOfWeek then
  begin
    FStartOfWeek := Value;
    UpdateCalendar;
  end;
end;

{ Given a value of 1 or -1, moves to Next or Prev month accordingly }
procedure TgCalendar.ChangeMonth(Delta: Integer);
var
  AYear, AMonth, ADay: Word;
  NewDate: TDateTime;
  CurDay: Integer;
begin
  DecodeDate(FDate, AYear, AMonth, ADay);
  CurDay := ADay;
  if Delta > 0 then ADay := DaysPerMonth(AYear, AMonth)
  else ADay := 1;
  NewDate := EncodeDate(AYear, AMonth, ADay);
  NewDate := NewDate + Delta;
  DecodeDate(NewDate, AYear, AMonth, ADay);
  if DaysPerMonth(AYear, AMonth) > CurDay then ADay := CurDay
  else ADay := DaysPerMonth(AYear, AMonth);
  CalendarDate := EncodeDate(AYear, AMonth, ADay);
end;

procedure TgCalendar.PrevMonth;
begin
  ChangeMonth(-1);
end;

procedure TgCalendar.NextMonth;
begin
  ChangeMonth(1);
end;

procedure TgCalendar.NextYear;
begin
  if IsLeapYear(Year) and (Month = 2) and (Day = 29) then Day := 28;
  Year := Year + 1;
end;

procedure TgCalendar.PrevYear;
begin
  if IsLeapYear(Year) and (Month = 2) and (Day = 29) then Day := 28;
  Year := Year - 1;
end;

procedure TgCalendar.UpdateCalendar;
var
  AYear, AMonth, ADay: Word;
  FirstDate: TDateTime;
begin
  FUpdating := True;
  try
    DecodeDate(FDate, AYear, AMonth, ADay);
    FirstDate := EncodeDate(AYear, AMonth, 1);
    FMonthOffset := 2 - ((DayOfWeek(FirstDate) - StartOfWeek + 7) mod 7); { day of week for 1st of month }
    if FMonthOffset = 2 then FMonthOffset := -5;
    MoveColRow((ADay - FMonthOffset) mod 7, (ADay - FMonthOffset) div 7 + 1,
      False, False);
    Invalidate;
  finally
    FUpdating := False;
  end;
end;

procedure TgCalendar.WMSize(var Message: TWMSize);
var GridLines: Integer;
begin
  GridLines         := 6 * GridLineWidth;
  DefaultColWidth   := (Message.Width - GridLines) div 7;
  DefaultRowHeight  := (Message.Height - GridLines) div 8;
  AdjustButtonSize;
end;

procedure TgCalendar.SetButtonImage( var sbRaw: TSpeedButton; ImgSpec: TBtnImage );
begin
  case ImgSpec of
    biFBck:   sbRaw.Glyph.Handle := LoadBitmap(hInstance, 'GCAL_FRWD' );
    biBck:    sbRaw.Glyph.Handle := LoadBitmap(hInstance, 'GCAL_RWD'  );
    biToday:  sbRaw.Glyph.Handle := LoadBitmap(hInstance, 'GCAL_TODAY');
    biFwd:    sbRaw.Glyph.Handle := LoadBitmap(hInstance, 'GCAL_FWD'  );
    biFFwd:   sbRaw.Glyph.Handle := LoadBitmap(hInstance, 'GCAL_FFWD' );
  end;
end;

procedure TgCalendar.AdjustButtonSize;
var intBtnTop,
    intBtnWidth,
    intBtnHeight:  Integer;
begin
  intBtnTop        := Height-DefaultRowHeight-(GridLineWidth*7);
  intBtnWidth      := Width div 5;
  intBtnHeight     := Height-intBtnTop;

  FFBckBtn.Top     := intBtnTop;
  FFBckBtn.Left    := 0;
  FFBckBtn.Height  := intBtnHeight;
  FFBckBtn.Width   := intBtnWidth;

  FBckBtn.Top      := intBtnTop;
  FBckBtn.Left     := intBtnWidth;
  FBckBtn.Height   := intBtnHeight;
  FBckBtn.Width    := intBtnWidth;

  FTodayBtn.Top    := intBtnTop;
  FTodayBtn.Left   := intBtnWidth*2;
  FTodayBtn.Height := intBtnHeight;
  FTodayBtn.Width  := intBtnWidth;

  FFwdBtn.Top      := intBtnTop;
  FFwdBtn.Left     := intBtnWidth*3;
  FFwdBtn.Height   := intBtnHeight;
  FFwdBtn.Width    := intBtnWidth;

  FFFwdBtn.Top     := intBtnTop;
  FFFwdBtn.Left    := intBtnWidth*4;
  FFFwdBtn.Height  := intBtnHeight;
  FFFwdBtn.Width   := intBtnWidth;
end;

procedure TAboutCalendar.Edit;
begin
  DisplayAbout( __OBJNAME, __OBJVER, __ME, __ADDRESS );
end;

procedure Register;
begin
  RegisterComponents( 'Suite16', [TgCalendar] );
  RegisterPropertyEditor( TypeInfo(TAboutCalendar), TgCalendar, 'ABOUT', TAboutCalendar );
end;

end.
