unit subclass;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TSubclassEvent = procedure (Sender: TObject; var msg : TMessage; var bContinue : boolean) of object;
  TSubclasser = class(TComponent)
    private
      FOnPreProcessMsg: TSubclassEvent;
      FOnPostProcessMsg: TSubclassEvent;
      FSubClassControl: TWinControl;
      FSubClassHandle : HWND;
      function GetSubClassControl: TWinControl;
      procedure SetSubClassControl(Value: TWinControl);
      function GetSubClassHandle: HWND;
      procedure SetSubClassHandle(Value: HWND);
    private //
      FOldWndProc: Pointer;
      FNewWndProc: Pointer;
      bSubClassed : boolean;
      procedure DoSubclass;
      procedure UndoSubclass;
      procedure SubclassProc( var msg:TMessage );
    public
      constructor Create(Owner: TComponent); override;
      destructor Destroy; override;
      property SubClassHandle: HWND read GetSubClassHandle write SetSubClassHandle;
    published
      property SubClassControl: TWinControl read GetSubClassControl write SetSubClassControl;
      property OnPreProcessMsg: TSubclassEvent read FOnPreprocessMsg write FOnPreprocessMsg;
      property OnPostProcessMsg: TSubclassEvent read FOnPostProcessMsg write FOnPostProcessMsg;
  end;

procedure Register;

implementation         
uses jitdbg32;
//-------------------------------------------------------------------------------------//
procedure Register;
begin
  RegisterComponents('Dyplom''s', [TSubclasser]);
end;

//-------------------------------------------------------------------------------------//
function TSubclasser.GetSubClassControl: TWinControl;
begin
	result := FSubClassControl;
end;

//-------------------------------------------------------------------------------------//
procedure TSubclasser.SetSubClassControl(Value: TWinControl);
begin
  if (FSubClassControl <> Value) then begin
    FSubClassControl := Value;
    if csDesigning in ComponentState then begin
       bSubClassed := false;
       FSubClassHandle := 0;
     end
    else begin
      SubClassHandle := FSubClassControl.Handle;
    end;
  end;
end;

//-------------------------------------------------------------------------------------//
function TSubclasser.GetSubClassHandle: HWND;
begin
  result := FSubclassHandle;
end;

//-------------------------------------------------------------------------------------//
procedure TSubclasser.SetSubClassHandle(Value: HWND);
begin
  if csDesigning in ComponentState then exit;
  if Value<>FSubClassHandle then begin
    UndoSubClass;
    FSubClassControl := nil;
    FSubClassHandle := Value;
    DoSubClass;
  end;
end;

//-------------------------------------------------------------------------------------//
constructor TSubclasser.Create(Owner: TComponent);
begin
  inherited Create(Owner);
  bSubClassed := false;
  DoSubclass;
end;

//-------------------------------------------------------------------------------------//
destructor TSubclasser.Destroy;
begin
   UndoSubclass;
   inherited Destroy;
end;

//-------------------------------------------------------------------------------------//
procedure TSubclasser.DoSubclass;
begin
  if not bSubClassed and (FSubClassHandle<>0) then begin
    FNewWndProc := MakeObjectInstance(SubclassProc);
    FOldWndProc := Pointer(GetWindowLong(FSubClassHandle, GWL_WNDPROC));
    SetWindowLong( FSubClassHandle, GWL_WNDPROC, Longint( FNewWndProc ));
    bSubClassed := true;
  end;
end;

//-------------------------------------------------------------------------------------//
procedure TSubclasser.UndoSubclass;
begin
  if bSubClassed and (FSubClassHandle<>0) then begin
     SetWindowLong( FSubClassHandle, GWL_WNDPROC, Longint( FOldWndProc) );
     FreeObjectInstance( FNewWndProc );
     bSubClassed := false;
  end;
end;

//-------------------------------------------------------------------------------------//
procedure TSubclasser.SubclassProc( var msg:TMessage );
var bContinue : boolean;
begin
  if not bSubclassed then exit;
  bContinue := true;
  if assigned( OnPreProcessMsg ) then
     OnPreProcessMsg( self, msg, bContinue );
  if bContinue then begin
    Msg.Result := CallWindowProc(FOldWndProc, FSubClassHandle, Msg.Msg, Msg.wParam, Msg.lParam);
  end;
  if bContinue and assigned( OnPostProcessMsg ) then
     OnPostProcessMsg( self, msg, bContinue );
  if Msg.Msg = WM_DESTROY then UndoSubclass;
end;


end.
