unit Strings;

{ Delphi 1.0/Delphi 2.0 string compatibility unit. Delphi 2.0 has long
  strings, and uses the SetLength and SetString functions to manipulate
  them. Delphi 1.0 lacks these functions. For portability, this unit
  defines these procedures appropriately for Delphi 1.0.  Also, Delphi 2.0
  has implicity conversion from string to PChar, so define AsPChar
  which uses the implicit conversion in Delphi 2.0 and does an explicit
  conversion in Delphi 1.0

  Copyright  1995 Tempest Software. All rights reserved.

  You are granted permission to use this software in a component or
  application, without fee or royalty, provided this notice is retained
  without modification. You can modify this software, but do not modify
  or delete this notice.
  
  Tempest Software
  10 November 1995
}

interface

function StrToPChar(const Str: string): PChar;

{$ifndef WIN32}
type
  ShortString = string;
  PShortString = ^ShortString;
  AnsiChar = Char;
  PAnsiChar = ^AnsiChar;

procedure SetLength(var Str: string; Length: Byte);
procedure SetString(var Str: string; From: PChar; Length: Byte);
function Trim(const Str: string): string;
function TrimLeft(const Str: string): string;
function TrimRight(const Str: string): string;
{$endif}

implementation

{$ifdef WIN32}
function StrToPChar(const Str: string): PChar;
begin
  Result := PChar(Str);
end;

{$else}
uses SysUtils;

{ Return a PChar representation of the string, Str. If Str already
  ends with a zero byte, then just return a pointer to the first
  character in Str. Otherwise, allocate a dynamic copy of the string.
  Keep a ring of 8 dynamic strings, and free the old strings. Thus,
  you can usually rely on the returned string being valid while it
  is needed. The most common need is to pass an argument to a Windows
  API function, so the need is temporary, but several such strings
  might be required. That's why the ring has 8 items in: more than
  enough for most uses. }
type
  TRingIndex = 0..7;
var
  Ring: array[TRingIndex] of PChar;
  RingIndex: TRingIndex;
  
function StrToPChar(const Str: string): PChar;
var
  Ptr: PChar;
begin
  Ptr := @Str[Length(Str)];
  Inc(Ptr);
  if Ptr^ = #0 then
    Result := @Str[1]
  else
  begin
    Result := StrAlloc(Length(Str)+1);    
    StrPCopy(Result, Str);
    StrDispose(Ring[RingIndex]);
    Ring[RingIndex] := Result;
    RingIndex := (RingIndex + 1) mod (High(TRingIndex) + 1);
  end;
end;

{ Set the length of a string. }
procedure SetLength(var Str: string; Length: Byte);
begin
  Str[0] := Chr(Length)
end;

{ Set the contents of a string. }
procedure SetString(var Str: string; From: PChar; Length: Byte);
var
  FromLen: Integer;
begin
  Str[0] := Chr(Length);
  { In Delphi 2.0, a nil pointer represents an empty string. It is sloppy
    to pass nil when you mean '', but some people do it. }
  if From <> nil then
  begin
    { Only copy as many bytes as are in the From string. }
    FromLen := StrLen(From);
    if FromLen < Length then
      Length := FromLen+1;        { include the trailing #0 byte }
    Move(From^, Str[1], Length);
  end;
end;

{ Return whether the character, C, is a white space character,
  or a non-printable control character. }
function IsWhiteSpace(C: Char): Boolean;
begin
  Result := C in [#0..' ']
end;

{ Trim all leading and trailing white space characters. }
function Trim(const Str: string): string;
var
  L, R: Integer;
begin
  L := 1;
  R := Length(Str);
  while (L < R) and IsWhiteSpace(Str[L]) do
    Inc(L);
  while (L < R) and IsWhiteSpace(Str[R]) do
    Dec(R);
  Result := Copy(Str, L, R-L+1);
end;

{ Trim leading white space characters. }
function TrimLeft(const Str: string): string;
var
  L, R: Integer;
begin
  L := 1;
  R := Length(Str);
  while (L < R) and IsWhiteSpace(Str[L]) do
    Inc(L);
  Result := Copy(Str, L, 255);
end;

{ Trim trailing white space characters. }
function TrimRight(const Str: string): string;
var
  R: Integer;
begin
  R := Length(Str);
  while (1 < R) and IsWhiteSpace(Str[R]) do
    Dec(R);
  Result := Copy(Str, 1, R);
end;

{ Free all the left over strings. }
procedure Terminate; far;
var
  I: TRingIndex;
begin
  for I := Low(TRingIndex) to High(TRingIndex) do
  begin
    StrDispose(Ring[I]);
    Ring[I] := nil; { just in case StrToPChar is called again }
  end;
end;

initialization
  AddExitProc(Terminate);

{$endif}
end.
