unit Dyninout;

interface

uses Forms, Classes, Dialogs, StdCtrls, Controls, Messages, Sysutils, DBTables, DB;

{Insert the header file - Has class and other type definitions.}
{$I dynio.inc}

implementation

{Constructor for InOut class}
constructor TDynSQLClass.Create(AOwner: TComponent);
begin
   InOutHeadPtr := nil;
   InOutLastPtr := nil;
   UpdateAddBPtr := nil;
   NewCancelBPtr := nil;
   DeleteBPtr := nil;
   LBHeadPtr := nil;
   LBLastPtr := nil;
   DynHeadPtr := nil;
   DynLastPtr := nil;
   DynFuncHeadPtr := nil;
   DynFuncLastPtr := nil;
   ListOfControlsByName := TStringList.Create;
   TypeOfCall := '';
   TypeOfSQLDef := '';
   TotalAltCols := 0;
   AlternateSQLCol := '';
   AlternateSQLData := '';
   AlternateSQLStmt := '';
   inherited Create(AOwner);
end;

{Destructor for InOut class}
destructor TDynSQLClass.Destroy;
var
   I: integer;
   CurPtr, NextPtr: pointer;
begin
   {Free the InOutObject's records}
   CurPtr := InOutHeadPtr;
   {Loop until the end is hit.}
   while (CurPtr <> nil) do begin
      NextPtr := InOutObject(CurPtr)^.NextOne;
      dispose(CurPtr);
      CurPtr := NextPtr;
   end;
   {Free the records to the button information}
   if (DeleteBPtr <> nil) then
      dispose(DeleteBPtr);
   if (UpdateAddBPtr <> nil) then
      dispose(UpdateAddBPtr);
   if (NewCancelBPtr <> nil) then
      dispose(NewCancelBPtr);

   {Free the SQLListBox's records}
   CurPtr := LBHeadPtr;
   {Loop until the end is hit.}
   while (CurPtr <> nil) do begin
      NextPtr := SQLListBox(CurPtr)^.NextOne;
      dispose(CurPtr);
      CurPtr := NextPtr;
   end;

   {Free the DynamicSQL Object's records}
   SQLDelFieldList;

   {Free the DynamicFunction Object's records}
   SQLDelFunctionList;

   {Free the Memory of the list of controls by name.}
   I := 0;
   while (I < ListOfControlsByName.Count) do begin
      dispose(pointer(ListOfControlsByName.Objects[I]));
      inc(I);
   end;
   ListOfControlsByName.Free;

   {Do the Form destroy}
   inherited Destroy;
end;

{Returns the first word from a string}
function TDynSQLClass.ReturnFirstWord(Data: string; Sep: string): string;
begin
   {If seperator is not found, return entire text.}
   if (pos(Sep, Data) = 0) then begin
      Result := Data;
   end
   else begin  {Else return the first word.}
      Result := copy(Data, 0, pos(Sep,Data) - 1);
   end;
end;

{Returns the string, minus the first word from a string}
function TDynSQLClass.TakeFirstWordOff(Data: string; Sep: string): string;
begin
   {If seperator is not found, return nothing.}
   if (pos(Sep, Data) = 0) then begin
      Result := '';
   end
   else begin {Else, return entire text minus first word.}
      Result := copy(Data, pos(Sep,Data) + 1, (length(Data) - pos(Sep,Data)));
   end;
end;

{Send out notify message to a list of controls.}
procedure TDynSQLClass.NotifyMe(ToWhom, Msg: Pointer);
var
   Result: LongInt;
begin
   {Use FindComponent (you send in the component name to get the index.)}
   Result := Perform(sx_MyMessage, 0, LongInt(Msg));
end;

{Testing receiving a message...}
procedure TDynSQLClass.SXMyMessage(var Msg: TMessage);
var
   Ptr: PChar;
   Test: String;
begin
   Ptr := PChar(Msg.LParam);
   MessageDlg(StrPas(Ptr), mtInformation, [mbOk], 0);
end;

{Returns the pointer to the head of the InOutList.}
function TDynSQLClass.GetHeadPtr: Pointer;
begin
   Result := InOutHeadPtr;
end;

{Returns the pointer to the end of the InOutList.}
function TDynSQLClass.GetLastPtr: Pointer;
begin
   Result := InOutLastPtr;
end;

{Sets the header pointer.}
procedure TDynSQLClass.SetHeadPtr(HeadPtr: Pointer);
begin
   InOutHeadPtr := HeadPtr;
end;

{Sets the trailer pointer.}
procedure TDynSQLClass.SetLastPtr(LastPtr: Pointer);
begin
   InOutLastPtr := LastPtr;
end;

{Returns the pointer to the head of the TSQLFieldInfo.}
function TDynSQLClass.DynGetHeadPtr: Pointer;
begin
   Result := DynHeadPtr;
end;

{Returns the pointer to the end of the TSQLFieldInfo.}
function TDynSQLClass.DynGetLastPtr: Pointer;
begin
   Result := DynLastPtr;
end;

{Sets the header pointer.}
procedure TDynSQLClass.DynSetHeadPtr(HeadPtr: Pointer);
begin
   DynHeadPtr := HeadPtr;
end;

{Sets the trailer pointer.}
procedure TDynSQLClass.DynSetLastPtr(LastPtr: Pointer);
begin
   DynLastPtr := LastPtr;
end;

{Run an SQL statement in the indicated TQuery object}
function TDynSQLClass.RunSQL(SQLStmt: string; SqlComp: Integer): Integer;
var
   Cnt: integer;
begin
   Result := 1;
   {Run the Insert/Update SQL statement.}
   if (SqlComp > -1) and (SqlStmt <> '') then begin {If it finds the TQuery component...}
      Screen.Cursor := crHourGlass;
      try
         try
            with TQuery(Components[SqlComp]) do begin
               Close;
               SQL.Clear;
               SQL.Add(SqlStmt);
               ExecSql;
            end;
         except
            on E: EDataBaseError do begin
               {If there was an error, see if it was doing an insert and there
                 were Max(column) + 1 columns. You get an error if there are records
                 in the table because Max(column) + 1 returns NULL!}
               if (TotalAltCols > 0) then begin
                  try
                     with TQuery(Components[SqlComp]) do begin
                        Close;
                        SQL.Clear;
                        SQL.Add(AlternateSQLStmt);
                        Prepare;
                        Cnt := 0;
                        while (Cnt < TotalAltCols) do begin
                           Params[Cnt].AsInteger := 1;
                           inc(Cnt);
                        end;
                        ExecSql;
                     end;
                  except
                     on E: EDataBaseError do begin
{Use function MessageBox(WndParent: HWnd; Txt, Caption: PChar; TextType: Word): Integer;
  instead of the MessageDlg. See if you can get more of the error message back.
  Include the SQL statement and put that 'happened here' code.}
                        MessageDlg(E.Message, mtError, [mbok], 0);
                        Result := 0;
                     end;
                  end;
               end
               else begin
{Use function MessageBox(WndParent: HWnd; Txt, Caption: PChar; TextType: Word): Integer;
  instead of the MessageDlg. See if you can get more of the error message back.
  Include the SQL statement and put that 'happened here' code.}
                  MessageDlg(E.Message, mtError, [mbok], 0);
                  Result := 0;
               end;
            end;
         end;
      finally
         Screen.Cursor := crDefault;
      end;
   end;
   TotalAltCols := 0;
   AlternateSQLCol := '';
   AlternateSQLData := '';
   AlternateSQLStmt := '';
end;

{Returns the data from a component for a SQL statment.}
function TDynSQLClass.ReturnText(CompNum: Integer): string;
begin
   Result := '';
   if (Components[CompNum] is TEdit) then begin  {Edit field}
      Result := TEdit(Components[CompNum]).Text;
   end
   else if (Components[CompNum] is TMemo) then begin {Memo field}
      Result := TMemo(Components[CompNum]).Text;
   end;
end;

{Builds SQL detail - like columns and values.}
procedure TDynSQLClass.BuildSqlDetail(HoldCol, Data: string; var SqlTmp1, SqlTmp2: string; TypeDet, KeyFlag: string);
begin
   if (TypeDet = 'Insert') then begin
      if (AlternateSQLCol = '') then begin
         AlternateSQLCol := HoldCol;
      end
      else begin
         AlternateSQLCol := AlternateSQLCol + ',' + HoldCol;
      end;
      if (AlternateSQLData = '') then begin
         if (pos('Max(', Data) <> 0) then begin
            AlternateSQLData := ':Data' + inttostr(TotalAltCols);
            inc(TotalAltCols);
         end
         else begin
            AlternateSQLData := Data;
         end;
      end
      else begin
         if (pos('Max(', Data) <> 0) then begin
            inc(TotalAltCols);
            AlternateSQLData := AlternateSQLData + ',' + ':Data' + inttostr(TotalAltCols);
         end
         else begin
            AlternateSQLData := AlternateSQLData + ',' + Data;
         end;
      end;
   end;
   {First part of SQL statement}
   if (SqlTmp1 = '') then begin
      if (TypeDet = 'Insert') then begin
         SqlTmp1 := HoldCol;  {Column to be used in insert}
      end
      else if (TypeDet = 'Update') then begin
         {If it is part of the primary key, it will be part of the Where clause}
         if (KeyFlag = 'P') then begin
            SqlTmp1 := SqlTmp1;
         end
         else begin  {This is a column to be updated.}
            SqlTmp1 := HoldCol + ' = ' + Data;
         end;
      end;
   end
   else begin
      if (TypeDet = 'Insert') then begin
         {Comma delimited list of column names to be used in the insert}
         SqlTmp1 := SqlTmp1 + ',' + HoldCol;
      end
      else if (TypeDet = 'Update') then begin
         {If it is part of the primary key, it will be part of the Where clause}
         if (KeyFlag = 'P') then begin
            SqlTmp1 := SqlTmp1;
         end
         else begin  {This is a column to be updated.}
            SqlTmp1 := SqlTmp1 + ',' + HoldCol + ' = ' + Data;
         end;
      end;
   end;

   {Second part of SQL statement}
   if (SqlTmp2 = '') then begin
      if (TypeDet = 'Insert') then begin
         SqlTmp2 := Data;     {Data to be used in insert}
      end
      else if (TypeDet = 'Update') or (TypeDet = 'Delete') then begin
         {If it is part of the primary key, it will be part of the Where clause}
         if (KeyFlag = 'P') then begin
            SqlTmp2 := HoldCol + ' = ' + Data;
         end
         else begin  {This is a column to be updated.}
            SqlTmp2 := SqlTmp2;
         end;
      end;
   end
   else begin
      if (TypeDet = 'Insert') then begin
         {Comma delimited list of data to be inserted}
         SqlTmp2 := SqlTmp2 + ',' + Data;
      end
      else if (TypeDet = 'Update') or (TypeDet = 'Delete') then begin
         {If it is part of the primary key, it will be part of the Where clause}
         if (KeyFlag = 'P') then begin
            SqlTmp2 := SqlTmp2 + ' and ' + HoldCol + ' = ' + Data;
         end
         else begin  {This is a column to be updated.}
            SqlTmp2 := SqlTmp2;
         end;
      end;
   end;
end;

{Takes the data and processes it according to type (or if there is none, puts
  the word NULL in its place.}
function TDynSQLClass.ProcessSQLData(Data: string; ColType: string): string;
var
   Word: string;
begin
   Result := 'NULL';
   if (Data = '') then begin
      Result := 'NULL';
   end
   else if (ColType = 'I') or (ColType = 'N') then begin {The data is an integer}
     {Do nothing, it is an integer}
     Result := Data;
   end
   {The data is a string - This is the default.}
   else begin {Types typically the fall into here are T, S, or D.}
      {If there are any single quotes in the outgoing data, this makes them into
        two single quotes (so that one can be output).}
      Word := Data;
      Data := '';
      while (pos('''', Word) <> 0) do begin
         AppendStr(Data, ((ReturnFirstWord(Word, '''')) + '''' + ''''));
         Word := TakeFirstWordOff(Word, '''');
      end;
      AppendStr(Data, Word);

      Result := '''' + Data + '''';
   end;
end;

{Sets the type of SQL processing for the screen.
 Valid types are:
  'ParmBuild' - Build SQL using parameters obtained from TEdit and TMemo fields on FormCreate.
  'DynBuild' - Build the SQL from what is specified by the sub-programs that were identified on FormCreate}
procedure TDynSQLClass.DynSetSQLProcessType(SQLProcessType: string);
begin
   {On FormCreate, read all TEdit, TMemo, and TQuery fields identifying how all Update, Insert,
    and Delete SQL will be generated and Processed}
   if (CompareText(SQLProcessType, 'ParmBuild') = 0) then begin
      TypeOfSQLDef := 'ParmBuild'
   end
   {On FormCreate, the programmer specifies what procedures will be run to build the
    data needed to identify how all Update, Insert and Delete SQL will be generated and
    Processed}
   else if (CompareText(SQLProcessType, 'DynBuild') = 0) then begin
      TypeOfSQLDef := 'DynBuild'
   end
   {Default Dynamic SQL build since the user must specify the functions necessary to
    make it work. ParmBuild might pick up false data.}
   else begin
      TypeOfSQLDef := 'DynBuild'
   end;
end;

{Return the SQL Process type}
function TDynSQLClass.DynGetSQLProcessType: string;
begin
   Result := TypeOfSQLDef;
end;

{Either set a property or clear all data entry Edit fields, Memo fields, and Query components.}
procedure TDynSQLClass.SetDataEntryProp(Prop: String; Value: Pointer);
var
   CurNode: Pointer;
begin
   if (TypeOfSQLDef = 'ParmBuild') then begin
      CurNode := GetHeadPtr; {Set current node pointer to the first node.}
      while (CurNode <> nil) do begin
         with InOutObject(CurNode)^ do begin
            ProcessSetDataEntryProp(Prop, Value, CompNum);
            CurNode := NextOne;
         end;
      end;
   end
   else if (TypeOfSQLDef = 'DynBuild') then begin
      CurNode := DynGetHeadPtr; {Set current node pointer to the first node.}
      while (CurNode <> nil) do begin
         with TSQLFieldInfo(CurNode)^ do begin
            if ((CompareText(Prop, 'Clear') = 0) and (CompareText(ExtraParm, 'NoClear') <> 0))
               or (CompareText(Prop, 'Clear') <> 0) then begin
               ProcessSetDataEntryProp(Prop, Value, CompNum);
            end;
            CurNode := NextOne;
         end;
      end;
   end;
end;

{Processes: Either set a property or clear all data entry Edit fields, Memo fields, and Query components.}
procedure TDynSQLClass.ProcessSetDataEntryProp(Prop: String; Value: Pointer; CompNum: Integer);
begin
   {Edit field}
   if (Components[CompNum] is TEdit) then begin
      with TEdit(Components[CompNum]) do begin
         if (Prop = 'Clear') then begin {Clear field}
            Clear;
         end;
      end;
   end
   {Memo field}
   else if (Components[CompNum] is TMemo) then begin
      with TMemo(Components[CompNum]) do begin
         if (Prop = 'Clear') then begin {Clear field}
            Clear;
         end;
      end;
   end
   {Query component}
   else if (Components[CompNum] is TQuery) then begin
      with TQuery(Components[CompNum]) do begin
         if (Prop = 'Clear') then begin {Clear component's SQL field}
            SQL.Clear;
         end;
      end;
   end;
end;

{Get the database names for components and clear them.}
procedure TDynSQLClass.FormCreate(Sender: TObject);
var
  I, Cnt, Cnt2: Integer;
  CurObject, TempPtr: pointer;
  ControlName, ControlType: string;
begin
  DeleteBPtr := nil;
  UpdateAddBPtr := nil;
  NewCancelBPtr := nil;
  {
     If the programmer will define the database fields through text in the control...
  }
  if (TypeOfSQLDef = 'ParmBuild') then begin
     for I := 0 to ComponentCount -1 do begin
        if (Components[I] is TEdit) then begin
           with TEdit(Components[I]) do begin
              if (Text <> '') then begin
                 {Create a new InOutObject}
                 CurObject := CreateNewInOut(Text, I, 'TEdit');
                 {Clear the text field.}
                 Clear;
              end;
           end;
        end
        else if (Components[I] is TMemo) then begin
           with TMemo(Components[I]) do begin
              if (Lines[0] <> '') then begin
                 {Create a new InOutObject}
                 CurObject := CreateNewInOut(Lines[0], I, 'TMemo');
                 {Clear the text field.}
                 Clear;
              end;
           end;
        end
        else if (Components[I] is TQuery) then begin
           with TQuery(Components[I]).SQL do begin
              if (CompareText(Strings[0], 'Insert') = 0) or (CompareText(Strings[0], 'Update') = 0)
                  or (CompareText(Strings[0], 'Delete') = 0) then begin
                 Cnt := 0;
                 while (Cnt < Count) do begin
                    {Create a new InOutObject}
                    CurObject := CreateNewInOut(Strings[Cnt], I, 'TQuery');
                    inc(Cnt);
                 end;
                {Clear the text field.}
                Clear;
              end;
           end;
        end
        else if (Components[I] is TButton) then begin
           if (MaxAvail < SizeOf(SQLButton)) then begin
              MessageDlg('Not enough memory', mtWarning, [mbOk], 0);
           end
           else
           begin
              with TButton(Components[I]) do begin
                 {Add/Update button found}
                 if (Name = 'AddUpdate') then begin
                    {Allocate memory for a new InOutObject}
                    UpdateAddBPtr := new(SQLButton);
                    SQLButton(UpdateAddBPtr)^.CompText := Caption;
                    SQLButton(UpdateAddBPtr)^.CompNum := I;
                 end
                 {New/Cancel button found}
                 else if (Name = 'NewCancel') then begin
                    {Allocate memory for a new InOutObject}
                    NewCancelBPtr := new(SQLButton);
                    SQLButton(NewCancelBPtr)^.CompText := Caption;
                    SQLButton(NewCancelBPtr)^.CompNum := I;
                 end
                 {Delete button found}
                 else if (Name = 'Delete') then begin
                    {Allocate memory for a new InOutObject}
                    DeleteBPtr := new(SQLButton);
                    SQLButton(DeleteBPtr)^.CompText := Caption;
                    SQLButton(DeleteBPtr)^.CompNum := I;
                 end;
              end;
           end;
        end
        {List box components.}
        else if (Components[I] is TListBox) then begin
           with TListBox(Components[I]) do begin
              if (Items[0] <> '') then begin
                 SetupListBox(I);
              end;
           end;
        end;
     end;
  end
  {
     If the programmer will define the database fields through functions...
  }
  {Builds a list of control names and their component number for the Dynamic SQL
   routines.}
  else if (TypeOfSQLDef = 'DynBuild') then begin
     ListOfControlsByName.Clear;
     {Cycle through the components making a string list of controls that I might be
      interested in.}
     for I := 0 to ComponentCount -1 do begin
        ControlName := '';
        ControlType := '';
        if (Components[I] is TButton) then begin
           ControlName := TButton(Components[I]).Name;
           ControlType := 'TButton';
           {Identify the big three buttons: New/Cancel, Insert/Update, and Delete}
           if (MaxAvail < SizeOf(SQLButton)) then begin
              MessageDlg('Not enough memory', mtWarning, [mbOk], 0);
           end
           else
           begin
              with TButton(Components[I]) do begin
                 {Add/Update button found}
                 if (Name = 'AddUpdate') then begin
                    {Allocate memory for a new InOutObject}
                    UpdateAddBPtr := new(SQLButton);
                    SQLButton(UpdateAddBPtr)^.CompText := Caption;
                    SQLButton(UpdateAddBPtr)^.CompNum := I;
                 end
                 {New/Cancel button found}
                 else if (Name = 'NewCancel') then begin
                    {Allocate memory for a new InOutObject}
                    NewCancelBPtr := new(SQLButton);
                    SQLButton(NewCancelBPtr)^.CompText := Caption;
                    SQLButton(NewCancelBPtr)^.CompNum := I;
                 end
                 {Delete button found}
                 else if (Name = 'Delete') then begin
                    {Allocate memory for a new InOutObject}
                    DeleteBPtr := new(SQLButton);
                    SQLButton(DeleteBPtr)^.CompText := Caption;
                    SQLButton(DeleteBPtr)^.CompNum := I;
                 end;
              end;
           end;
        end
        else if (Components[I] is TEdit) then begin
           ControlName := TEdit(Components[I]).Name;
           ControlType := 'TEdit';
        end
        else if (Components[I] is TMemo) then begin
           ControlName := TMemo(Components[I]).Name;
           ControlType := 'TMemo';
        end
        else if (Components[I] is TQuery) then begin
           ControlName := TQuery(Components[I]).Name;
           ControlType := 'TQuery';
        end;

        if (ControlName <> '') then begin
           if (MaxAvail < SizeOf(InOutObject)) then begin
              MessageDlg('Not enough memory', mtWarning, [mbOk], 0);
           end
           else
           begin
              {Allocate memory for a new InOutObject}
              CurObject := new(TSQLControlInfo);

              {Cast 'CurObject' as type InOutObject, which must be dereferenced to get at the
               record fields.}
              with TSQLControlInfo(CurObject)^ do begin
                 CICompType := ControlType;
                 CICompNumber := I;
                 {This is PRETTY close to an associative array (an array that is keyed
                  by anything). All that would need to be done is to see if the string
                  already exists before adding a new one (if it does, over write it
                  with the new value). The second parameter in the AddObject is the
                  value for the array entry (with the ControlName being the key!).}
                 ListOfControlsByName.AddObject(uppercase(ControlName), TObject(CurObject));
              end;
           end;
        end;
     end;
  end;
end;

{Add or update button clicked.}
procedure TDynSQLClass.AddUpdateClick(Sender: TObject; CaptionText: string);
var
   SqlStmt, SqlTmp1, SqlTmp2, Table: string;
   SqlComp, MaxFlag, SqlError: Integer;
begin
   {Initialize local variables}
   SqlStmt := '';  {Final SQL statement.}
   SqlTmp1 := '';  {First half of the user's data for the SQL statement.}
   SqlTmp2 := '';  {Second half of the user's data for the SQL statement.}
   Table := '';    {Table name.}
   SqlComp := -1;  {TQuery component number to be used for the Update/Insert}
   MaxFlag := 0;   {Max plus one flag for quickie unique Number on Inserts}

   if (CaptionText = '&Add') then begin  {Add button clicked}
      if (TypeOfSQLDef = 'ParmBuild') then begin
         AddUpdateClickParmBuild(Table, SqlTmp1, SqlTmp2, CaptionText, SqlComp, MaxFlag);
      end
      else if (TypeOfSQLDef = 'DynBuild') then begin
         AddUpdateClickDynBuild(Table, SqlTmp1, SqlTmp2, CaptionText, SqlComp, MaxFlag);
      end;
      {Build the Insert SQL statement.}
      if (MaxFlag = 1) then begin  {Insert with a Max plus one on one or more columns}
         SqlStmt := 'INSERT INTO ' + Table + '(' + SqlTmp1 + ') select ' + SqlTmp2 + ' from ' + Table;
         AlternateSQLStmt := 'INSERT INTO ' + Table + ' (' + AlternateSQLCol + ')' + ' values (' + AlternateSQLData + ')'
      end
      else begin   {Insert normally}
         SqlStmt := 'INSERT INTO ' + Table + '(' + SqlTmp1 + ') VALUES (' + SqlTmp2 + ')';
      end;
   end

   else begin {Update button clicked}
      if (TypeOfSQLDef = 'ParmBuild') then begin
         AddUpdateClickParmBuild(Table, SqlTmp1, SqlTmp2, CaptionText, SqlComp, MaxFlag);
      end
      else if (TypeOfSQLDef = 'DynBuild') then begin
         AddUpdateClickDynBuild(Table, SqlTmp1, SqlTmp2, CaptionText, SqlComp, MaxFlag);
      end;

      {Build the Update SQL statement.}
      SqlStmt := 'UPDATE ' + Table + ' SET ' + SqlTmp1 + ' WHERE ' + SqlTmp2;
   end;

   {Run the SQL statement.
    If SqlError = 1, ran fine. Else, if SqlError = 0, there was an error.}
   SqlError := RunSQL(SQLStmt, SqlComp);

   if (SqlError = 1) and (CaptionText = '&Add') then
      SetDataEntryProp('Clear', nil); {Clear Edit fields, Memo Fields, and Query components.}
end;

{The Delete button has been hit}
procedure TDynSQLClass.DeleteClick(Sender: TObject);
var
   SqlStmt, SqlTmp2, Table: string;
   SqlComp, SqlError: Integer;
begin

   SqlStmt := '';  {Final SQL statement.}
   SqlTmp2 := '';  {Second half of the user's data for the SQL statement.}
   Table := '';    {Table name.}
   SqlComp := -1;  {TQuery component number to be used for the Update/Insert}

   if (TypeOfSQLDef = 'ParmBuild') then begin
      DeleteClickParmBuild(Table, SqlTmp2, SqlComp);
   end
   else if (TypeOfSQLDef = 'DynBuild') then begin
      DeleteClickDynBuild(Table, SqlTmp2, SqlComp);
   end;

   {Build the Delete SQL statement.}
   SqlStmt := 'DELETE FROM ' + Table + ' WHERE ' + SqlTmp2;

   {Run the SQL statement.
    If SqlError = 1, ran fine. Else, if SqlError = 0, there was an error.}
   SqlError := RunSQL(SQLStmt, SqlComp);
   if (SqlError = 1) then
      SetDataEntryProp('Clear', nil); {Clear Edit fields, Memo Fields, and Query components.}
end;

{The New/Cancel button has been hit.}
procedure TDynSQLClass.NewCancelClick(Sender: TObject);
begin
   with SQLButton(NewCancelBPtr)^ do begin
      if (TButton(Components[CompNum]).Caption = '&New') then begin  {New button has been hit.}
         TButton(Components[CompNum]).Caption := 'Canc&el';
         TButton(Components[SQLButton(UpdateAddBPtr)^.CompNum]).Caption := '&Add';
         TButton(Components[SQLButton(DeleteBPtr)^.CompNum]).Enabled := False;
      end
      else begin  {Cancel button has been hit.}
         TButton(Components[CompNum]).Caption := '&New';
         TButton(Components[SQLButton(UpdateAddBPtr)^.CompNum]).Caption := '&Update';
         TButton(Components[SQLButton(DeleteBPtr)^.CompNum]).Enabled := True;
      end;
   end;
   {I figure out what fields need to be cleared through a call to Insert because it is
    these fields that will be edited.}
   if (TypeOfSQLDef = 'DynBuild') then begin
      TypeOfCall := 'CalledByClear';
      SQLRunFunction('Insert');
      TypeOfCall := '';
   end;
   SetDataEntryProp('Clear', nil); {Clear Edit fields, Memo Fields, and Query components.}
end;

{------------------------------------------------------------------------------------}
{ Member functions to handle SQL generated from info. found in TEdit and TMemo fields}
{------------------------------------------------------------------------------------}

{Allocates memory for and creates the next InOutObject (which is basically used
 to track database names for components.)}
function TDynSQLClass.CreateNewInOut(DBColumn: string; I: Integer; ColumnType: string): Pointer;
var
   CurObject, next: pointer;
   Temp, Temp2: string;
begin
   if (MaxAvail < SizeOf(InOutObject)) then begin
     MessageDlg('Not enough memory', mtWarning, [mbOk], 0);
     Result := nil;
   end
   else
   begin
     { Allocate memory for a new InOutObject}
     CurObject := new(InOutObject);

     {Cast 'CurObject' as type InOutObject, which must be dereferenced to get at the
      record fields.}
     with InOutObject(CurObject)^ do begin
        {Set this to 'nil' because it is the last one.}
        NextOne := nil;
        {Initialize all other fields.}
        Column := '';
        ColType := '';
        KeyFlag := '';
        CompType := '';
        ExtraFlag := '';
        CompNum := -1;
        DataGroup := 0;

        {Get the formally last pointer.}
        next := InOutLastPtr;
        {Cast 'next' as type InOutObject, which must be dereferenced to get at the record fields.
         Set this to 'CurObject' so that it will point to its next one.}
        if (next <> nil) then begin
           InOutObject(next)^.NextOne := CurObject;
        end;
        {'CurObject' is now the new last pointer.}
        InOutLastPtr := CurObject;
        {If the header has not yet been defined, set him to CurObject}
        if (InOutHeadPtr = nil) then begin
           InOutHeadPtr := CurObject;
        end;

        {These are non-TQuery components}
        if (ColumnType <> 'TQuery') then begin
           {Used to group data}
           DataGroup := StrToInt(ReturnFirstWord(DBColumn, ' '));
           Temp := TakeFirstWordOff(DBColumn, ' ');
           {Assign the database name to the Column field of the InOutObject}
           Column := ReturnFirstWord(Temp, ' ');
           Temp := TakeFirstWordOff(Temp, ' ');
           if (pos(' ', Temp) <> 0) then begin   {There is a primary key flag}
              {Column type}
              ColType := ReturnFirstWord(Temp, ' ');
              Temp := TakeFirstWordOff(Temp, ' ');
              if (pos(' ',Temp) = 0) and ((CompareText(Temp, 'P') = 0) or (CompareText(Temp, 'F') = 0)) then begin
                 KeyFlag := UpperCase(Temp);  {This is the primary key flag}
              end
              else if (pos(' ',Temp) = 0) and (Temp = 'MaxPlusOne') then begin
                 ExtraFlag := Temp;  {This is the max plus one flag}
              end
              else begin
                 Temp2 := ReturnFirstWord(Temp, ' ');
                 Temp := TakeFirstWordOff(Temp, ' ');
                 if (CompareText(Temp2, 'P') = 0) or (CompareText(Temp2, 'F') = 0) then begin
                    KeyFlag := UpperCase(Temp2);  {This is the primary key flag}
                    ExtraFlag := Temp;  {This is the max plus one flag}
                 end
                 else begin
                    KeyFlag := UpperCase(Temp);  {This is the primary key flag}
                    ExtraFlag := Temp2;  {This is the max plus one flag}
                 end;
              end;
           end
           else begin
              ColType := Temp;
           end;
        end
        else begin  {This is a query component}
           ColType := DBColumn;
        end;
        {Assign the component number to the CompNum field of the InOutObject}
        CompNum := I;
        CompType := ColumnType;
     end;

     Result := CurObject;
   end;
end;

{Add/Update button clicked - This uses the information collected from the TEdit
 and TMemo text fields.}
procedure TDynSQLClass.AddUpdateClickParmBuild(var Table, SqlTmp1, SqlTmp2, CaptionText: string;
  var SqlComp, MaxFlag: integer);
var
   CurNode: pointer;
   Data, HoldCol: string;
begin
   {Initialize local variables}
   CurNode := GetHeadPtr; {Set current node pointer to the first node.}
   Data := '';     {Specific data element.}
   HoldCol := '';  {Column name.}
   if (CaptionText = '&Add') then begin  {Add button clicked}
      while (CurNode <> nil) do begin
         with InOutObject(CurNode)^ do begin
            if (CompType = 'TQuery') then begin
               if (CompareText(ColType, 'Insert') = 0) then begin
                   SqlComp := CompNum;
               end;
            end
            else begin
               Table := ReturnFirstWord(Column, '.');
               HoldCol := TakeFirstWordOff(Column, '.');
               if (CompareText(ExtraFlag, 'MaxPlusOne') = 0) then begin {Wants to be max + 1}
                   MaxFlag := 1;
                   Data := 'Max(' + HoldCol + ') + 1'
               end
               else begin
                  Data := ReturnText(CompNum);  {Get the data from the control}
                  Data := ProcessSQLData(Data, ColType);  {Process data for SQL}
               end;
               BuildSqlDetail(HoldCol, Data, SqlTmp1, SqlTmp2, 'Insert', KeyFlag);
            end;
            CurNode := NextOne;
         end;
      end;
   end
   else begin {Update button clicked}
      while (CurNode <> nil) do begin
         with InOutObject(CurNode)^ do begin
            if (CompType = 'TQuery') then begin
               if (CompareText(ColType, 'Update') = 0) then begin
                   SqlComp := CompNum;
               end;
            end
            else begin
               Table := ReturnFirstWord(Column, '.');
               HoldCol := TakeFirstWordOff(Column, '.');
               Data := ReturnText(CompNum);  {Get the data from the control}
               Data := ProcessSQLData(Data, ColType);  {Process data for SQL}
               BuildSqlDetail(HoldCol, Data, SqlTmp1, SqlTmp2, 'Update', KeyFlag);
            end;
            CurNode := NextOne;
         end;
      end;
   end;
end;

{Delete button clicked - This uses the information collected from the TEdit
 and TMemo text fields.}
procedure TDynSQLClass.DeleteClickParmBuild(var Table, SqlTmp2: String; var SqlComp: Integer);
var
   CurNode: pointer;
   SqlTmp1, Data, HoldCol: string;
begin
   {Initialize local variables}
   SqlTmp1 := '';  {First half of the user's data for the SQL statement.}
   Data := '';     {Specific data element.}
   HoldCol := '';  {Column name.}
   CurNode := GetHeadPtr; {Set current node pointer to the first node.}

   {Loop through InOutObject's and build column and data part of SQL Delete statement.}
   while (CurNode <> nil) do begin
      with InOutObject(CurNode)^ do begin
         if (CompType = 'TQuery') then begin
            if (CompareText(ColType, 'Delete') = 0) then begin
                SqlComp := CompNum;
            end;
         end
         else begin
            Table := ReturnFirstWord(Column, '.');
            HoldCol := TakeFirstWordOff(Column, '.');
            Data := ReturnText(CompNum);  {Get the data from the control}
            Data := ProcessSQLData(Data, ColType);  {Process data for SQL}
            BuildSqlDetail(HoldCol, Data, SqlTmp1, SqlTmp2, 'Delete', KeyFlag);
         end;
         CurNode := NextOne;
      end;
   end;
end;

{------------------------------------------------------------------------------}
{                    Member functions to handle Dynamic SQL                    }
{------------------------------------------------------------------------------}

{Deletes Function Object list.}
procedure TDynSQLClass.SQLDelFunctionList;
var
   NextPtr,
   CurNode: Pointer;
begin
   {Free the DynamicFunction Object's records}
   CurNode := DynFuncHeadPtr;
   {Loop until the end is hit.}
   while (CurNode <> nil) do begin
      NextPtr := TSQLFunctions(CurNode)^.NextOne;
      dispose(CurNode);
      CurNode := NextPtr;
   end;
   DynFuncHeadPtr := nil;
   DynFuncLastPtr := nil;
end;

{Deletes Function Object list.}
procedure TDynSQLClass.SQLDelFieldList;
var
   NextPtr,
   CurNode: Pointer;
begin
   {Free the DynamicFunction Object's records}
   CurNode := DynHeadPtr;
   {Loop until the end is hit.}
   while (CurNode <> nil) do begin
      NextPtr := TSQLFieldInfo(CurNode)^.NextOne;
      dispose(CurNode);
      CurNode := NextPtr;
   end;
   DynHeadPtr := nil;
   DynLastPtr := nil;
end;

{Allow user to add functions to the SQL Update, Insert, and Delete list.
 They will be called to prep the Update, Insert, and Delete SQL statements.}
procedure TDynSQLClass.SQLAddFunction(Func: Pointer; Code: string);
var
   NextPtr,
   CurNode: Pointer;
begin
   {If told to, destroy the function list.}
   if (CompareText(Code,'Destroy') = 0) then begin
      SQLDelFunctionList;
   end;
   {Check to see if a function pointer was sent. If so, create a TSQLFunctions object.}
   if (Func <> nil) then begin
      {Allocate memory for a new InOutObject}
      CurNode := new(TSQLFunctions);
      with (TSQLFunctions(CurNode)^) do begin
          FuncPtr := Func;
          NextOne := nil;
      end;
      {It is the very first object created for this list.}
      if (DynFuncHeadPtr = nil) then begin
         DynFuncHeadPtr := CurNode;
      end
      {There are already objects in the list.}
      else begin
         TSQLFunctions(DynFuncLastPtr)^.NextOne := CurNode;
      end;
      {The current node is now the last node.}
      DynFuncLastPtr := CurNode;
   end;
end;

{Runs all of the functions, building objects to cause an Update, Insert or Delete}
procedure TDynSQLClass.SQLRunFunction(Code: string);
var
   NextPtr,
   CurNode: Pointer;
   ReturnVal: string;
   CurObject: Pointer;
   OffSet: integer;
begin
   {Clear out the field object list - we build a new one here.}
   SQLDelFieldList;

   {Cycle through DynamicFunction Object's records}
   CurNode := DynFuncHeadPtr;
   {Loop until the end is hit.}
   while (CurNode <> nil) do begin
      if (MaxAvail < SizeOf(TSQLFieldInfo)) then begin
         MessageDlg('Not enough memory', mtWarning, [mbOk], 0);
      end
      else
      begin
         CurObject := new(TSQLFieldInfo);
         with TSQLFieldInfo(CurObject)^ do begin
            Table := '';
            Column := '';
            ExtraParm := '';
            ComponentName := '';
            ComponentType := '';
            StringData := '';
            DataType := '';
            KeyFlag := '';
            CompNum := -1;
            PtrData := nil;
            NextOne := nil;

            ReturnVal := SQLFuncType(TSQLFunctions(CurNode)^.FuncPtr) (CurObject, Code);
            {Check to see if the object should not be created.}
            if (CompareText(ExtraParm, 'DeleteMe') = 0) then begin
               dispose(CurObject);
            end
            else begin
               {It is the very first object created for this list.}
               if (DynGetHeadPtr = nil) then begin
                  DynSetHeadPtr(CurObject);
               end
               else begin
                  with TSQLFieldInfo(DynGetLastPtr)^ do begin
                     NextOne := CurObject;
                  end;
               end;
               {Set current object as the last object.}
               DynSetLastPtr(CurObject);
               {Convert user's input to uppercase.}
               DataType := uppercase(DataType);
               KeyFlag := uppercase(KeyFlag);

               {Find the component name in the list of names - I store them in uppercase.}
               OffSet := ListOfControlsByName.IndexOf(uppercase(ComponentName));
               {The object part of the ListOfControlsByName string list is the
                 component number. So, do that assignment if found.}
               if (OffSet <> -1) then begin
                  with (TSQLControlInfo(pointer(ListOfControlsByName.Objects[OffSet]))^) do begin
                     CompNum := CICompNumber;
                     ComponentType := CICompType;
                     {Check to see if the user sent the data...}
                     if (StringData = '') then begin
                        {Go get the data from the component.}
                        StringData := ReturnText(CompNum);
                     end;
                  end;
               end;
            end;
            CurNode := TSQLFunctions(CurNode)^.NextOne;
         end;
      end;
   end;
end;

{Add/Update button clicked - This uses the information collected from the TEdit
 and TMemo text fields.}
procedure TDynSQLClass.AddUpdateClickDynBuild(var DelTable, SqlTmp1, SqlTmp2, CaptionText: string;
  var SqlComp, MaxFlag: integer);
var
   CurNode: pointer;
   Data, HoldCol: string;
begin
   {Run the user's functions to build the information needed to create the SQL for the delete.}
   if (CaptionText = '&Add') then   {Add button clicked}
      SQLRunFunction('Insert')
   else
      SQLRunFunction('Update');


   {Initialize local variables}
   CurNode := DynGetHeadPtr; {Set current node pointer to the first node.}
   Data := '';     {Specific data element.}
   HoldCol := '';  {Column name.}
   DelTable := ''; {Table to be written to.}
   if (CaptionText = '&Add') then begin  {Add button clicked}
      while (CurNode <> nil) do begin
         with TSQLFieldInfo(CurNode)^ do begin
            if (ComponentType = 'TQuery') then begin
               if (CompareText(ExtraParm, 'Insert') = 0) then begin
                   SqlComp := CompNum;
               end;
            end
            else begin
               if (Table <> '') then begin
                  DelTable := Table;
               end;
               Data := ProcessSQLData(StringData, DataType);  {Process data for SQL}
               if (CompareText(ExtraParm, 'MaxPlusOne') = 0) then begin {Wants to be max + 1}
                   MaxFlag := 1;
                   Data := 'Max(' + Column + ') + 1'
               end;
               BuildSqlDetail(Column, Data, SqlTmp1, SqlTmp2, 'Insert', KeyFlag);
            end;
            CurNode := TSQLFieldInfo(CurNode)^.NextOne;
         end;
      end;
   end
   else begin {Update button clicked}
      while (CurNode <> nil) do begin
         with TSQLFieldInfo(CurNode)^ do begin
            if (ComponentType = 'TQuery') then begin
               if (CompareText(ExtraParm, 'Update') = 0) then begin
                   SqlComp := CompNum;
               end;
            end
            else begin
               if (Table <> '') then begin
                  DelTable := Table;
               end;
               Data := ProcessSQLData(StringData, DataType);  {Process data for SQL}
               BuildSqlDetail(Column, Data, SqlTmp1, SqlTmp2, 'Update', KeyFlag);
            end;
            CurNode := TSQLFieldInfo(CurNode)^.NextOne;
         end;
      end;
   end;
end;

{Delete button clicked - This uses the information collected from programmer defined
 functions.}
procedure TDynSQLClass.DeleteClickDynBuild(var DelTable, SqlTmp2: string; var SqlComp: Integer);
var
   CurNode: pointer;
   SqlTmp1, Data: string;
begin
   {Run the user's functions to build the information needed to create the SQL for the delete.}
   SQLRunFunction('Delete');

   {Initialize local variables}
   SqlTmp1 := '';  {First half of the user's data for the SQL statement.}
   Data := '';     {Specific data element.}
   CurNode := DynGetHeadPtr; {Set current node pointer to the first node.}
   DelTable := '';

   {Loop through InOutObject's and build column and data part of SQL Delete statement.}
   while (CurNode <> nil) do begin
      with TSQLFieldInfo(CurNode)^ do begin
         if (ComponentType = 'TQuery') then begin
            if (CompareText(ExtraParm, 'Delete') = 0) then begin
               SqlComp := CompNum;
            end;
         end
         else begin
            if (Table <> '') then begin
               DelTable := Table;
            end;
            Data := ProcessSQLData(StringData, DataType);  {Process data for SQL}
            BuildSqlDetail(Column, Data, SqlTmp1, SqlTmp2, 'Delete', KeyFlag);
         end;
         CurNode := TSQLFieldInfo(CurNode)^.NextOne;
      end;
   end;
end;

{------------------------------------------------------------------------------}
{                    Member functions to handle List Boxes                     }
{------------------------------------------------------------------------------}
{Set up the information on the listbox.}
procedure TDynSQLClass.SetupListBox(ComponentNum: integer);
var
   Data, Word: String;
   CurNode, next: pointer;
   InnerCnt: Integer;
begin
  {Will hold information about the list boxes.}
   if (MaxAvail < SizeOf(SQLListBox)) then begin
      MessageDlg('Not enough memory', mtWarning, [mbOk], 0);
   end
   else
   begin
      CurNode := new(SQLListBox);
      SQLListBox(CurNode)^.NextOne := nil;
      {Set the head pointer for the list box list}
      if (LBHeadPtr = nil) then begin
         LBHeadPtr := CurNode;
      end;
      {Set the last node pointer for the list box list}
      if (LBLastPtr = nil) then begin
         LBLastPtr := CurNode;
         next := nil;
      end
      else begin
         next := LBLastPtr;
         LBLastPtr := CurNode;
      end;
      {Update NextOne for the, now, second to last object}
      if (next <> nil) then begin
         SQLListBox(next)^.NextOne := CurNode;
      end;

      with TListBox(Components[ComponentNum]) do begin
         InnerCnt := 0;
         while (InnerCnt < Items.Count) do begin
            Data := Items[InnerCnt];
            with SQLListBox(CurNode)^ do begin
               CompNum := ComponentNum;  {Assign the component number.}
               Word := ReturnFirstWord(Data, ' ');
               Data := TakeFirstWordOff(Data, ' ');
               {What columns the list box displays - I only support 1 for now...}
               if (CompareText(Word,'Display') = 0) then begin
                  DispCols := Data;
               end
               {What detail is associated with the list box.}
               else if (CompareText(Word,'DataGroup') = 0) then begin
                  DataGroup := StrToInt(Data);
               end;
            end;
            inc(InnerCnt);
         end;
         Clear;
      end;
   end;
end;

end.

