{*********************************************************************************
 *                    Spotlite Component By Ken Friesen                          *
 *                         Creation Date Jan 9/97                                *
 *                                                                               *
 *  Graphic component with gradient fade - simulation  of a light beam fading    *
 *  into the background.  Good example of using the Palette Manager in 256 Color *
 *  Mode.  Use DisablePaint and EnablePaint if repeatedly repainting and using   *
 *  an older machine.  (Takes   some time to repaint)                            *
 *                                                                               *
 *********************************************************************************

                                                                                 }
unit Spotlite;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms,Typinfo, ExtCtrls, StdCtrls, Dialogs;

Type
  TDirection = (drWestToEast, drEastToWest, drNorthToSouth, drSouthToNorth);

type
  TSpotlite = class(TGraphicControl)
  private
    FColor: TColor;
    FStartColor : TColor;
    FEndColor: TColor;
    FBGColor: TColor;
    FDirection: TDirection;
    FTransparent: Boolean;
    FSP1,FSP2,FEP1,FEP2: Integer;
    StartClrR,StartClrG, StartClrB, EndClrR, EndClrG, EndClrB: Integer;
    StartClr,EndClr: LongInt;
    RedStep,GreenStep,BlueStep: Single;
    SLPalette: HPalette;
    NoColors,SysPalSize: Integer;
    HoldPaint: Boolean;

    procedure WMEraseBkgnd(var Message: TWMEraseBkgnd); message wm_EraseBkgnd;
    procedure WMPaint(var Message: TWMPaint); message WM_Paint;
    procedure SetColor(Color: TColor);
    procedure SetBGColor(Color: TColor);
    procedure SetDirection(Dir: TDirection);
    procedure SetSP1(Point: Integer);
    procedure SetSP2(Point: Integer);
    procedure SetEP1(Point: Integer);
    procedure SetEP2(Point: Integer);
    procedure SetTransparent(Value: Boolean);
    function LoadColors: HPalette;
    procedure delay(milliseconds: integer);
  protected
    procedure Paint; override;

  public
    Palette: HPalette;
    procedure DisablePaint;
    procedure EnablePaint;
    constructor Create(AComponent: TComponent); override;
    destructor Destroy; override;
  published
    property Color : TColor read FColor write SetColor;
    property BackGroundColor: TColor read FBGColor write SetBGColor;
    property Direction: TDirection read FDirection write SetDirection;
    property StartPoint1: Integer read FSP1 write SetSP1;
    property StartPoint2: Integer read FSP2 write SetSP2;
    property EndPoint1: Integer read FEP1 write SetEP1;
    property EndPoint2: Integer read FEP2 write SetEP2;
    property Align;
    property Visible;
    property Transparent: Boolean read FTransparent write SetTransparent;


  end;

procedure Register;

implementation

{$R SPOTLITE.RES}

{********************** Constructor Destructor Procs ***************************}

constructor TSpotlite.Create(AComponent: TComponent);
begin
  inherited Create(AComponent);
  Height:= 200;
  Width:= 200;
  FColor:= clYellow;
  FBGColor:= clBtnFace;
  FDirection:= drNorthToSouth;
  FSP1:= 0;
  FEP1:= 100;
  FSP2:= 20;
  FEP2:= Height;
  FTransparent:= False;
  HoldPaint:= False;
  ControlStyle:= Self.ControlStyle+ [csOpaque];
  SetColor(FColor);
end;

destructor TSpotlite.Destroy;
begin
  If SLPalette<>0 then DeleteObject(SLPalette);
  Inherited Destroy;
end;

procedure Register;
begin
  RegisterComponents('IA', [TSpotlite]);
end;

{ ***************************** Painting Procs *******************************}

function TSpotlite.LoadColors: HPALETTE;
var DstPal: PLogPalette;
    x,y, Size: integer;
    Worked: Boolean;
    Focus: HWND;
    DC: HDC;
    DevicePalSize: Integer;
    I: Integer;
begin
  Focus:= GetFocus;
  DC:= GetDC(Focus);
  try
    SysPalSize := GetDeviceCaps(DC, SIZEPALETTE);
    DevicePalSize:=GetDeviceCaps(DC,NUMCOLORS);
    If SysPalSize = 0 then NoColors:= 256 else NoColors:= SysPalSize-DevicePalSize;
    {Set Colors for the fade}
    StartClr:= ColorToRGB(FStartColor);
    StartClrR:= GetRValue(StartClr);
    StartClrG:= GetGValue(StartClr);
    StartClrB:= GetBValue(StartClr);
    EndClr:= ColorToRGB(FEndColor);
    EndClrR:= GetRValue(EndClr);
    EndClrG:= GetGValue(EndClr);
    EndClrB:= GetBValue(EndClr);
    RedStep:= (StartClrR- EndClrR) / NoColors;
    GreenStep:= (StartClrG- EndClrG) / NoColors;
    BlueStep:= (StartClrB- EndClrB) / NoColors;
    {Not in 256 Color mode - no need to create a palette}
    If (SysPalSize = 0) or (NoColors<0) then
    begin
      LoadColors:= 0;
      Exit;
    end;
    {If Palette=0 then Palette:= GetPalette;}{ Get the original palette to restore on SL Destroy}
    {Create the Palette}
    Size:= SizeOf(TLogPalette) + ((NoColors - 1) * SizeOf(TPaletteEntry));
    DstPal := MemAlloc(Size);
    try
      FillChar(DstPal^, Size, 0);
      with DstPal^ do
      begin
        palNumEntries := NoColors;
        palVersion := $0300;
        {Add sytem palette to my palette - probably shouldn't have to do as system
        palette Stock Palette entries should not be overwritable}
        SysPalSize:= GetSystemPaletteEntries(GetStockObject(Default_Palette), 0, 20, palPalEntry);
        {Set the colors}
        for x:= SysPalSize to NoColors+SysPalSize-1 do
        begin
          palPalEntry[x].peRed:= Abs(StartClrR - Trunc(x*RedStep));
          palPalEntry[x].peGreen:= Abs(StartClrG - Trunc(x*GreenStep));
          palPalEntry[x].peBlue:= Abs(StartClrB - Trunc(x*BlueStep));
          palPalEntry[x].peFlags:= 0;
        end;
      end;
      LoadColors := CreatePalette(DstPal^);
    finally
      FreeMem(DstPal, Size);
    end;
  finally
    ReleaseDC(Focus, DC);
  end;
end ;

procedure TSpotlite.Paint;
var x: integer;
    Bitmap, bmpMask: TBitmap;
    Step: Single;
begin
  If SLPalette<> 0 then
  begin
    SelectPalette(canvas.Handle,SLPalette,False);
    RealizePalette(canvas.handle);
  end;
  If (Direction= drNorthToSouth) or (Direction=drSouthToNorth) then
    Step := Height / NoColors  else Step := Width / NoColors;
  Bitmap:= TBitmap.Create;
  try
    Bitmap.Height:= Height;
    Bitmap.Width:= Width;
    with Bitmap.Canvas do
    begin
      If SLPalette<>0 then
      begin
        SelectPalette(Handle,SLPalette,False);
        RealizePalette(Handle);
      end;
      for x:= SysPalSize to NoColors+SysPalSize-1  do
      begin
        Brush.Color:= PaletteRGB((StartClrR - Trunc(x*RedStep)),
                                 (StartClrG - Trunc(x*GreenStep)),
                                 (StartClrB - Trunc(x*BlueStep)));
        If (Direction= drNorthToSouth) or (Direction=drSouthToNorth) then
          FillRect(Rect(0,Trunc(x*Step),Width,Trunc((x+1)*Step)))
        else FillRect(Rect(Trunc(x*Step),0, Trunc((x+1)*Step), Height));
      end;
      Brush.Color:= FBGColor;
      Pen.Color:= FBGColor;
      If (Direction= drNorthToSouth) or (Direction=drSouthToNorth) then
      begin
        Polygon([Point(0,0),Point(FSP1,0),Point(FEP1,Height),Point(0,Height)]);
        Polygon([Point(FSP2,0),Point(Width,0),Point(Width,Height),Point(FEP2,Height)]);
      end else begin
        Polygon([Point(0,0),Point(Width,0),Point(Width,FEP1),Point(0,FSP1)]);
        Polygon([Point(0,FSP2),Point(Width,FEP2),Point(Width,Height),Point(0,Height)]);
      end;
    end;
    If FTransparent then
    begin
      bmpMask:= TBitmap.Create;
      try
        bmpMask.Width:= Bitmap.Width;
        bmpMask.Height:= Bitmap.Height;
        bmpMask.Monochrome:= true;
        bmpMask.Canvas.Brush.Color:= FBGColor;
        SetBkColor(Bitmap.Canvas.Handle, ColorToRGB(FBGColor));
        BitBlt(bmpMask.Canvas.Handle,0,0,Bitmap.Width,Bitmap.Height, Bitmap.Canvas.Handle,0,0,SRCAND);
        BitBlt(Canvas.Handle,0,0,Bitmap.Width,Bitmap.Height, Bitmap.Canvas.Handle,0,0, SRCINVERT);
        BitBlt(Canvas.Handle,0,0,Bitmap.Width,Bitmap.Height, bmpMask.Canvas.Handle,0,0, SRCAND);
        BitBlt(Canvas.Handle,0,0,Bitmap.Width,Bitmap.Height, Bitmap.Canvas.Handle,0,0, SRCINVERT);
      finally
        bmpMask.Free;
      end;
    end else BitBlt(Canvas.Handle,0,0,Width,Height,Bitmap.Canvas.Handle,0,0,SRCCopy);
  finally
    Bitmap.Free;
  end;
end;


{*********************** Public Procs *********************************}

procedure TSpotlite.DisablePaint;
begin
  HoldPaint:= True;
end;

procedure TSpotlite.EnablePaint;
begin
  HoldPaint:= False;
  Repaint;
end;

{*****************  Property Setting Procs ********************************}

procedure TSpotlite.SetColor(Color: TColor);
var focus: hWnd;
    DC: HDC;
begin
  If (FDirection=drNorthToSouth) or (FDirection=drWestToEast) then
  begin
    FStartColor:= Color;
    FEndColor:= FBGColor;
  end else begin
    FEndColor:= Color;
    FStartColor:= FBGColor;
  end;
  FColor:= Color;
  If SLPalette<>0 then DeleteObject(SLPalette);
  SLPalette:= LoadColors;
  Repaint;
end;

procedure TSpotlite.SetBGColor(Color: TColor);
begin
 FBGColor:= Color;
 SetColor(FColor);
end;

procedure TSpotlite.SetDirection(Dir: TDirection);
begin
  FDirection:= Dir;
  SetColor(FColor);
end;

procedure TSpotlite.SetSP1(Point: Integer);
begin
  If (Direction= drNorthToSouth) or (Direction=drSouthToNorth) then
  begin
    If (Point<0)  then FSP1:=0 else FSP1:= Point;
  end
  else
  begin
    If (Point<0) then FSP1:=0 else FSP1:= Point;
  end;

  Repaint;
end;

procedure TSpotlite.SetSP2(Point: Integer);
begin
  If (Direction= drNorthToSouth) or (Direction=drSouthToNorth) then
  begin
    If (Point<0)  then FSP2:=width else FSP2:= Point;
  end
  else
  begin
    If (Point<0) then FSP2:=height else FSP2:= Point;
  end;
  Repaint;
end;

procedure TSpotlite.SetEP1(Point: Integer);
begin
  If (Direction= drNorthToSouth) or (Direction=drSouthToNorth) then
  begin
    If (Point<0)  then FEP1:=0 else FEP1:= Point;
  end
  else
  begin
    If (Point<0) then FEP1:=0 else FEP1:= Point;
  end;
  Repaint;
end;

procedure TSpotlite.SetEP2(Point: Integer);
begin
  If (Direction= drNorthToSouth) or (Direction=drSouthToNorth) then
  begin
    If (Point<0)  then FEP2:=Width else FEP2:= Point;
  end
  else
  begin
    If (Point<0) then FEP2:=Height else FEP2:= Point;
  end;
  Repaint;
end;

procedure TSpotlite.SetTransparent(Value: Boolean);
begin
  FTransparent:= Value;
  Repaint;
end;

{********************* Message Trapping Routines ****************************}

procedure TSpotlite.WMEraseBkgnd(var Message: TWMEraseBkgnd);
begin
  Message.Result:=1; {Don't erase the background}
end;

procedure TSpotlite.WMPaint(var Message: TWMPaint);
begin
  If HoldPaint then
  begin
    Message.Result:= 1;
    Exit;
  end;
  Inherited;
end;

{************************* Misc Procs *********************************}

procedure TSpotlite.delay(milliseconds: integer);
var x: longint;
begin
  x:= GetTickCount;
  while (GetTickCount-x)< milliseconds do Application.ProcessMessages;
end;


end.
