{# description
//
// GNU Regex 0.12 wrapper class
//
// Author:
// Ezio Vernacotola
// River Software House S.r.l.
// via degli Agricoltori, 57
// 00052 CIVITAVECCHIA (RM)
// ITALY
// email: ezio@etruria.net
//
// Date: 15 Sep 1996
//
}
{# freeware }

unit Regex;

interface

uses
  SysUtils, GnuRegex;

type

  TReSyntax  = (resyAwk, resyPosyxAwk, resyGrep, resyEgrep,
              resyPosixEgrep, resyPosixBasic, resyPosixExtended);

  TRegexException = class(Exception)
  end;

  TRegex = class(TObject)
  private
    FPatBuf : re_pattern_buffer;
    FPattern : string;
    FSyntax : TReSyntax;
    FResult : integer;
    FRegs : re_registers;
    FText : string;
    FPos : integer;

    procedure SetPattern( const sPat : string );
    procedure SetSyntax( Synt : TReSyntax );
    function GetSub( Idx : integer ) : string;
    function GetNumSubs : integer;
    function isFound : boolean;

  public
    property Pattern : string read FPattern write SetPattern;
    property Syntax : TReSyntax read FSyntax write SetSyntax;
    property Sub[Idx : integer] : string read GetSub; default;
    property Pos : integer read FResult;
    property NumSubs : integer read GetNumSubs;
    property Found : boolean read isFound;

    constructor create;
    destructor destroy; override;

    function match( const sText: string ) : integer;
    function search( const sText: string ) : integer;

  end;

{$ifdef OLD}
function ReMatch(sPattern : string; const sText: string; var aMatch : variant) : integer;
{$endif}

implementation


const
  SYNTAX_DEFAULT = resyPosixExtended;

{$ifdef OLD}
function ReMatch(sPattern : string; const sText: string; var aMatch : variant ) : integer;
var
  PatBuf : re_pattern_buffer;
  Regs : re_registers;
  iMatchRes : integer;
  CompRes : pchar;
  i : integer;
begin
  fillchar(PatBuf, sizeof(PatBuf), 0);
  re_set_syntax(SYNTAX_DEFAULT);
  CompRes := re_compile_pattern(pchar(sPattern), length(sPattern), PatBuf);
  if CompRes <> nil then        // non sono sicuro se questo deve andare dopo il try
    raise TRegexException.create(CompRes);
  try
    iMatchRes := re_match(PatBuf, pchar(sText), length(sText), 0, Regs );
    if iMatchRes >= 0 then
    begin
      aMatch := VarArrayCreate([0, PatBuf.re_nsub-1], varOleStr);
      for i:= 1 to PatBuf.re_nsub do
      begin
        aMatch[i-1] := copy(sText, regs.start[i]+1, regs.endr[i] - regs.start[i]);
      end;
    end;
    result := iMatchRes;
  finally
    regfree(PatBuf);
  end;
end;
{$endif}

procedure TRegex.SetPattern( const sPat : string );
var
  res : pchar;
begin
  res := re_compile_pattern(pchar(sPat), length(sPat), FPatBuf);
  if res <> nil then
    raise TRegexException.create(res);
  FPattern := sPat;
end;

procedure TRegex.SetSyntax( Synt : TReSyntax );
var
  sy : integer;
begin
  if Synt <> FSyntax then
  begin
    case Synt of
      resyAwk:
        sy := RE_SYNTAX_AWK;
      resyPosyxAwk:
        sy := RE_SYNTAX_POSIX_AWK;
      resyGrep:
        sy := RE_SYNTAX_GREP;
      resyEgrep:
        sy := RE_SYNTAX_EGREP;
      resyPosixEgrep:
        sy := RE_SYNTAX_POSIX_EGREP;
      resyPosixBasic:
        sy := RE_SYNTAX_POSIX_BASIC;
      resyPosixExtended:
        sy := RE_SYNTAX_POSIX_EXTENDED;
    else
      raise TRegexException.create('Unknow syntax');    
    end;
    re_set_syntax( sy );
    if FPattern <> '' then  // ricompila
      SetPattern(FPattern);
    FSyntax := Synt;
  end;
end;

function TRegex.GetSub( Idx : integer ) : string;
begin
  if (Idx < 0 ) or (Idx > GetNumSubs ) then
    raise TRegexException.create('Sub index out of range');
  result := copy(FText, FRegs.start[Idx]+1, FRegs.endr[Idx] - FRegs.start[Idx]);
end;

constructor TRegex.create;
begin
  inherited create;
  SetSyntax(SYNTAX_DEFAULT);
  FPos := -1;
end;

destructor TRegex.destroy;
begin
  regfree(FPatBuf);
  inherited destroy;
end;

function TRegex.match( const sText: string ) : integer;
begin
  FText := sText;
  FPos := re_match(FPatBuf, pchar(sText), length(sText), 0, FRegs );
  if Fpos = -2 then
    raise TRegexException.create('Internal error');  
  result := FPos;
end;

function TRegex.search( const sText: string ) : integer;
begin
  FText := sText;
  FPos := re_search(FPatBuf, pchar(sText), length(sText), 0, length(sText), FRegs );
  if Fpos = -2 then
    raise TRegexException.create('Internal error');  
  result := FPos;
end;

function TRegex.GetNumSubs : integer;
begin
  if FPos = -1 then
    result := 0
  else
    result := FPatBuf.re_nsub;
end;

function TRegex.isFound : boolean;
begin
  result := (FPos >= 0);
end;

end.
