{$B-,D-,L-,Y-}

unit skApollo;
interface

uses
   Apollo,
   {$IFDEF WIN32}
   Windows,
   {$ELSE}
   WinTypes,
   {$ENDIF}
   Classes,
   Forms, 
   SysUtils,
   SkTable;

type
   dBaseDateFormat = String[8];{CCYYMMDD}
   ESKIndexNotFound = class(Exception);

{Converts Delphi date to dBase}
function DTOS(D: TDateTime) : dBaseDateFormat;

{Tries to find the component owner form.
Returns nil if not found}
function GetOwnerForm(AComponent:TComponent) : TForm;

const
{ This constant was not added to DbiProcs.dcu }
   SDE_SP_PUTOBUFFER = 1007;
{Those two are the possible values for APOLLOENGINETYPE constant}
   SDE_ONLY = 1;
   SDE_BDE  = 2;

type
   {here we overwrite TskTable in the component library to add
   DoBeforeOpen method.}
   TskTable = class(skTable.TskTable)
      procedure DoBeforeOpen; override;
   end;{TskTable}

   { TskApollo - TApollo enhancement.
   ________________________
   Published Properties:
   None. Why waste .DFM file space?
   ________________________
      Public Properties:
    1. RecNum - allows developer to see the record number in the Watch Window;
    2. IndexCount - easier way to get the number of
       the open tags/single-index files.
    3. IndexOrder - easy way to set/view current index order.
    4. IndexExpr - easy way to set/view current index expression.
   ________________________
   Methods:
   FlushOBuffer - easy way to flush optimistic buffers when OptBuffering is on;
   CheckIndex / CheckIndexTag - check for the presence of the index/tag
      needed and create it only if it was not found - doesn't support
      empty(Roll-Your-Own) and descending indexes
   ________________________
   New constructor:
   Automatically sets gauge hook to the owner form's handle.}
   TskApollo = class(TApollo)
   private
      function GetIndexCount : LongInt;
      function LookForTag(sFileName, sTagName, sExpr: String;
         iOption : Integer; sCondition: String) : Boolean;
      function GetActive: Boolean;
      function GetIndexOrd: Integer;
      procedure SetIndexOrd(AOrder : Integer);
      function GetIndexExpr: String;
      procedure SetIndexExpr(AExpr: String);
   public
      constructor Create(AOwner: TComponent); override;
      function CheckIndex(sFileName, sExpr: String;
         iOption : Integer; bDescend: Bool;
         sCondition: String): Integer;
      function CheckIndexTag(sFileName, sTagName, sExpr: String;
         iOption: Integer; bDescend: Bool;
         sCondition: String): Integer;
      procedure FlushOBuffer;
      property Active : Boolean
         read GetActive;
      property RecNum : LongInt
         read RecNo
         write Go
         stored False;
      property IndexCount : LongInt
         read GetIndexCount
         stored False;
      property IndexOrder : Integer
         read GetIndexOrd
         write SetIndexOrd
         stored False;
      property IndexExpression : String
         read GetIndexExpr
         write SetIndexExpr
         stored False;
  end;{TskApollo}

 {TApolloEnv enhancement:
________________________
 Public Property:
 1. OptBuffering -provides an easy way to turn Optimistic Buffering on/off.
    It is not published so you won't see it in Object Inspector. Changing
    its value after the table(s) is(are) opened should have no effect
    so be careful.
________________________
New constructor:
Default OptBuffering setting is "off" rather then SuccessWare's "on".}
  TskApolloEnv = class(TApolloEnv)
  private
     function  GetOptBuffering : Bool;
     procedure SetOptBuffering(NewOptBuffering : Bool);
  public
     constructor Create(AOwner: TComponent); override;
     property OptBuffering : Bool
        read GetOptBuffering
        write SetOptBuffering
        stored False;
  end;{TskApolloEnv}

procedure Register;

implementation
uses DB, DbiProcs, dbTables, Dialogs;

function DTOS(D : TDateTime) : dBaseDateFormat;
begin
   Result := FormatDateTime('yyyymmdd', D)
end;{DTOS}

function GetOwnerForm(AComponent:TComponent) : TForm;
begin
   Result := nil;
   while (AComponent <> nil) and not (AComponent is TForm) do
      AComponent := AComponent.Owner;
end;{GetOwnerForm}

procedure SlashDir( var Dir: TFileName );
begin
  if Dir[Length(Dir)]<>'\' then AppendStr( Dir, '\' );
end;{SlashDir}

function ExtractNameOnly( const FileName : TFileName ) : TFileName;
begin
   Result := ExtractFileName( FileName );
   while Pos( '.', Result ) <> 0 do Delete( Result, Length( Result ), 1 )
end;{ExtractNameOnly}

constructor TskApollo.Create(AOwner : TComponent);
begin
   inherited Create(AOwner);
   AOwner := GetOwnerForm(AOwner);
   {GetOwnerForm may return a TForm object or nil}
   if Assigned(AOwner) then
      SetGaugeHook((AOwner as TForm).Handle)
end;{Create}

constructor TskApolloEnv.Create(AOwner : TComponent);
begin
   inherited Create(AOwner);
   OptBuffering := False
end;{Create}

function TskApolloEnv.GetOptBuffering : Bool;
begin
   Result := (sx_SysProp(SDE_SP_GETOBUFFER, nil) = 1)
end;{GetOptBuffering}

procedure TskApolloEnv.SetOptBuffering(NewOptBuffering : Bool);
begin
   if OptBuffering <> NewOptBuffering
      then sx_SysProp(SDE_SP_SETOBUFFER, Pointer(NewOptBuffering))
end;{SetOptBuffering}

function TskApollo.GetIndexCount : LongInt;
begin
   if Active then Result:=SysProp(SDE_SP_GETINDEXCOUNT, nil) else Result:=0
end;{GetIndexCount}

function TskApollo.LookForTag(sFileName, sTagName, sExpr: String;
   iOption : Integer; sCondition: String) : Boolean;
var
   i : Byte;
begin
   Result := False;
   for i := 1 to IndexCount do begin
      SetOrder(i);

      {IndexTag with blank index name defaults to the structural index}
      if (sFileName = '') and
          (AnsiCompareText(ExtractNameOnly(IndexName(i)),
                             ExtractNameOnly(BaseName)) <> 0)
          then Continue;
      if (sFileName <> '') and
         (AnsiCompareText(IndexName(i), sFileName) <> 0)
         then Continue; {Wrong file name}
      if (Length(sTagName) > 0) and
         (AnsiCompareText(sTagName, TagName(i)) <> 0)
         then Continue; {Wrong tag name}
      if AnsiCompareText(IndexKey, sExpr) <> 0
         then Continue; {Wrong expression}
      if AnsiCompareText(IndexCondition, sCondition) <> 0
         then Continue; {Wrong condition}
      if (Length(sCondition) = 0) and
         (((IndexType = INDEX_STANDARD_UNIQUE) and
             (iOption <> IDX_UNIQUE)) or
           ((IndexType = INDEX_STANDARD) and
             (iOption <> IDX_NONE)))
         then Continue{discrepancy in index type}
         else if (Length(sCondition) = 0) and
                 (((IndexType = INDEX_CONDITIONAL_UNIQUE) and
                     (iOption <> IDX_UNIQUE)) or
                   ((IndexType = INDEX_CONDITIONAL) and
                     (iOption <> IDX_NONE)))
         then Continue; {discrepancy in index type}
      Result := True;
      Break {if LookForTag returns True, the current order is already
             set to the needed one}
   end
end;{LookForTag}

function TskApollo.CheckIndex(sFileName, sExpr: String;
   iOption : Integer; bDescend: Bool; sCondition: String): Integer;
begin
   if LookForTag(sFileName, EmptyStr, sExpr, iOption, sCondition)
      then Result := IndexOrd
      else Result := Index(sFileName, sExpr, iOption,
                            bDescend, sCondition)
end;{CheckIndex}

function TskApollo.CheckIndexTag(sFileName, sTagName, sExpr: String;
   iOption: Integer; bDescend: Bool; sCondition: String): Integer;
begin
   if LookForTag(sFileName, sTagName, sExpr, iOption, sCondition)
      then Result := IndexOrd
      else Result := IndexTag(sFileName, sTagName, sExpr,
                               iOption, bDescend, sCondition)
end;{CheckIndexTag}

procedure TskApollo.FlushOBuffer;
begin
   SysProp(SDE_SP_PUTOBUFFER, nil)
end;{FlushOBuffer}

function TskApollo.GetActive : Boolean;
begin
   Result := False;
   {Function Assigned() will not work since DataSource and
    DataSource.DataSet are properties, not fields}
   if DataSource <> nil
      then if Self.DataSource.DataSet <> nil
         then Result := Self.DataSource.DataSet.Active
end;{GetActive}

function TskApollo.GetIndexOrd : Integer;
begin
   if Active then Result := IndexOrd else Result := 0
end;{GetIndexOrd}

procedure TskApollo.SetIndexOrd(AOrder : Integer);
begin
{Make sure the table is active}
   if Active then
   {if AOrder is different from the current one}
      if AOrder <> IndexOrd then
         SetOrder(AOrder)
{Note that there is no checking for existing of the open
index file under that order. Depending on ErrorLevel setting
attempt to set IndexOrder to nonexistent index will either generate
a run-time error or simply set the index order to zero (unsorted)
Since IndexCount is not always working properly, I restricted from
implementing such a check. Garbage in, cabbage out :) }
end;{SetIndexOrd}

function TskApollo.GetIndexExpr : String;
begin
   if Active then Result := IndexKey else Result := EmptyStr
end;{GetIndexExpr}

procedure TskApollo.SetIndexExpr(AExpr : String);
begin
   if LookForTag(EmptyStr, EmptyStr, AExpr, IDX_NONE, Filter)
      then Exit;{Standard index with such index expression is already open}
   if LookForTag(EmptyStr, EmptyStr, AExpr, IDX_UNIQUE, Filter)
      then Exit;{Unique index with such index expression is already open}
   if LookForTag(EmptyStr, EmptyStr, AExpr, IDX_EMPTY, Filter)
      then Exit;{RYO index with such index expression is already open}
   {if we got to this point then there is no currently open index with
    the same index expression}
   raise ESKIndexNotFound.Create('No current index with the expression ' + AExpr)
end;{SetIndexExpr}

procedure TskTable.DoBeforeOpen;
var
   dbfFile : TFileStream;
   CurDir  : TFileName;
   i, n    : Integer;
   dbfType : Byte;
begin
   inherited DoBeforeOpen;
   {DataBaseName property can contain the actual path, or the name of a
   TDataBase component. We handle both situations}

   {If DataBaseName property contains the name of the TDataBase component,
   try to get the actual path from it. It could be stuffed in TDataBase.Params
   property in the form of a string 'PATH=<filepath>'. WARNING:
   the first 5 letters of that string MUST be "path=" (case insensitive).
   No spaces allowed there!}
   CurDir := UpperCase(DatabaseName);
   if Length(CurDir) > 0 then begin
      for i := 0 to Session.DatabaseCount-1 do
         if ANSICompareText(CurDir, Session.Databases[i].DatabaseName) = 0
            then with Session.DataBases[i] do begin
               for n := 0 to Params.Count-1 do
               if ANSICompareText(System.Copy(Params[n], 1, 5),
                                   'PATH=') = 0{found the 'path=' statement}
                  then begin
                     CurDir := UpperCase(System.Copy(Params[n], 6,
                               System.Length(Params[n])- 5));
                     Break
                  end;{found the 'path=' statement}
               Break;
            end;{all set with loop on Session.Databases}
      end{Something was in DataBaseName}
      else GetDir(0, CurDir);

    SlashDir(CurDir);

    dbfType := $03;
    dbfFile := nil;
    {read the first byte from the header}
    try
       dbfFile := TFileStream.Create(CurDir + TableName, fmShareDenyNone);
       dbfFile.Read(dbfType, 1);
    finally
       {Free is supposed to test the object for nil-ness but just in case...}
       if dbfFile <> nil
          then dbfFile.Free
    end;
    {set the TableType property according to the header information.
     If the TableType will be set to ttdBase and the whole project
     is compiled under SDE_ONLY, an exception with the message
     "File open error" will be raised while trying to open the table}
    case dbfType of
       $03, $06, $86 :
          {if the user already set the TableType to the correct one
          he/she needs, don't change it, otherwise change it to ttSXFOX
          by default}
          if not(TableType in [ttSXNTX, ttAXNTX, ttSXNSX ])
             then TableType := ttSXFOX;
       $F5, $F6 :
          { FPT memo}
          TableType := ttSXFOX;
       $83 :
          {Clipper NTX memo}
          TableType := ttSXNTX;
       $E5, $E6 :
          {HiPerSIX memo}
          TableType := ttSXNSX;
       $43, $63, $8B, $CB :
          {these types are not supported by Apollo. If you run under SDE_BDE,
          you still can open the table under BDE, if you run under SDE_ONLY,
          you will get an exception}
          if APOLLOENGINETYPE = SDE_ONLY
             then raise Exception.Create(
                'This table type is not supported by SDE')
             else TableType := ttdBase;{SDE_BDE}
    end;{case dbfType}

    if TableType in [ ttSXNTX, ttAXNTX ]
       then begin{if NTX file with the same file name exists}
          if FileExists(CurDir + ExtractNameOnly(TableName) + '.NTX') and
             (Length(IndexName) = 0)
             then IndexName := ExtractNameOnly(TableName) + '.NTX'
       end;
end;{DoBeforeOpen}

procedure Register;
begin
   {I didn't try calling UnregisterClass for TskTable to ease switching
   to Apollo from BDE. Make sure you deinstall skTable from your component
   library before you install skApollo}
   RegisterComponents('Data Access', [ skApollo.TskTable, TskApollo, TskApolloEnv ]);
end;{Register}

end.
