unit EnhEdit;

{ TEnhancedEdit (C)opyright 2000 Version 1.46
  Autor : Simon Reinhardt
  eMail : reinhardt@picsoft.de
  Internet : http://www.picsoft.de

  Diese Komponente ist eine Ableitung von TEdit, welche eine Ausrichtung
  des Textes erlaubt und neben anderen zustzlichen Funktionen auch
  Value-Eigenschaften fr Zahlenwerte bietet. Sie ist Public Domain,
  das Urheberrecht liegt aber beim Autor. }

interface

{$I SRDefine.inc}

uses
  {$IFDEF SR_Win32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF} Messages, Classes,
  Controls, StdCtrls, Clipbrd, SysUtils, Forms, Graphics;

type
  TEnhancedEdit = class(TEdit)
  private
    FAlignment      : TAlignment;
    FCanvas         : TControlCanvas;
    FDecSeparator   : char;
    FDigits         : byte;
    FExitOnEnterKey : boolean;
    FFormat         : TFloatFormat;
    {$IFDEF SR_Delphi2_Up}
    FGrayDisabled   : boolean;
    {$ENDIF}
    FPrecision      : byte;
    FOldText        : string;
    FValue          : extended;
    FValueInt       : integer;

    function FormatEditString(AValue:extended):string;

  protected
    procedure CreateParams(var Params:TCreateParams); override;
    procedure KeyPress(var Key:Char); override;
    procedure Change; override;
    procedure SetAlignment(NewValue:TAlignment);
    procedure SetDecSeparator(NewValue:char);
    procedure SetDigits(NewValue:byte);
    procedure SetFormat(NewValue:TFloatFormat);
    {$IFDEF SR_Delphi2_Up}
    procedure SetGrayDisabled(NewValue:boolean);
    {$ENDIF}
    procedure SetPrecision(NewValue:byte);
    procedure SetValue(NewValue:extended);
    procedure SetValueInt(NewValue:integer);

    procedure CMEnter(var Msg: TCMGotFocus); message CM_ENTER;
    procedure CMExit(var Message: TWMNoParams); message CM_EXIT;
    procedure WMPaste(var Message: TWMPaste); message WM_PASTE;
    {$IFDEF SR_Delphi2_Up}
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    function GetTextMargins: TPoint;
    {$ENDIF}

  public
    constructor Create(AOwner:TComponent); override;
    procedure Clear; {$IFDEF SR_Delphi3_Up} override;{$ENDIF}

  published
    property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
    property DecSeparator: char read FDecSeparator write SetDecSeparator;
    property Digits: byte read FDigits write SetDigits default 2;
    property ExitOnEnterKey: boolean read FExitOnEnterKey write FExitOnEnterKey default true;
    property Format: TFloatFormat read FFormat write SetFormat;
    {$IFDEF SR_Delphi2_Up}
    property GrayDisabled: boolean read FGrayDisabled write SetGrayDisabled default true;
    {$ENDIF}
    property Precision: byte read FPrecision write SetPrecision default 8;
    property Value: extended read FValue write SetValue;
    property ValueInt: integer read FValueInt write SetValueInt;

  end;

procedure Register;

implementation

uses SRUtils;

{$IFDEF SR_Delphi1}
{$R *.D16}
{$ELSE}
{$R *.D32}
{$ENDIF}

{ Komponente TEnhancedEdit }
Constructor TEnhancedEdit.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  FAlignment:=taLeftJustify;
  FDecSeparator:=DecimalSeparator;
  FDigits:=2;
  FExitOnEnterKey:=true;
  FFormat:=ffNumber;
  {$IFDEF SR_Delphi2_Up}
  FGrayDisabled:=true;
  {$ENDIF}
  FPrecision:=8;
end;

procedure TEnhancedEdit.CreateParams(var Params: TCreateParams);
{$IFDEF SR_Delphi1}
const
   Alignments: array[TAlignment] of longint =
    (ES_LEFT,ES_RIGHT,ES_CENTER);
{$ELSE}
const
   Alignments: array[TAlignment] of DWord =
    (ES_LEFT,ES_RIGHT,ES_CENTER);
{$ENDIF}
begin
   inherited CreateParams(Params);
   Params.Style:=Params.Style or
                 ES_MULTILINE or
                 Alignments[FAlignment];
end;

function TEnhancedEdit.FormatEditString(AValue:extended):string;
var P : integer;
begin
  Result:=FloatToStrF(FValue,FFormat,FPrecision,FDigits);
  P:=Pos(DecimalSeparator,Result);
  if P>0 then
    Result[P]:=FDecSeparator;
end;

procedure TEnhancedEdit.Clear;
begin
  SetWindowText(Handle, '');
  Change;
end;

procedure TEnhancedEdit.KeyPress(var Key:Char);
begin
  inherited KeyPress(Key);
  if (Key=#13) or (Key=#10) then begin
    Key:=#0;
    if FExitOnEnterKey then
      (Parent as TControl).Perform(WM_NextDlgCtl, 0, 0);
  end;
  if Key=#27 then
    Text:=FOldText;
end;

procedure TEnhancedEdit.Change;
var P    : integer;
    Temp : string;
begin
  try
    Temp:=Text;
    P:=Pos(FDecSeparator,Temp);
    if P>0 then
      Temp[P]:=DecimalSeparator;
    FValue:=StrToFloatDef(Temp,0);
    FValueInt:=trunc(FValue);
  except
    FValue:=0;
    FValueInt:=0;
  end;
  inherited Change;
end;

procedure TEnhancedEdit.SetAlignment(NewValue:TAlignment);
begin
  if FAlignment<>NewValue then begin
    FAlignment:=NewValue;
    RecreateWnd;
  end;
end;

procedure TEnhancedEdit.SetDecSeparator(NewValue:char);
begin
  if FDecSeparator<>NewValue then begin
    FDecSeparator:=NewValue;
    try
      Text:=FormatEditString(FValue);
    except
      Text:='';
    end;
    Invalidate;
  end;
end;

procedure TEnhancedEdit.SetDigits(NewValue:byte);
begin
  if FDigits<>NewValue then begin
    FDigits:=NewValue;
    try
      Text:=FormatEditString(FValue);
    except
      Text:='';
    end;
    Invalidate;
  end;
end;

procedure TEnhancedEdit.SetFormat(NewValue:TFloatFormat);
begin
  if FFormat<>NewValue then begin
    FFormat:=NewValue;
    try
      Text:=FormatEditString(FValue);
    except
      Text:='';
    end;
    Invalidate;
  end;
end;

{$IFDEF SR_Delphi2_Up}
procedure TEnhancedEdit.SetGrayDisabled(NewValue:boolean);
begin
  if FGrayDisabled<>NewValue then begin
    FGrayDisabled:=NewValue;
    if not Enabled then
      Invalidate;
  end;
end;
{$ENDIF}

procedure TEnhancedEdit.SetPrecision(NewValue:byte);
begin
  if FPrecision<>NewValue then begin
    FPrecision:=NewValue;
    try
      Text:=FormatEditString(FValue);
    except
      Text:='';
    end;
    Invalidate;
  end;
end;

procedure TEnhancedEdit.SetValue(NewValue:extended);
begin
  if FValue<>NewValue then begin
    FValue:=NewValue;
    try
      Text:=FormatEditString(FValue);
    except
      Text:='';
    end;
    Invalidate;
  end;
end;

procedure TEnhancedEdit.SetValueInt(NewValue:integer);
begin
  if FValueInt<>NewValue then begin
    FValueInt:=NewValue;
    Text:=IntToStrFixed(FValueInt,FDigits);
    Invalidate;
  end;
end;

procedure TEnhancedEdit.CMEnter(var Msg:TCMGotFocus);
begin
  FOldText:=Text;
  if AutoSelect and not (csLButtonDown in ControlState) then
    SelectAll;
  inherited;
end;

procedure TEnhancedEdit.CMExit(var Message: TWMNoParams);
begin
  inherited;
end;

procedure TEnhancedEdit.WMPaste(var Message: TWMPaste);
var
  SGlobalHnd : THandle;
  Ptr        : PChar;
  Size,i     : Longint;
  s          : string;
begin
  s:='';
  if OpenClipboard(Handle) then begin
    try
      if Clipboard.HasFormat(CF_TEXT) then begin
        SGlobalHnd:=GetClipboardData(CF_TEXT);
        if SGlobalHnd<>0 then begin
          Size:=GlobalSize(SGlobalHnd);
          Ptr:=GlobalLock(SGlobalHnd);
          if Ptr<>nil then begin
            i:=0;
            while (i<size) and (Ptr[i]>=#32) do begin
              s:=s+Ptr[i];
              inc(i);
            end;
          end;
          GlobalUnlock(SGlobalHnd);
        end;
      end;
    finally
      CloseClipboard;
    end;
  end;
  SelText:=s;
end;

{$IFDEF SR_Delphi2_Up}
function TEnhancedEdit.GetTextMargins: TPoint;
var DC         : HDC;
    SaveFont   : HFont;
    i          : Integer;
    SysMetrics : TTextMetric;
    Metrics    : TTextMetric;
begin
  if NewStyleControls then begin
    if BorderStyle=bsNone then
      i:=0
    else
      if Ctl3D then
        i:=1
      else
        i:=2;
    Result.X:=SendMessage(Handle, EM_GETMARGINS, 0, 0) and $0000FFFF+i;
    Result.Y:=i;
  end
  else begin
    if BorderStyle = bsNone then
      i:=0
    else begin
      DC:=GetDC(0);
      GetTextMetrics(DC, SysMetrics);
      SaveFont:=SelectObject(DC, Font.Handle);
      GetTextMetrics(DC, Metrics);
      SelectObject(DC, SaveFont);
      ReleaseDC(0, DC);
      i:=SysMetrics.tmHeight;
      if i>Metrics.tmHeight then
        i:=Metrics.tmHeight;
      i:=i div 4;
    end;
    Result.X:=i;
    Result.Y:=i;
  end;
end;

procedure TEnhancedEdit.WMPaint(var Message: TWMPaint);
const AlignStyle : array[Boolean, TAlignment] of DWORD =
	 ((WS_EX_LEFT, WS_EX_RIGHT, WS_EX_LEFT),
          (WS_EX_RIGHT, WS_EX_LEFT, WS_EX_LEFT));
var Left       : Integer;
    Margins    : TPoint;
    R          : TRect;
    PS         : TPaintStruct;
    AAlignment : TAlignment;
    DC         : HDC;
    S          : string;
begin
  AAlignment := FAlignment;
  If FGrayDisabled then begin
    inherited;
    Exit;
  end;
  if FCanvas = nil then begin
    FCanvas := TControlCanvas.Create;
    FCanvas.Control := Self;
  end;
  DC:=Message.DC;
  if DC=0 then
    DC:=BeginPaint(Handle, PS);
  FCanvas.Handle:=DC;
  try
    FCanvas.Font:=Font;
    with FCanvas do begin
      R:=ClientRect;
      if not (NewStyleControls and Ctl3D) and (BorderStyle=bsSingle) then begin
        Brush.Color:=clWindowFrame;
        FrameRect(R);
        InflateRect(R, -1, -1);
      end;
      Brush.Color:=Color;
      if not Enabled then
        Font.Color:=clWindowText;
      S:=Text;
      if (csPaintCopy in ControlState) then begin
      case CharCase of
        ecUpperCase: S:=AnsiUpperCase(S);
        ecLowerCase: S:=AnsiLowerCase(S);
      end;
    end;
    if PasswordChar<>#0 then
      FillChar(S[1], Length(S), PasswordChar);
    Margins:=GetTextMargins;
    case AAlignment of
      taLeftJustify  : Left:=Margins.X;
      taRightJustify : Left:=ClientWidth-TextWidth(S)-Margins.X-1;
      else             Left:=(ClientWidth-TextWidth(S)) div 2;
    end;
    TextRect(R, Left, Margins.Y, S);
  end;
  finally
    FCanvas.Handle:=0;
    if Message.DC=0 then
      EndPaint(Handle, PS);
  end;
end;
{$ENDIF}

procedure Register;
begin
  RegisterComponents('Simon', [TEnhancedEdit]);
end;

end.
