{
 TScrollMessage: a scrolling windows to show messages,
                 including Text or Image.

                 Designed by Dan Ho
                danho@cs.nthu.edu.tw
}

unit Scrolmsg;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, Menus;

type
  TScrollDirection=(sdHorizontal,sdVertical);
  TScrollMessage = class(TGraphicControl)
  private
    { Private declarations }
    FColor:TColor;
    FParentColor:Boolean;
    FTextWindowLeft:integer;
    FTextWindowTop:integer;
    FTextToCenter:Boolean;
    FText:string;
    FBitmap:TBitmap;
    FTextBitmap:TBitmap;
    FTextBmp:TBitmap;
    FTimer:TTimer;
    FXPos,FYPos:Integer;
    FMemoryWidth,FMemoryHeight:Integer;
    FScrollDirection:TScrollDirection;
    FScrollInterval:Integer;
    FScrollStep:Integer;
    FScrollInverse:Boolean;
    FEnabled:Boolean;
    FMemoryBmp:TBitmap;
    FAlreadyIniMemoryBmp:Boolean;
    FAlreadyIniTextMemoryBmp:Boolean;
    FAutoSize:Boolean;
    FParentCenter:Boolean;
    FPause:Boolean;
    procedure SetBitmap(Value:TBitmap);
    procedure MoveImage(Sender:TObject);
    procedure SetScrollDirection(Value:TScrollDirection);
    procedure SetScrollInterval(Value:Integer);
    procedure SetEnabled(Value:Boolean);
    procedure SetAutoSize(Value:Boolean);
    procedure SetParentCenter(Value:Boolean);
    procedure SetPause(Value:Boolean);
    procedure SetText(Value:string);
    procedure IniMemoryBmp;
    procedure IniTextMemoryBmp;
    procedure SetTextBitmap;
    function  GetColor:TColor;
    procedure SetColor(Value:TColor);
    procedure CMFontChanged(var Message:TMessage); message CM_FONTCHANGED;
    procedure SetParentColor(Value:Boolean);
    procedure SetTextWindowLeft(Value:Integer);
    procedure SetTextWindowTop(Value:Integer);
    procedure SetTextToCenter(Value:Boolean);
  protected
    { Protected declarations }
    procedure Paint;override;
  public
    { Public declarations }
    constructor Create(AOwner:TComponent);override;
    destructor Destroy;override;
    procedure Loaded;override;
  published
    { Published declarations }
    property Width default 100;
    property Height default 100;
    property Align;
    property ParentColor read FParentColor write SetParentColor default True;
    property Color read GetColor write SetColor stored True;
    property Font;
    property Text:string read FText write SetText;
    property Bitmap:TBitmap read FBitmap write SetBitmap;
    property AutoSize:Boolean read FAutoSize write SetAutoSize default False;
    property ScrollDirection:TScrollDirection read FScrollDirection write SetScrollDirection
               default sdHorizontal;
    property ScrollInterval:Integer read FScrollInterval write SetScrollInterval
               default 50;
    property ScrollStep:Integer read FScrollStep write FScrollStep default 1;
    property ScrollInverse:Boolean read FScrollInverse write FScrollInverse default False;
    property Enabled:Boolean read FEnabled write SetEnabled default True;
    property Visible;
    property ShowHint;
    property ParentShowHint;
    property ParentCenter:Boolean read FParentCenter write SetParentCenter default False;
    property Pause:Boolean read FPause write SetPause default False;
    property TextWindowLeft:integer read FTextWindowLeft write SetTextWindowLeft default 10;
    property TextWindowTop:integer read FTextWindowTop write SetTextWindowTop default 10;
    property TextToCenter:Boolean read FTextToCenter write SetTextToCenter default True;
    property PopupMenu;
    property Tag;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Dan', [TScrollMessage]);
end;

constructor TScrollMessage.Create(AOwner:TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle + [csFramed, csOpaque];
   FBitmap:=TBitmap.Create;
   FTextBitmap:=TBitmap.Create;
   FTimer:=TTimer.Create(Self);
   FScrollInterval:=50;
   FTimer.Interval:=FScrollInterval;
   FEnabled:=True;
   FTimer.Enabled:=FEnabled;
   FTimer.OnTimer:=MoveImage;
   Width:=100;
   Height:=100;
   FXPos:=0;
   FYPos:=0;
   FScrollDirection:=sdHorizontal;
   FScrollStep:=1;
   FScrollInverse:=False;
   FMemoryBmp:=TBitmap.Create;
   FAlreadyIniMemoryBmp:=False;
   FAlreadyIniTextMemoryBmp:=False;
   FAutoSize:=False;
   FParentCenter:=False;
   FPause:=False;
   FColor:=clWindow;
   FParentColor:=True;
   FTextWindowLeft:=10;
   FTextWindowTop:=10;
   FTextToCenter:=True;
end;

destructor TScrollMessage.Destroy;
begin
   FBitmap.Free;
   FTextBitmap.Free;
   FMemoryBmp.Free;
   inherited Destroy;
end;

{This makes Bitmap has higher precedence than Text}
procedure TScrollMessage.Loaded;
begin
   if (FTextToCenter) and (Text<>'') then SetTextToCenter(True);
   if Text<>'' then SetTextBitmap;
   if not FBitmap.Empty then IniMemoryBmp;
end;

procedure TScrollMessage.SetBitmap(Value:TBitmap);
begin
   FBitmap.Assign(Value);
   IniMemoryBmp;
   Repaint;
end;

procedure TScrollMessage.SetTextBitmap;
begin
   FTextBitmap.Width:=Width;
   FTextBitmap.Height:=Height;
   FTextBitmap.Canvas.Brush.Color:=Color;
   FTextBitmap.Canvas.FillRect(Rect(0,0,FTextBitmap.Width,FTextBitmap.Height));
   FTextBitmap.Canvas.Font.Assign(Font);
   FTextBitmap.Canvas.TextOut(FTextWindowLeft,FTextWindowTop,FText);
   IniTextMemoryBmp;
end;

procedure TScrollMessage.SetText(Value:string);
begin
   FText:=Value;
   if (FTextToCenter) then SetTextToCenter(True);
   SetTextBitmap;
   Invalidate;
end;

procedure TScrollMessage.SetColor(Value:TColor);
begin
   if (Value<>FColor) then
      begin
         FColor:=Value;
         ParentColor:=False;
         SetTextBitmap;
         Invalidate
      end
end;

function TScrollMessage.GetColor:TColor;
begin
   if ParentColor then
      begin
         if FColor<>Parent.Brush.Color then
            FColor:=Parent.Brush.Color;
         Result:=Parent.Brush.Color
      end
   else
      Result:=FColor;
end;

procedure TScrollMessage.SetParentColor(Value:Boolean);
begin
   if (Value<>FParentColor) then
      begin
         FParentColor:=Value;
         if FParentColor then
            Color:=Parent.Brush.Color;
      end
end;

procedure TScrollMessage.CMFontChanged(var Message:TMessage);
begin
   inherited;
   if (Text<>'') then
      begin
         if (FTextToCenter) then SetTextToCenter(True);
         SetTextBitmap;
         Invalidate
      end
end;

procedure TScrollMessage.IniTextMemoryBmp;
begin
   if (FTextBitmap.Empty or (Text=''))then exit;
   FAlreadyIniTextMemoryBmp:=True;
   FAlreadyIniMemoryBmp:=False;

   { for AutoSize, adjust the control to fit the bitmap,
     but if Align is not alNone, fit the align property }
   if AutoSize and (Align=alNone) then
      SetBounds(Left,Top,FTextBitmap.Width,FTextBitmap.Height);

   FMemoryBmp.Width:=FTextBitmap.Width;
   FMemoryBmp.Height:=FTextBitmap.Height;
   FMemoryBmp.Canvas.Draw(0,0,FTextBitmap);
   FMemoryWidth:=FTextBitmap.Width;
   FMemoryHeight:=FTextBitmap.Height;
end;

procedure TScrollMessage.IniMemoryBmp;
var
   OldPos,I:Integer;
   HorizontalCount:Integer;
begin
   if FBitmap.Empty then exit;
   FAlreadyIniMemoryBmp:=True;
   FAlreadyIniTextMemoryBmp:=False;

   { for AutoSize, adjust the control to fit the bitmap,
     but if Align is not alNone, fit the align property }
   if AutoSize and (Align=alNone) then
      SetBounds(Left,Top,FBitmap.Width,FBitmap.Height);

   FMemoryBmp.Width:=FBitmap.Width;
   FMemoryBmp.Height:=FBitmap.Height;
   FMemoryBmp.Canvas.Draw(0,0,FBitmap);
   HorizontalCount:=1;
   while (FMemoryBmp.Width<Width) do
      begin
         OldPos:=FMemoryBmp.Width;
         Inc(HorizontalCount);
         FMemoryBmp.Width:=FMemoryBmp.Width+FBitmap.Width;
         FMemoryBmp.Canvas.Draw(OldPos,0,FBitmap);
      end;
   while (FMemoryBmp.Height<Height) do
      begin
         OldPos:=FMemoryBmp.Height;
         FMemoryBmp.Height:=FMemoryBmp.Height+FBitmap.Height;
         for I:=1 to HorizontalCount do
            FMemoryBmp.Canvas.Draw(FBitmap.Width*(I-1),OldPos,FBitmap);
      end;
   FMemoryWidth:=FMemoryBmp.Width;
   FMemoryHeight:=FMemoryBmp.Height;
end;

procedure TScrollMessage.Paint;
var
   PaintBmp:TBitmap;
   DestRect,SourceRect:TRect;
   OldPos,I:Integer;
   HorizontalCount:Integer;
begin
   { Draw A rectangle when Designing }
   if csDesigning in ComponentState then
      with Canvas do
      begin
         Pen.Style:=psDash;
         Brush.Style:=bsClear;
         Rectangle(0,0,Width,Height);
      end;

   if (FBitmap.Empty and (Text='')) then exit;
   {Run-time: only paint when not initialize memory bitmap,
    Design-time: always repaint}
   if (((not FAlreadyIniTextMemoryBmp) and FBitmap.Empty) or (csDesigning in ComponentState)) then
      if (Text<>'') then
         begin
            if (FTextToCenter) then SetTextToCenter(True);
            SetTextBitmap
         end;
   if ((not FAlreadyIniMemoryBmp) or (csDesigning in ComponentState)) then
      if not FBitmap.Empty then IniMemoryBmp;

   PaintBmp:=TBitmap.Create;
   PaintBmp.Width:=Width;
   PaintBmp.Height:=Height;
   PaintBmp.Canvas.CopyMode:=cmSrcCopy;
   if (ScrollDirection=sdHorizontal) then
      begin
         if (FMemoryBmp.Width-FXPos)>=PaintBmp.Width then
            begin
               PaintBmp.Canvas.CopyRect(Rect(0,0,Width,Height),FMemoryBmp.Canvas,
                    Rect(FXPos,FYPos,FXPos+Width,FYPos+Height));
            end
         else
            begin
               PaintBmp.Canvas.CopyRect(Rect(0,0,FMemoryBmp.Width-FXPos,Height),FMemoryBmp.Canvas,
                    Rect(FXPos,0,FMemoryBmp.Width,Height));
               PaintBmp.Canvas.CopyRect(Rect(FMemoryBmp.Width-FXPos,0,Width,Height),FMemoryBmp.Canvas,
                    Rect(0,FYPos,Width-(FMemoryBmp.Width-FXPos),FYPos+Height));
            end;
      end
   else      { for ScrollDirection=sdVertical }
      begin
         if (FMemoryBmp.Height-FYPos)>=PaintBmp.Height then
            begin
               PaintBmp.Canvas.CopyRect(Rect(0,0,Width,Height),FMemoryBmp.Canvas,
                    Rect(FXPos,FYPos,FXPos+Width,FYPos+Height));
            end
         else
            begin
               PaintBmp.Canvas.CopyRect(Rect(0,0,Width,FMemoryBmp.Height-FYPos),FMemoryBmp.Canvas,
                    Rect(0,FYPos,Width,FMemoryBmp.Height));
               PaintBmp.Canvas.CopyRect(Rect(0,FMemoryBmp.Height-FYPos,Width,Height),FMemoryBmp.Canvas,
                    Rect(0,0,Width,Height-(FMemoryBmp.Height-FYPos)));
            end;
      end;

   SourceRect:=Rect(0,0,Width,Height);
   DestRect:=Rect(0,0,Width,Height);
   Canvas.CopyRect(DestRect,PaintBmp.Canvas,SourceRect);

   PaintBmp.Free;
end;

procedure TScrollMessage.SetAutoSize(Value:Boolean);
begin
   if (Value<>FAutoSize) then
      begin
         FAutoSize:=Value;
         IniMemoryBmp;
         Invalidate;
      end
end;

procedure TScrollMessage.MoveImage(Sender:TObject);
begin
   if ((csDesigning in ComponentState) or (Bitmap.Empty and (Text=''))) then exit;
   if (ScrollDirection=sdHorizontal) then
      begin
         if (FScrollInverse) then
            begin
               FXPos:=FXPos-ScrollStep;
               while FXPos<0 do
                  FXPos:=FXPos+FMemoryWidth
            end
         else
            begin
               FXPos:=FXPos+ScrollStep;
               while FXPos>FMemoryWidth do
                  FXPos:=FXPos-FMemoryWidth
            end
      end
   else
      begin
         if (FScrollInverse) then
            begin
               FYPos:=FYPos-ScrollStep;
               while FYPos<0 do
                  FYPos:=FYPos+FMemoryHeight
            end
         else
            begin
               FYPos:=FYPos+ScrollStep;
               while FYPos>FMemoryHeight do
                  FYPos:=FYPos-FMemoryHeight
            end
      end;
   Repaint;
end;

procedure TScrollMessage.SetScrollDirection(Value:TScrollDIrection);
begin
   if (Value<>FScrollDirection) then
      begin
         FScrollDirection:=Value;
         FXPos:=0;
         FYPos:=0;
         Repaint;
      end;
end;

procedure TScrollMessage.SetScrollInterval(Value:Integer);
begin
   if (Value<>ScrollInterval) then
      begin
         FScrollInterval:=Value;
         FTimer.Interval:=FScrollInterval;
      end;
end;

procedure TScrollMessage.SetEnabled(Value:Boolean);
begin
   if (Value<>Enabled) then
      begin
         FEnabled:=Value;
         FTimer.Enabled:=Value;
      end
end;

procedure TScrollMessage.SetParentCenter(Value:Boolean);
begin
   if (Value<>FParentCenter) then
      begin
         FParentCenter:=Value;
         if (FParentCenter) then
            begin
               if Width<=Parent.ClientWidth then
                  Left:=(Parent.ClientWidth-Width) div 2;
               if Height<=Parent.ClientHeight then
                  Top:=(Parent.ClientHeight-Height) div 2;
            end;
      end;
end;

procedure TScrollMessage.SetPause(Value:Boolean);
begin
   if (Value<>FPause) then
      begin
         FPause:=Value;
         FTimer.Enabled:=not Value;
      end
end;

procedure TScrollMessage.SetTextWindowLeft(Value:integer);
begin
   if (Value<>FTextWindowLeft) and (Value>=0) then
      begin
         FTextWindowLeft:=Value;
         FTextToCenter:=False;
         SetTextBitmap;
         Invalidate
      end;
end;

procedure TScrollMessage.SetTextWindowTop(Value:integer);
begin
   if (Value<>FTextWindowTop) and (Value>=0) then
      begin
         FTextWindowTop:=Value;
         FTextToCenter:=False;
         SetTextBitmap;
         Invalidate
      end;
end;

procedure TScrollMessage.SetTextToCenter(Value:Boolean);
var
   MsgWidth,MsgHeight:Integer;
begin
   FTextToCenter:=Value;
   if (Text<>'') then
      begin
         Canvas.Font.Assign(Font);
         {for calculation of TextWidth & TextHeight}
         MsgWidth:=Canvas.TextWidth(Text);
         MsgHeight:=Canvas.TextHeight(Text);
         if MsgWidth<=Width then
            FTextWindowLeft:=(Width-MsgWidth) div 2;
         if MsgHeight<=Height then
            FTextWindowTop:=(Height-MsgHeight) div 2;
      end;
end;

end.
