unit Sclimage;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls, Menus,
  Forms, Dialogs, ExtCtrls, StdCtrls;

Type
    TChangeView = (VShow, VHide);
    TScaleStyle = (ssNormal, ssScale, ssStretchXY, ssStretchX, ssStretchY);

  TScaleImage = class(TGraphicControl)
    procedure Loaded; override;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  private
    FScaleStyle: TScaleStyle;
    FBorder,
    FCenter: Boolean;
    FPicture: TPicture;
    function GetNormal: Boolean;
    function GetStretch: Boolean;
    function GetScale: Boolean;
    procedure SetScaleStyle(Value: TScaleStyle);
    procedure SetCenter(Value: Boolean);
    procedure SetPicture(Value: TPicture);
    procedure SetBorder(Value: Boolean);
  { *** GLOBAL VARIABLES ***}
  private
    ScaleBmp: TPicture;
    ImageFile: String;
    PictWidth,
    PictHeight: Integer;
    SizeRatio: Real;
    procedure DrawNormal;
    procedure DrawScaled;
    procedure DrawXYStretched;
    procedure DrawXStretched;
    procedure DrawYStretched;
    procedure ChangeView(View: TChangeView);
    procedure GetPictSize;
  protected
    { Protected declarations }
    procedure Paint; override;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
  public
    property IsStretched: Boolean read GetStretch;
    property IsScaled: Boolean read GetScale;
    property IsNormal: Boolean read GetNormal;
  published
    { Published declarations }
    property Picture: TPicture read FPicture write SetPicture;
    property ScaleStyle: TScaleStyle read FScaleStyle write SetScaleStyle default ssNormal;
    property Center: Boolean read FCenter write SetCenter default True;
    property Align;
    property Color;
    property Border: Boolean read FBorder write SetBorder default True;
    property DragCursor;
    property DragMode;
    property PopupMenu;
    property ParentColor;
    property ParentShowHint;
    property ShowHint;
    property Visible;
    property Enabled;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Samples', [TScaleImage]);
end;

procedure TScaleImage.Loaded;
begin
     inherited Loaded;
     GetPictSize;
end;

constructor TScaleImage.Create(AOwner: TComponent);
begin
    inherited Create(AOwner);
    TWinControl(AOwner).InsertControl(Self);
    ControlStyle := ControlStyle + [csOpaque];
    ScaleBmp := TPicture.Create;
    FPicture := TPicture.Create;
    FCenter := True;
    FBorder := True;
    FScaleStyle := ssNormal;
    Height := 100;
    Width := 100;
end;

destructor TScaleImage.Destroy;
begin
    ScaleBmp.Destroy;
    FPicture.Destroy;
    Inherited Destroy;
end;

function TScaleImage.GetStretch: Boolean;
begin
    if Ord(FScaleStyle) >= 2 then
        Result := True
    else
        Result := False;
end;

function TScaleImage.GetNormal: Boolean;
begin
    if FScaleStyle = ssNormal then
        Result := True
    else
        Result := False;
end;

function TScaleImage.GetScale: Boolean;
begin
    if FScaleStyle = ssScale then
        Result := True
    else
        Result := False;
end;

procedure TScaleImage.SetScaleStyle(Value: TScaleStyle);
begin
    if FScaleStyle <> Value then
    begin
        FScaleStyle := Value;
        Invalidate;
    end;
end;

procedure TScaleImage.SetCenter(Value: Boolean);
begin
    if FCenter <> Value then
    begin
        FCenter := Value;
        Invalidate;
    end;
end;

procedure TScaleImage.SetPicture(Value: TPicture);
begin
    if Value.Graphic is TIcon then
    begin
        MessageDlg('You may only use .BMP or .WMF images.',mtWarning,[mbOK],0);
        Picture := FPicture;
        Exit;
    end;
    if FPicture <> Value then
    begin
        FPicture.Assign(Value);
        ScaleBmp.Assign(Value);
        Invalidate;
    end;
end;

procedure TScaleImage.SetBorder(Value: Boolean);
begin
    if FBorder <> Value then
    begin
        FBorder := Value;
        Invalidate;
    end;
end;

procedure TScaleImage.Paint;
var
    Rect: TRect;
begin
    Rect := ClientRect;
    Canvas.CopyMode := cmSrcCopy;
    Canvas.Pen.Color := Self.Color;
    Canvas.Brush.Color := Self.Color;
    Canvas.FillRect(Rect);
    GetPictSize;
    ScaleBmp.Assign(FPicture);
    case Ord(FScaleStyle) of
        0: DrawNormal;
        1: DrawScaled;
        2: DrawXYStretched;
        3: DrawXStretched;
        4: DrawYStretched;
    end;
    Canvas.Brush.Color := clBlack;
    if FBorder then
        Canvas.FrameRect(Rect);
end;

procedure TScaleImage.WMPaint(var Message: TWMPaint);
begin
    if Message.DC <> 0 then
    begin
        Canvas.Handle := Message.DC;
        try
            Paint;
        finally
            Canvas.Handle := 0;
        end;
    end;
end;

procedure TScaleImage.DrawNormal;
var
    CenterW, CenterH: Integer;
begin
    if Center then
    begin
        CenterW := (Width - PictWidth) div 2;
        CenterH := (Height - PictHeight) div 2;
    end
    else
    begin
        CenterW := 0;
        CenterH := 0;
    end;
    Canvas.Draw(CenterW,CenterH,ScaleBmp.Bitmap);
end;

procedure TScaleImage.DrawScaled;
var
   RatioW, RatioH: Real;
   SizeW, SizeH: Integer;
   CenterW, CenterH: Integer;
   TmpBmp: TBitmap;
begin
    if FPicture.Bitmap.Empty then Exit;
    TmpBmp := TBitmap.Create;
    TmpBmp.Assign(ScaleBmp.Bitmap);
    Canvas.Draw((Width * 2),(Height * 2),ScaleBmp.Bitmap);
    RatioW := Width / PictWidth;
    RatioH := Height / PictHeight;
    if RatioW > RatioH then
        SizeRatio := RatioH
    else
         SizeRatio := RatioW;
    SizeW := Trunc(PictWidth * SizeRatio);
    SizeH := Trunc(PictHeight * SizeRatio);
    if SizeW > Width then
        SizeW := Width;
    if SizeH > Height then
        SizeH := Height;
    if Center then
    begin
        CenterW := (Width - SizeW) div 2;
        if CenterW < 0 then
            CenterW := 0;
        CenterH := (Height - SizeH) div 2;
        if CenterH < 0 then
            CenterH := 0;
    end
    else
    begin
        CenterW := 0;
        CenterH := 0;
    end;
    TmpBmp.Height := SizeH;
    TmpBmp.Width := SizeW;
    SetStretchBltMode(TmpBmp.Canvas.Handle, STRETCH_DELETESCANS);
    if SizeRatio < 1 then
    begin
         with TmpBmp.Canvas do
              StretchBlt(Handle,0,0,SizeW,SizeH,ScaleBmp.Bitmap.Canvas.Handle,0,0,PictWidth,PictHeight,SRCCOPY);
    end
    else
    begin
        if SizeRatio > 1 then
        begin
            ScaleBmp.Bitmap.Height := SizeH;
            ScaleBmp.Bitmap.Width := SizeW;
            with TmpBmp.Canvas do
                StretchBlt(Handle,0,0,SizeW,SizeH,Handle,0,0,PictWidth,PictHeight,SRCCOPY);
        end
        else
        begin
            with TmpBmp.Canvas do
                StretchBlt(Handle,0,0,SizeW,SizeH,ScaleBmp.Bitmap.Canvas.Handle,0,0,PictWidth,PictHeight,SRCCOPY);
        end;
    end;
    ScaleBmp.Assign(TmpBmp);
    Canvas.Draw(CenterW,CenterH,ScaleBmp.Bitmap);
    TmpBmp.Destroy;
end;

procedure TScaleImage.DrawXYStretched;
var
    Rect: TRect;
begin
    Rect := ClientRect;
    Canvas.StretchDraw(Rect,ScaleBmp.Bitmap);
end;

procedure TScaleImage.DrawXStretched;
var
    Rect: TRect;
    CenterH: Integer;
begin
    if Center then
        CenterH := (Height - PictHeight) div 2
    else
        CenterH := 0;
    Rect := ClientRect;
    Rect.Top := CenterH;
    Rect.Bottom := Rect.Top + PictHeight;
    Canvas.StretchDraw(Rect,ScaleBmp.Bitmap);
end;

procedure TScaleImage.DrawYStretched;
var
    Rect: TRect;
    CenterW: Integer;
begin
    if Center then
        CenterW := (Width - PictWidth) div 2
    else
        CenterW := 0;
    Rect := ClientRect;
    Rect.Left := CenterW;
    Rect.Right := Rect.Left + PictWidth;
    Canvas.StretchDraw(Rect,ScaleBmp.Bitmap);
end;

procedure TScaleImage.GetPictSize;
begin
     PictHeight := FPicture.Bitmap.Height;
     PictWidth := FPicture.Bitmap.Width;
end;

procedure TScaleImage.ChangeView(View: TChangeView);
begin
     if View = VHide then
     begin
          Hide;
          Update;
     end
     else
     begin
         Show;
         Update;
     end;
end;

end.
