{--------------------------------------------------------------------------


  RZNEdit, a component with edit controls for 

    real numbers      -> TRealEdit

    integers          -> TIntegerEdit

    and

    positive integers -> TCardinalEdit
    
  Version 1.1

  This is my version of a numeric input component. It comes with three
  types, each handling its special input range ( real, integer and
  cardinal ).
  All input is watched, a TIntegerEdit accepts only characters in
  the set [-,0,1,..,9] for instance.
  The clipboard is supported.
  An empty input is interpreted as 0.
  For TRealEdit and TIntegerEdit an input containing only a plus-sign
  or a minus-sign is also regarded as 0.
  Ancestor class is the TCustomEdit, new properties are:

    FormatString : is used in the FormatFloat-routine from Delphi
    MaxValue \
              : determine the input boundaries, if Validate is True  
    MinValue /
    Validate  : determines if input is checked against boundaries
    Value     : the result 


  This is my adaption of several numerical input controls, which I
  found on the Delphi Super Page. I give credits to DaveH, Arnt Kern,
  Richard Vowles and Robert Vivrette.

  
  rk 08.02.96

  Robert Kratz                                  kratz@labs.polycnrs-gre.fr
 
 --------------------------------------------------------------------------}

Unit RZNEdit;

Interface

{--------------------------------------------------------------------------}

Uses WinTypes, Classes, StdCtrls, ExtCtrls, Controls, Messages, SysUtils,
  Forms, Graphics, Menus, Buttons, Spin, Clipbrd, Dialogs;

{--------------------------------------------------------------------------}

Type

 TRealEdit = class(TCustomEdit)
  private
    FFormatString, VText : String;
    FMinValue, FMaxValue : Extended;
    FValidate            : Boolean;

    procedure SetValue( NewValue : Extended );
    Function  GetValue : Extended;
    function  CheckValue( NewValue : Extended ) : Extended;

    Procedure SetValidate( B :  Boolean );
    Procedure SetFormatString( A : String );

    Function  SyntaxOk : Boolean;
    Function  EmptyOrSign : Boolean;

    procedure CMEnter( var Message : TCMGotFocus ); message CM_ENTER;
    procedure CMExit( var Message : TCMExit );   message CM_EXIT;

  protected
    function  IsValidChar( Key : Char) : Boolean; virtual;
    Procedure KeyDown( Var Key : Word; Shift : TShiftState ); Override;
    procedure KeyPress( var Key : Char ); override;
  public
    constructor Create( AOwner : TComponent ); override;
  published
    property AutoSelect;
    property AutoSize;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property Cursor;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    Property FormatString : String read  FFormatString
                                   write SetFormatString;
    property Height;
    property HelpContext;
    property Hint;
    property Left;
    property MaxLength;
    property MaxValue : Extended read FMaxValue write FMaxValue;
    property MinValue : Extended read FMinValue write FMinValue;
    property Name;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Tag;
    property Validate : Boolean read FValidate write SetValidate;
    property Value : Extended read GetValue write SetValue;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

{--------------------------------------------------------------------------}

  TIntegerEdit = class(TCustomEdit)
  private
    FFormatString, VText : String;
    FMinValue, FMaxValue : Longint;
    FValidate            : Boolean;

    procedure SetValue( NewValue : Longint );
    Function  GetValue : Longint;
    function  CheckValue( NewValue : Longint ): Longint;

    procedure SetMaxValue( NewValue : Longint );   Virtual;
    procedure SetMinValue( NewValue : Longint );   Virtual;

    Procedure SetValidate( B :  Boolean );
    Procedure SetFormatString( A : String );

    Function  SyntaxOk : Boolean;
    Function  EmptyOrSign : Boolean;

    procedure CMEnter( var Message : TCMGotFocus ); message CM_ENTER;
    procedure CMExit( var Message : TCMExit );   message CM_EXIT;

  protected
    function  IsValidChar( Key : Char) : Boolean; virtual;
    Procedure KeyDown( Var Key : Word; Shift : TShiftState ); Override;
    procedure KeyPress( var Key : Char ); override;
  public
    constructor Create( AOwner : TComponent ); override;
  published
    property AutoSelect;
    property AutoSize;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property Cursor;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    Property FormatString : String read  FFormatString
                                   write SetFormatString;
    property Height;
    property HelpContext;
    property Hint;
    property Left;
    property MaxLength;
    property MaxValue : Longint read FMaxValue write SetMaxValue;
    property MinValue : Longint read FMinValue write SetMinValue;
    property Name;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Tag;
    property Validate : Boolean read FValidate write SetValidate;
    property Value : Longint read GetValue write SetValue;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

{--------------------------------------------------------------------------}

  TCardinalEdit = class(TIntegerEdit)
  Private
    procedure SetMaxValue( NewValue : Longint );   Override;
    procedure SetMinValue( NewValue : Longint );   Override;
  protected
    function  IsValidChar( Key : Char) : Boolean; Override;
  end;

{--------------------------------------------------------------------------}

procedure Register;

{--------------------------------------------------------------------------}

implementation

{--------------------------------------------------------------------------}

uses WinProcs, DsgnIntf;

{--------------------------------------------------------------------------}
function TRealEdit.GetValue : Extended;
{--------------------------------------------------------------------------}
begin
  try
    If Length( Text ) <> 0 Then Result := StrToFloat( Text )
                           Else Result := 0;
  except
    Result := FMinValue;
  end;
end;

{--------------------------------------------------------------------------}
procedure TRealEdit.SetValue( NewValue : Extended );
{--------------------------------------------------------------------------}
begin
  try
    Text := FormatFloat( FFormatString, CheckValue( NewValue ) );
  except
  end;
end;

{--------------------------------------------------------------------------}
function TRealEdit.CheckValue( NewValue : Extended ): Extended;
{--------------------------------------------------------------------------}
begin
  Result := NewValue;
  If FValidate Then Begin
    If NewValue < FMinValue Then
      Result:= FMinValue
    Else
      If NewValue > FMaxValue Then Result:= FMaxValue;
  end;
end;

{--------------------------------------------------------------------------}
Procedure TRealEdit.SetValidate( B :  Boolean );
{--------------------------------------------------------------------------}
begin
  FValidate := B;
  SetValue( Value );
end;

{--------------------------------------------------------------------------}
Procedure TRealEdit.SetFormatString( A : String );
{--------------------------------------------------------------------------}
Begin
  FFormatString := A;
  SetValue( Value );
End;

{--------------------------------------------------------------------------}
Function TRealEdit.EmptyOrSign : Boolean;
{--------------------------------------------------------------------------}
Begin
  Result := (VText='') Or (VText='-') Or (VText='+');
End;

{--------------------------------------------------------------------------}
Function TRealEdit.SyntaxOk : Boolean;
{--------------------------------------------------------------------------}
Begin
  try
    {-----------------------------------------------------------
      Syntax correct if transformation causes no exception.
      If the input is empty or a sign ( VText = '', '+', '-' ),
      it is interpreted as 0
     -----------------------------------------------------------}

    If Not EmptyOrSign Then StrToFloat( VText );
    Result := True;

  except
    On EConvertError Do Result := False;
  end;
End;

{--------------------------------------------------------------------------}
procedure TRealEdit.CMEnter( var Message : TCMGotFocus );
{--------------------------------------------------------------------------}
begin
  if AutoSelect and not( csLButtonDown in ControlState ) then SelectAll;
  inherited;
end;

{--------------------------------------------------------------------------}
procedure TRealEdit.CMExit( var Message : TCMExit );
{--------------------------------------------------------------------------}
begin
  inherited;
  SetValue( Value );
end;

{--------------------------------------------------------------------------}
function TRealEdit.IsValidChar( Key : Char ) : Boolean;
{--------------------------------------------------------------------------}
begin
  Case Key Of
    '0'..'9', '+', '-', 'E', 'e', '.' :
       Begin
         Vtext := Copy( Text, 1, SelStart ) + Key +
                  Copy( Text, SelStart+1+SelLength, 255 );
         Result := SyntaxOk;
       End;

    #8 : Begin
           If SelLength = 0 Then
             Vtext := Copy( Text, 1, SelStart-1 ) +
                      Copy( Text, SelStart+1, 255 )
           Else
             Vtext := Copy( Text, 1, SelStart ) +
                      Copy( Text, SelStart+1+SelLength, 255 );
           Result := SyntaxOk;
         End;

  Else
    Result := False;
  End;
end;

{--------------------------------------------------------------------------}
Procedure TRealEdit.KeyDown( Var Key : Word; Shift : TShiftState );
{--------------------------------------------------------------------------}
Begin
  If ssCtrl In Shift Then Begin
    Case Key Of
      88 : { Ctrl X }
           If SelLength <> 0 Then Begin
             Vtext := Copy( Text, 1, SelStart ) +
                      Copy( Text, SelStart+1+SelLength, 255 );
             If SyntaxOk Then Begin
               Clipboard.AsText := SelText;
               Key := VK_DELETE;
               KeyDown( Key, [] );
             End
             Else Key := 0;
           End;

      67 : { Ctrl C }
           Clipboard.AsText := SelText;

      86 : { Ctrl V }
           If Clipboard.HasFormat(CF_TEXT) Then Begin
             Vtext := Copy( Text, 1, SelStart ) +
                      Clipboard.AsText +
                      Copy( Text, SelStart+1, 255 );
             If SyntaxOk Then Begin
               SelText := Clipboard.AsText;
               Key := 0;
             End;
           End
           Else
             MessageDlg( 'No text on the Clipboard', mtInformation,
                         [mbOK],0 );

    Else End;
  End
  Else
    If Key = VK_DELETE Then Begin
      If SelLength = 0 Then
        Vtext := Copy( Text, 1, SelStart ) +
                 Copy( Text, SelStart+2, 255 )
      Else
        Vtext := Copy( Text, 1, SelStart ) +
                 Copy( Text, SelStart+1+SelLength, 255 );
      If Not SyntaxOk Then Key := 0;
    End;

  inherited KeyDown( Key, Shift );

End;

{--------------------------------------------------------------------------}
procedure TRealEdit.KeyPress( var Key : Char );
{--------------------------------------------------------------------------}
begin
  if IsValidChar( Key ) then inherited KeyPress( Key )
                        else Key := #0;
end;

{--------------------------------------------------------------------------}
constructor TRealEdit.Create( AOwner : TComponent );
{--------------------------------------------------------------------------}
begin
  inherited Create( AOwner );
  CharCase := ecUpperCase;
  FMinValue := 0;
  FMaxValue := 0;
  FValidate := False;
  FFormatString := '';
  SetValue( 0 );
end;


{++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++}


{--------------------------------------------------------------------------}
function TIntegerEdit.GetValue : Longint;
{--------------------------------------------------------------------------}
begin
  try
    If Length( Text ) <> 0 Then Result := StrToInt( Text )
                           Else Result := 0;
  except
    Result := FMinValue;
  end;
end;

{--------------------------------------------------------------------------}
procedure TIntegerEdit.SetValue( NewValue : Longint );
{--------------------------------------------------------------------------}
begin
  try
    Text := FormatFloat( FFormatString, CheckValue( NewValue ) );
  except
  end;
end;

{--------------------------------------------------------------------------}
function TIntegerEdit.CheckValue( NewValue : Longint ): Longint;
{--------------------------------------------------------------------------}
begin
  Result := NewValue;
  if FValidate then begin
    if NewValue < FMinValue then
      Result:= FMinValue
    else
      if NewValue > FMaxValue then Result:= FMaxValue;
  end;
end;

{--------------------------------------------------------------------------}
procedure TIntegerEdit.SetMaxValue( NewValue : Longint );
{--------------------------------------------------------------------------}
Begin FMaxValue := NewValue; End;

{--------------------------------------------------------------------------}
procedure TIntegerEdit.SetMinValue( NewValue : Longint );
{--------------------------------------------------------------------------}
Begin FMinValue := NewValue; End;

{--------------------------------------------------------------------------}
Procedure TIntegerEdit.SetValidate( B :  Boolean );
{--------------------------------------------------------------------------}
begin
  FValidate := B;
  SetValue( Value );
end;

{------------------------------------------------------------------------------}
Procedure TIntegerEdit.SetFormatString( A : String );
{------------------------------------------------------------------------------}
Begin
  FFormatString := A;
  SetValue( Value );
End;

{--------------------------------------------------------------------------}
Function TIntegerEdit.EmptyOrSign : Boolean;
{--------------------------------------------------------------------------}
Begin
  Result := (VText='') Or (VText='-') Or (VText='+');
End;

{--------------------------------------------------------------------------}
Function TIntegerEdit.SyntaxOk : Boolean;
{--------------------------------------------------------------------------}
Begin
  try
    {-----------------------------------------------------------
      Syntax correct if transformation causes no exception.
      If the input is empty or a sign ( VText = '', '+', '-' ),
      it is interpreted as 0
     -----------------------------------------------------------}

    If Not EmptyOrSign Then StrToInt( VText );
    Result := True;
  except
    On EConvertError Do Result := False;
  end;
End;

{------------------------------------------------------------------------------}
procedure TIntegerEdit.CMEnter( var Message : TCMGotFocus );
{------------------------------------------------------------------------------}
begin
  if AutoSelect and not( csLButtonDown in ControlState ) then SelectAll;
  inherited;
end;

{--------------------------------------------------------------------------}
procedure TIntegerEdit.CMExit( var Message : TCMExit );
{--------------------------------------------------------------------------}
begin
  inherited;
  SetValue( Value );
end;

{--------------------------------------------------------------------------}
function TIntegerEdit.IsValidChar( Key : Char ) : Boolean;
{--------------------------------------------------------------------------}
begin
  Case Key Of
    '0'..'9', '+', '-' :
       Begin
         Vtext := Copy( Text, 1, SelStart ) + Key +
                  Copy( Text, SelStart+1+SelLength, 255 );
         Result := SyntaxOk;
       End;

    #8 : Begin
           If SelLength = 0 Then
             Vtext := Copy( Text, 1, SelStart-1 ) +
                      Copy( Text, SelStart+1, 255 )
           Else
             Vtext := Copy( Text, 1, SelStart ) +
                      Copy( Text, SelStart+1+SelLength, 255 );
           Result := SyntaxOk;
         End;

  Else
    Result := False;
  End;
End;

{--------------------------------------------------------------------------}
Procedure TIntegerEdit.KeyDown( Var Key : Word; Shift : TShiftState );
{--------------------------------------------------------------------------}
Begin
  If ssCtrl In Shift Then Begin
    Case Key Of
      88 : { Ctrl X }
           If SelLength <> 0 Then Begin
             Vtext := Copy( Text, 1, SelStart ) +
                      Copy( Text, SelStart+1+SelLength, 255 );
             If SyntaxOk Then Begin
               Clipboard.AsText := SelText;
               Key := VK_DELETE;
               KeyDown( Key, [] );
             End
             Else Key := 0;
           End;

      67 : { Ctrl C }
           Clipboard.AsText := SelText;

      86 : { Ctrl V }
           If Clipboard.HasFormat(CF_TEXT) Then Begin
             Vtext := Copy( Text, 1, SelStart ) +
                      Clipboard.AsText +
                      Copy( Text, SelStart+1, 255 );
             If SyntaxOk Then Begin
               SelText := Clipboard.AsText;
               Key := 0;
             End;
           End
           Else
             MessageDlg( 'No text on the Clipboard', mtInformation,
                         [mbOK],0 );

    Else End;
  End
  Else
    If Key = VK_DELETE Then Begin
      If SelLength = 0 Then
        Vtext := Copy( Text, 1, SelStart ) +
                 Copy( Text, SelStart+2, 255 )
      Else
        Vtext := Copy( Text, 1, SelStart ) +
                 Copy( Text, SelStart+1+SelLength, 255 );
      If Not SyntaxOk Then Key := 0;
    End;

  inherited KeyDown( Key, Shift );

End;

{--------------------------------------------------------------------------}
procedure TIntegerEdit.KeyPress( var Key : Char );
{--------------------------------------------------------------------------}
begin
  if IsValidChar( Key ) then inherited KeyPress( Key )
                        else Key := #0;
end;

{--------------------------------------------------------------------------}
constructor TIntegerEdit.Create( AOwner : TComponent );
{--------------------------------------------------------------------------}
begin
  inherited Create( AOwner );
  FMinValue := 0;
  FMaxValue := 0;
  FValidate := False;
  FFormatString := '';
  SetValue( 0 );
end;


{++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++}

{--------------------------------------------------------------------------}
procedure TCardinalEdit.SetMaxValue( NewValue : Longint );
{--------------------------------------------------------------------------}
Begin
  If NewValue<0 Then NewValue:=0;
  FMaxValue := NewValue;
End;

{--------------------------------------------------------------------------}
procedure TCardinalEdit.SetMinValue( NewValue : Longint );
{--------------------------------------------------------------------------}
Begin
  If NewValue<0 Then NewValue:=0;
  FMinValue := NewValue;
End;

{--------------------------------------------------------------------------}
function TCardinalEdit.IsValidChar( Key : Char ) : Boolean;
{--------------------------------------------------------------------------}
begin
  Case Key Of
    '0'..'9' :
       Begin
         Vtext := Copy( Text, 1, SelStart ) + Key +
                  Copy( Text, SelStart+1+SelLength, 255 );
         Result := SyntaxOk;
       End;

    #8 : Begin
           If SelLength = 0 Then
             Vtext := Copy( Text, 1, SelStart-1 ) +
                      Copy( Text, SelStart+1, 255 )
           Else
             Vtext := Copy( Text, 1, SelStart ) +
                      Copy( Text, SelStart+1+SelLength, 255 );
           Result := SyntaxOk;
         End;

  Else
    Result := False;
  End;
End;

{--------------------------------------------------------------------------}
procedure Register;
{--------------------------------------------------------------------------}
begin
  RegisterComponents('MyComponents', [TRealEdit]);
  RegisterComponents('MyComponents', [TIntegerEdit]);
  RegisterComponents('MyComponents', [TCardinalEdit]);
end;


end.

