unit Rrwxbase;

{ A VCL Component to interface to R and R for Windows - XBase Edition 6.0
  Not tested with any other release of R and R.

  This, RREPORT.PAS and the normal R and R runtime engine are all that is required.

  Simply install the RRWxBase component, drop a component on the form and set the parameters
  and called Execute/Run.

  Use the ErrorText property to find out any errors that occurred.

  This is Work In Progress. I'll add features from time to time. If you have any suggestions (good or otherwise)
  please e-mail me.

  This component does not support all RR engine features, but enough to
  do serious work. In particular the areas that need improving are in
  the setting of database and report library paths

  Copyright 1995. John Wright. CIS 100335,322
  }

interface

uses
  Classes, wintypes;

const
  RRLibraryNameLength  = 255 ;
  RRReportNameLength   = 41 ;
  MaxCopiesAllowed     = 99 ;
  MaxPages             = 99999999 ;
  ErrorMsgLen          = 200 ;

type

  TRRLibrary      = string[RRLibraryNameLength] ;
  TRRReportName   = string[RRReportNameLength] ;
  TNoOfCopies     = 0..MaxCopiesAllowed ;
  TPages          = 0..MaxPages ;
  TReportHandle   = integer ;

  TErrorCode      = char ;
  TErrorText      = string[ErrorMsgLen] ;

  { As well as being more efficient, the userdefined types are easier to
  read and the user is less likely to pass an illegal value to the R and
  R engine.        }

  TReportDestn    = (rdDisplay, rdPrinter, rdTextFile, rdWorksheet, rdXBaseFile, rdUserSelection) ;
  TReturnCode     = (rcSuccess, rcPrinterOffline, rcInvalidLibrary, rcInvalidReport) ;
  TErrorInfo      = (eCancel, eDiagnostic, eIteration, eJobControl, eLibrary, eSyntax, eValue, eUnknown) ;
  TRRIndexType    = (tNone, tFoxproMulti, tFoxproSingle, tdBaseMulti, tdBaseSingle, tSWare, tClipper, tWDX) ;
  TTestPattern    = (NoTestPattern, PrintATestPattern) ;
  TDisplayStatus  = (dsShow, dsDontShow) ;

  TRRXBase = class(TComponent)
  private

    FReportHandle  : TReportHandle ;

    FRRLibrary     : TRRLibrary    ;
    FRRReportName  : TRRReportName ;

    FNoOfCopies    : TNoOfCopies   ;
    FStartPage     : TPages        ;
    FEndPage       : TPages        ;
    FReportDestn   : TReportDestn  ;

    ErrorMsgBuffer : PChar ;

    RRErrorCode    : PInteger ;
    FPagesPrinted  : PLongInt ;

    FErrorInfo     : TErrorInfo ;
    FErrorText     : TErrorText ;
    FDisplayStatus : TDisplayStatus ;
    FRRIndexType   : TRRIndexType ;

    FTestPattern   : TTestPattern ;

    Function GetPagesPrinted : LongInt ;
    procedure SetupPrintCharacteristics ;
    procedure SetupDatabaseCharacteristics ;
    procedure SetStartEndPages ;
    procedure SetNoOfCopies ;

  protected
    function GetErrorCode : TErrorInfo ;
    function GetErrorText : TErrorText ;

    Property ReportHandle : TReportHandle read FReportHandle ;

  public
    constructor Create(AOwner : TComponent) ; override ;
    destructor Destroy ; override ;

    Function Execute(var ReturnCode : TReturnCode) : Boolean ;
    Function Run(var ReturnCode : TReturnCode) : Boolean ;

    Property ErrorText    : TErrorText read GetErrorText ;

  published
    property DisplayStatus : TDisplayStatus read FDisplayStatus write FDisplayStatus default dsShow ;
    property LibraryName   : TRRLibrary     read FRRLibrary     write FRRLibrary ;
    property ReportName    : TRRReportName  read FRRReportName  write FRRReportName ;
    property NoOfCopies    : TNoOfCopies    read FNoOfCopies    write FNoOfCopies ;
    Property StartPage     : TPages         read FStartPage     write FStartPage ;
    Property EndPage       : TPages         read FEndPage       write FEndPage ;
    Property Destination   : TReportDestn   read FReportDestn   write FReportDestn default rdPrinter ;
    Property PagesPrinted  : Longint        read GetPagesPrinted ;
    Property IndexType     : TRRIndexType   read FRRIndexType   write FRRIndexType default tNone ;
    Property TestPattern   : TTestPattern   read FTestpattern   write FTestPattern default NoTestPattern ;

  end;

procedure Register;

implementation

uses
  Sysutils, RReport;

procedure Register;
begin
  RegisterComponents('R and R For Windows', [TRRXBase]);
end;

constructor TRRXbase.Create(AOwner : TComponent) ;
Begin
  inherited Create(Aowner);

  { Initialise the R and R Engine  - Is this valid if we have more than one
    rrwxbase component on a form? If so we could move this and the endruntimeinstance
    to the Initialisation and exitproc section - check with Concentric. }

  InitRunTimeInstance ;

  { Allocate memory for the various output buffers }

  new(RRErrorCode) ;
  new(FPagesPrinted) ;

  ErrorMsgBuffer := AllocMem(ErrorMsgLen) ;

  { set Defaults for printing and misc. }

  FReportDestn   := rdPrinter ;
  FDisplayStatus := dsShow ;
  FRRIndexType   := tNone ;
  FTestPattern   := NoTestPattern ;

  { zero values mean; use the embedded values in the report }

  FNoOfCopies    := 0 ;
  FStartPage     := 0 ;
  FEndPage       := 0 ;
End ;

destructor TRRXBase.Destroy ;
Begin

  { Free R and R Runtime Engine }

  EndRunTimeInstance ;

  { Free Memory }

  FreeMem(ErrorMsgBuffer, ErrorMsgLen) ;
  dispose(FPagesPrinted) ;
  dispose(RRErrorCode) ;

  inherited Destroy ;

End ;

Function TRRXbase.GetPagesPrinted : LongInt ;
Begin
  Result := FPagesPrinted^ ;
End ;

function TRRXbase.GetErrorText : TErrorText ;
Begin
  Result := FErrorTExt ;
End ;

function TRRXbase.GetErrorCode : TErrorInfo ;
var
  c : char ;
Begin

  { Interrogate the engine }

  GetErrorInfo(ErrorMsgBuffer, ErrorMsgLen, RRErrorCode) ;

  { convert the c style string holding the error message to a _real_
  string }

  FErrorText := strpas(ErrorMsgBuffer) ;

  { convert the error code from the low byte of an integer,
   then to a user defined type }

  c := Char(Lo(RRErrorCode^)) ;

  case c of
    'C' : Result := eCancel ;
    'D' : Result := eDiagnostic ;
    'I' : Result := eIteration ;
    'J' : Result := eJobControl ;
    'L' : Result := eLibrary ;
    'S' : Result := eSyntax ;
    'V' : Result := eValue ;
  else
    Result := eUnknown ;
  End ;

end ;

procedure TRRXBase.SetStartEndPages ;
Begin

  { if the values of the pages are not zero then set the pages in the
  engine. If they are zero then leave it. Zero means use the values
  embedded in the report }

  if FStartPage <> 0 then
    SetBeginPage(FReportHandle, FStartPage) ;

  if FEndPage <> 0 then
    SetEndPage(FReportHandle, FEndPage) ;

End ;

procedure TRRXBase.SetNoOfCopies ;
Begin

  { if the number of copies property was set, call RR Engine to set them }

  if FNoOfCopies <> 0 then
    SetCopies(FReportHandle, FNoOfCopies) ;

End ;

procedure TRRXBase.SetupPrintCharacteristics ;
Begin

  if FReportHandle <> 0 then
  Begin

    { Setup the Display status, true means display a counter as records are sorted etc }

    if FDisplayStatus = dsShow then
      SetDisplayStatus(FReportHandle, True)
    else
      SetDisplayStatus(FReportHandle, False) ;

    { Set the output destination based upon the output property
     ? means ask the user what they want to do. Try the Excel Chart and
     cross-tab features. Pretty good! }

    case FReportDestn of
      rdDisplay       : SetOutPutDest(FReportHandle, 'D') ;
      rdPrinter       : SetOutPutDest(FReportHandle, 'P') ;
      rdTextFile      : SetOutPutDest(FReportHandle, 'T') ;
      rdWorksheet     : SetOutPutDest(FReportHandle, 'W') ;
      rdXBaseFile     : SetOutPutDest(FReportHandle, 'X') ;
      rdUserSelection : SetOutPutDest(FReportHandle, '?') ;
    end ;

    SetStartEndPages ;
    SetNoOfCopies ;

    { Determine whether to print a test pattern }

    if FTestPattern = PrintATestPattern then
      SetTestPattern(FReportHandle, True)
    else
      SetTestPattern(FReportHandle, False) ;

  End ;

End ;

procedure TRRXbase.SetupDatabaseCharacteristics ;
const
  ITNone = 0 ;
  ITCDX  = 1 ;
  ITIDX  = 2 ;
  ITMDX  = 3 ;
  ITNDX  = 4 ;
  ITNSX  = 5 ;
  ITNTX  = 6 ;
  ITWDX  = 7 ;
Begin

  { Choose Index type - change here, in the class definition and in the
    constructor to reflect your own preferences. E.g CDX }

  case FRRIndexType of
    tNone         : SetIndexExtension(FReportHandle, ITNone) ;
    tFoxproMulti  : SetIndexExtension(FReportHandle, ITCDX) ;
    tFoxproSingle : SetIndexExtension(FReportHandle, ITIDX) ;
    tdBaseMulti   : SetIndexExtension(FReportHandle, ITMDX) ;
    tdBaseSingle  : SetIndexExtension(FReportHandle, ITNDX) ;
    tSware        : SetIndexExtension(FReportHandle, ITNSX) ;
    tClipper      : SetIndexExtension(FReportHandle, ITNTX) ;
    tWDX          : SetIndexExtension(FReportHandle, ITWDX) ;
  end ;
End ;

Function TRRXbase.Run(var ReturnCode : TReturnCode) : Boolean ;
var
  LibraryName : string ;
  ReportName  : string ;
Begin
  Result := False ;
  ReturnCode := rcSuccess ;

  { convert library names to c style strings }

  LibraryName := FRRLibrary + #0 ;
  ReportName  := FRRReportName + #0 ;

  { Setup the report, based upon library and report name }

  FReportHandle := ChooseReport('', @LibraryName[1], @ReportName[1], RRReportNameLength) ;

  { If report setup didn't work get the error code
    if it did work, setup various parameters and then run the report }

  if FReportHandle = 0 then
    FErrorInfo := GetErrorCode
  else
  Begin
    SetupDatabaseCharacteristics ;
    SetupPrintCharacteristics ;
    ExecRunTime(FReportHandle, False, SW_SHOW, RRErrorCode, FPagesPrinted, ErrorMsgBuffer, ErrorMsgLen) ;
    Result := true ;
  End ;

  { Tidy up the report }

  EndReport(FReportHandle) ;

End ;

Function TRRXbase.Execute(var ReturnCode : TReturnCode) : Boolean ;
Begin

  { Some report writers use Run some use Execute, allow the user to
    choose }

  Result := self.Run(ReturnCode) ;
End ;

end.
