unit Refreshr;

{ Version 2.01  Apr-20-1997
  (C) 1997 Christoph R. Kirchner

  TRefresher manages the refreshes for TSelfRefreshTable.
  Only one instance of this object gets created and it gets
  destroyed by an ExitProc.

  It uses different methods for Delphi 1 and 2, but this is the
  source for both versions.

  Delphi 1: Paradox-tables only.
  Delphi 2: Paradox- and dBase-tables.

  TSelfRefreshTable uses the Application.OnIdle-event to do the
  refresh if it is save.

  Set DontRefreshIfAppDeactivated to true if your application gets
  confused by self-refreshing tables while the program is not active.

  The author assumes no liability for damages and disclaims all 
  warranties. See disclaimer in Readme.txt.
  Please forward any comments or suggestions to Christoph Kirchner at:
  ckirchner@geocities.com
}

{$IFNDEF WIN32}{$K+}{$ENDIF NDEF WIN32}

interface

uses
  WinTypes, WinProcs, SysUtils, Messages, Classes, Controls,
  Forms, DB, DbiErrs, DBITypes, DBIProcs;

type

{$IFNDEF WIN32}
{ TBDECallback is used to handle a chain of BDE-callbacks.
  In Delphi 2.0 it is defined in the unit DB. }

  TBDECallbackEvent = function(CBInfo: Pointer): CBRType of object;

  TBDECallback = class
  private
    FHandle: hDBICur;
    FOwner: TObject;
    FCBType: CBType;
    FOldCBData: Longint;
    FOldCBBuf: Pointer;
    FOldCBBufLen: Word;
    FOldCBFunc: Pointer; { pfDBICallBack }
    FInstalled: Boolean;
    FCallbackEvent: TBDECallbackEvent;
  protected
    function Invoke(CallType: CBType; CBInfo: Pointer): CBRType;
  public
    constructor Create(AOwner: TObject; Handle: hDBICur; CBType: CBType;
      CBBuf: Pointer; CBBufSize: Word; CallbackEvent: TBDECallbackEvent;
      Chain: Boolean);
    destructor Destroy; override;
  end;
{$ENDIF NDEF WIN32}

  TBDERefreshCallback = class(TBDECallback)
  public
    constructor Create(AOwner: TDataset; RefreshCallback: TBDECallbackEvent);
    destructor Destroy; override;
  end;

  TRefreshEvent = procedure (RefreshDataset: TDataset; var Done: Boolean);

  TRefresher = class(TObject)
  private
    FAppOnActivate: TNotifyEvent;
    FAppOnDeactivate: TNotifyEvent;
    FAppOnIdle: TIdleEvent;
    FDatasetsToRefresh: TList;
    FRefreshEvents: TList;
    FRefreshTimer: Word;
    FRefreshCheckPeriod: Integer;
    procedure SetRefreshCheckPeriod(Value: Integer);
  protected
    procedure AppOnActivate(Sender: TObject);
    procedure AppOnDeactivate(Sender: TObject);
    procedure AppOnIdle(Sender: TObject; var Done: Boolean);
  public
    AppActivated: Boolean;
    constructor Create;
    destructor Destroy; override;
    procedure RefreshDatasets;
    procedure AddDataset(Dataset: TDataset);
    procedure RemoveDataset(Dataset: TDataset);
    procedure DatasetNeedsRefresh(Dataset: TDataset; RefreshEvent: Pointer);
    procedure ForceDatasetRefreshNow;
    property RefreshCheckPeriod: Integer
      read FRefreshCheckPeriod write SetRefreshCheckPeriod;
  end;

const
{ set DontRefreshIfAppDeactivated to true, if your application cannot
  handle refreshes while not activated: }
  DontRefreshIfAppDeactivated: Boolean = false;

{ call this if you don't want to wait for Application.Idle
  to make sure every Dataset is refreshed: }
procedure ForceTableRefreshNow;

{ check for changed Datasets every ... ms (default 1000): }
procedure SetRefreshCheckPeriod(Value: Integer);


const
  Refresher: TRefresher = nil;

implementation


const
  DefaultRefreshCheckPeriod = 1000; { check for changed Datasets every ... ms }

const
{$IFDEF WIN32}
  FActiveDatasets: TList = nil;
  FCheckDatasetNumber: Integer = 0;
{$ELSE  DEF WIN32}
  BDERefreshCallbacks: TList = nil;
{$ENDIF DEF WIN32}


{ TimerCalledRefreshCheck is called every RefreshCheckPeriod ms.
  Application.Idle is called until Done is true (it is true by default).
  Setting Done to false consumes too much processor-time. With Done = true,
  the Application waits for the next message before it calls Idle again and
  no refresh will occur until the user moves the mouse over the Application-
  window or something else happens. But DBICheckRefresh sends a message - for
  that we will call it here regular. }

procedure TimerCalledRefreshCheck(
{$IFDEF WIN32}
  hWnd: HWND; Message: Word; TimerID: Word; SysTime: LongInt); stdcall;
{$ELSE DEF WIN32}
  hWnd: HWND; Message: Word; TimerID: Word; SysTime: LongInt); export;
{$ENDIF DEF WIN32}
begin
{ DbiCheckRefresh checks for remote updates to Datasets for all cursors in
  the current session, and refreshes the cursors if changed. }
  DBICheckRefresh;
end;


{ ExitProc to destroy the Refresh-manager }
procedure MyExitProc; far;
begin
  if Assigned(Refresher) then
  begin
    Refresher.Free;
    Refresher := nil;
  end;
end;




{$IFNDEF WIN32}

{ RefreshWindow --------------------------------------------------------- }

const
  CM_EXECPROC = CM_ACTIVATE;

var
  RefreshWindow: HWND;

function RefreshWndProc(
  Window: HWND; Message, wParam: Word; lParam: Longint): Longint; export;
begin
  if (Message = CM_EXECPROC) then
  begin
    Result := 0;
    TBDERefreshCallback(BDERefreshCallbacks[lParam]).Invoke(
      CBType(wParam), nil);
  end
  else
    Result := DefWindowProc(Window, Message, wParam, lParam);
end;

const
  RefreshWindowClass: TWndClass = (
    style: 0;
    lpfnWndProc: @RefreshWndProc;
    cbClsExtra: 0;
    cbWndExtra: 0;
    hInstance: 0;
    hIcon: 0;
    hCursor: 0;
    hbrBackground: 0;
    lpszMenuName: nil;
    lpszClassName: '');

procedure AllocateRefreshWindow;
var
  TempClass: TWndClass;
  ClassRegistered: Boolean;
  B: array[0..79] of Char;
begin
  RefreshWindowClass.lpszClassName :=
    StrPCopy(B, 'RefreshWindow' + IntToHex(HInstance, 4)); { make it unique }
  RefreshWindowClass.hInstance := HInstance;
  RefreshWindowClass.lpfnWndProc :=
    MakeProcInstance(@RefreshWndProc, HInstance); {}
  Winprocs.RegisterClass(RefreshWindowClass);
  RefreshWindow := CreateWindow(RefreshWindowClass.lpszClassName, '', 0,
                                0, 0, 0, 0, 0, 0, HInstance, nil);
end;

procedure DestroyRefreshWindow;
begin
  DestroyWindow(RefreshWindow);
  FreeProcInstance(RefreshWindowClass.lpfnWndProc);
end;


{ TBDECallback --------------------------------------------------------- }

function BdeCallBack(ecbType: CBType; iClientData: Longint;
                     var CBInfo: Pointer): CBRType; export;
begin
  Result := cbrUSEDEF;
  if (iClientData <> 0) then
  begin
  { Here we have to post a message to the RefreshWindow.
    If another instance of this application is active, sometimes we will
    get the callback that should happen in this instance - the other instance
    then gets nothing. For this we have to distribute the message to the
    correct RefreshWindow. This is why we stored the handle of the
    RefreshWindow in the ClientData and we could not store a pointer to an
    object in it. }
    PostMessage(iClientData and $FFFF, CM_EXECPROC, Word(ecbType),
                iClientData shr 16);
{   if (iClientData and $FFFF) <> RefreshWindow then
      MessageBeep(MB_ICONQUESTION); { debug-beep: called by another instance }
  end;
end;

constructor TBDECallback.Create(AOwner: TObject; Handle: hDBICur;
  CBType: CBType; CBBuf: Pointer; CBBufSize: Word;
  CallbackEvent: TBDECallbackEvent; Chain: Boolean);
begin
  FOwner := AOwner;
  FHandle := Handle;
  FCBType := CBType;
  FCallbackEvent := CallbackEvent;
  DbiGetCallBack(Handle, FCBType, FOldCBData, FOldCBBufLen,
                 FOldCBBuf, FOldCBFunc);
  if not Assigned(FOldCBFunc) or Chain then
  begin
    Check(DbiRegisterCallback(FHandle, FCBType,
      ((Longint(BdeRefreshCallBacks.Count) - 1) shl 16) + RefreshWindow,
      CBBufSize, CBBuf, BdeCallBack));
    FInstalled := True;
  end;
end;

destructor TBDECallback.Destroy;
begin
  if FInstalled then
  begin
    if Assigned(FOldCBFunc) then
    try
      DbiRegisterCallback(FHandle, FCBType, FOldCBData, FOldCBBufLen,
                          FOldCBBuf, pfDBICallBack(FOldCBFunc));
    except
    end
    else
      DbiRegisterCallback(FHandle, FCBType, 0, 0, nil, nil);
  end;
end;

function TBDECallback.Invoke(CallType: CBType; CBInfo: Pointer): CBRType;
begin
  if CallType = FCBType then
    Result := FCallbackEvent(CBInfo)
  else
    Result := cbrUSEDEF;
  if Assigned(FOldCBFunc) then
  begin
{   MessageBeep(MB_ICONSTOP); { debug }
    Result := pfDBICallBack(FOldCBFunc)(CallType, FOldCBData, CBInfo);
  end;
end;
{$ENDIF NDEF WIN32}




{ TBDERefreshCallback --------------------------------------------------------- }

{$IFDEF WIN32}

constructor TBDERefreshCallback.Create(
  AOwner: TDataset; RefreshCallback: TBDECallbackEvent);
begin
  inherited Create(AOwner, AOwner.Handle, cbTableChanged,
                   nil, 0, RefreshCallBack, true);
end;

destructor TBDERefreshCallback.Destroy;
begin
  inherited Destroy;
end;

{$ELSE DEF WIN32}

constructor TBDERefreshCallback.Create(
  AOwner: TDataset; RefreshCallback: TBDECallbackEvent);
var
  i: Integer;
  Done: Boolean;
begin
  if (BDERefreshCallbacks.Count >= 64) then
  begin
  { look for free entry, do not let the list grow too much: }
    Done := false;
    for i := 0 to BDERefreshCallbacks.Count -1 do
      if (BDERefreshCallbacks[i] = nil) then
      begin
        BDERefreshCallbacks[i] := self;
        Done := true;
        break;
      end;
    if not Done then
      BDERefreshCallbacks.Add(self);
  end
  else
    BDERefreshCallbacks.Add(self);
  inherited Create(AOwner, AOwner.Handle, cbTableChanged,
                   nil, 0, RefreshCallBack, true);
end;

destructor TBDERefreshCallback.Destroy;
var
  IndexOfSelf: Integer;
begin
  IndexOfSelf := BDERefreshCallbacks.IndexOf(self);
  if (IndexOfSelf >= 0) then
    BDERefreshCallbacks[IndexOfSelf] := nil;
  inherited Destroy;
end;
{$ENDIF DEF WIN32}




{ TRefresher ---------------------------------------------------------------- }

constructor TRefresher.Create;
begin
  inherited Create;
  AddExitProc(MyExitProc);
{$IFDEF WIN32}
  FActiveDatasets := TList.Create;
  FCheckDatasetNumber := 0;
{$ELSE DEF WIN32}
  BDERefreshCallbacks := TList.Create;
  AllocateRefreshWindow;
{$ENDIF DEF WIN32}
  FDatasetsToRefresh := TList.Create;
{ FDatasetsToRefresh.Capacity := 255; { get memory now, not in the callback }
  FRefreshEvents := TList.Create;
{ FRefreshEvents.Capacity := 255; { get memory now, not in the callback }
  AppActivated := true;
{ Save application-events: }
  FAppOnActivate := Application.OnActivate;
  FAppOnDeactivate := Application.OnDeactivate;
  FAppOnIdle := Application.OnIdle;
{ Redirect application-events: }
  Application.OnActivate := AppOnActivate;
  Application.OnDeactivate := AppOnDeactivate;
  Application.OnIdle := AppOnIdle;
  FRefreshCheckPeriod := DefaultRefreshCheckPeriod;
  FRefreshTimer :=
    SetTimer(0, 0, FRefreshCheckPeriod, @TimerCalledRefreshCheck);
end;

destructor TRefresher.Destroy;
begin
{ TRefresher.Destroy is called from MyExitProc only }
  KillTimer(0, FRefreshTimer);
{$IFDEF WIN32}
  FActiveDatasets.Free;
{$ELSE DEF WIN32}
  DestroyRefreshWindow;
  BDERefreshCallbacks.Free;
{$ENDIF DEF WIN32}
  FDatasetsToRefresh.Free;
  FRefreshEvents.Free;
  Application.OnActivate := FAppOnActivate;
  Application.OnDeactivate := FAppOnDeactivate;
  Application.OnIdle := FAppOnIdle;
  inherited Destroy;
end;

procedure TRefresher.AppOnActivate(Sender: TObject);
begin
  AppActivated := true;
{ we want fresh data, whatever happend while we were out: }
  ForceDatasetRefreshNow;
  if Assigned(FAppOnActivate) then
    FAppOnActivate(Sender);
end;

procedure TRefresher.AppOnDeactivate(Sender: TObject);
begin
  AppActivated := false;
  if Assigned(FAppOnDeactivate) then
    FAppOnDeactivate(Sender);
end;

procedure TRefresher.AppOnIdle(Sender: TObject; var Done: Boolean);
{$IFDEF WIN32} var iSeqNo: Longint; {$ENDIF DEF WIN32}
begin
  if Assigned(FAppOnIdle) then
    FAppOnIdle(Sender, Done);
  if AppActivated or not DontRefreshIfAppDeactivated then
  begin
  {$IFDEF WIN32}
    if (FActiveDatasets.Count > 0) then
    begin
      if (FCheckDatasetNumber + 1 >= FActiveDatasets.Count) then
        FCheckDatasetNumber := 0
      else
        Inc(FCheckDatasetNumber);
      with TDataset(FActiveDatasets[FCheckDatasetNumber]) do
      begin
        if (State = dsBrowse) then
        begin
        { Force the BDE to check if the Dataset was changed.
          DbiGetRecord or DbiGetSeqNo will do this, but I have not searched
          for the most fast procedure that will do the job.
          If we do not do this, the RefreshCallBack will be only called if
          the Dataset got changed by this application. Maybe this is an error
          only in the BDE-version I use. }
          DbiGetSeqNo(Handle, iSeqNo);
        { DbiGetRecord(Handle, dbiNoLock, nil, nil); {}
        end;
      end;
    end;
  {$ENDIF DEF WIN32}
    if (FDatasetsToRefresh.Count > 0) then
      RefreshDatasets;
  end;
end;

procedure TRefresher.RefreshDatasets;
var
  RefreshEvent: TRefreshEvent;
  DatasetIndex: Integer;
  Done: Boolean;
begin
{ Don't be confused about the use of a TRefreshEvent instead of calling
  TSelfRefreshTable(FDatasetsToRefresh[DatasetIndex]).DoRefresh(Done)
  directly. This is made to avoid a "uses Refresh" in this unit. So it 
  is possible to make another TSelfRefreshXXX without changing this unit. }
  DatasetIndex := 0;
  while (FDatasetsToRefresh.Count > DatasetIndex) do
  begin
    try
      @RefreshEvent := FRefreshEvents[DatasetIndex];
      RefreshEvent(FDatasetsToRefresh[DatasetIndex], Done);
      if Done then
      begin
        FRefreshEvents.Delete(DatasetIndex);
        FDatasetsToRefresh.Delete(DatasetIndex);
      end
      else
        Inc(DatasetIndex); { try later }
    except
    { ?!... maybe SelfRefreshTable.ComponentState = csDestroying }
      FRefreshEvents.Delete(DatasetIndex);
      FDatasetsToRefresh.Delete(DatasetIndex);
    end;
  end;
end;

procedure TRefresher.AddDataset(Dataset: TDataset);
begin
{$IFDEF WIN32}
  FActiveDatasets.Add(Dataset);
{$ENDIF DEF WIN32}
end;

procedure TRefresher.RemoveDataset(Dataset: TDataset);
var
  DatasetIndex: Integer;
begin
{$IFDEF WIN32}
  FActiveDatasets.Remove(Dataset);
{$ENDIF DEF WIN32}
  repeat
    DatasetIndex := FDatasetsToRefresh.Remove(Dataset);
    if (DatasetIndex >= 0) then
      FRefreshEvents.Delete(DatasetIndex);
  until (DatasetIndex = -1);
end;

procedure TRefresher.DatasetNeedsRefresh(
  Dataset: TDataset; RefreshEvent: Pointer);
begin
  if (FDatasetsToRefresh.IndexOf(Dataset) = -1) then
  begin
    FDatasetsToRefresh.Add(Dataset);
    FRefreshEvents.Add(RefreshEvent);
  end;
end;

procedure TRefresher.ForceDatasetRefreshNow;
{$IFDEF WIN32}
var
  i: Integer;
  iSeqNo: Longint;
{$ENDIF DEF WIN32}
begin
  DBICheckRefresh;
{$IFDEF WIN32}
  if (FActiveDatasets.Count > 0) then
  begin
    for i := 0 to FActiveDatasets.Count - 1 do
      with TDataset(FActiveDatasets[i]) do
      begin
        if (State = dsBrowse) then
        begin
        { Force the BDE to check if the Dataset was changed: }
          DbiGetSeqNo(Handle, iSeqNo);
        { DbiGetRecord(Handle, dbiNoLock, nil, nil); { will do also }
        end;
      end;
  end;
{$ENDIF DEF WIN32}
  if (FDatasetsToRefresh.Count > 0) then
    RefreshDatasets;
end;

procedure TRefresher.SetRefreshCheckPeriod(Value: Integer);
begin
  if (FRefreshCheckPeriod <> Value) then
  begin
    FRefreshCheckPeriod := Value;
    KillTimer(0, FRefreshTimer);
    FRefreshTimer :=
      SetTimer(0, 0, FRefreshCheckPeriod, @TimerCalledRefreshCheck);
  end;
end;




procedure ForceTableRefreshNow;
begin
  if Assigned(Refresher) then
    Refresher.ForceDatasetRefreshNow;
end;

procedure SetRefreshCheckPeriod(Value: Integer);
begin
  if Assigned(Refresher) then
    Refresher.SetRefreshCheckPeriod(Value);
end;


end.
