unit RefreshQ;

{ Version 2.1  Apr-20-1997
  (C) 1997 Christoph R. Kirchner

  TSelfRefreshQuery is a TQuery-descendant that uses BDE-Callbacks
  to refresh itself if the dataset got modified by another program
  or by another TDataset-component in the same application.

  It uses different methods for Delphi 1 and 2, but this is the
  source for both versions.

  Delphi 1: Query on Paradox-tables only.
  Delphi 2: Query on Paradox- and dBase-tables, others not tested yet.

  TSelfRefreshQuery uses the Application.OnIdle-event to do the
  refresh if it is save.

  Please Note that querys cannot get refreshed if RequestLive is false.

  The author assumes no liability for damages and disclaims all 
  warranties. See disclaimer in Readme.txt.
  Please forward any comments or suggestions to Christoph Kirchner at:
  ckirchner@geocities.com
}


interface

uses
  WinTypes, WinProcs, SysUtils, Classes, Controls,
  DB, DBTables, DBITypes, Refreshr;

type

  TSelfRefreshQuery = class(TQuery)
  private
    FBeforeRefresh: TDataSetNotifyEvent;
    FAfterRefresh: TDataSetNotifyEvent;
    FSelfRefreshEnabled: Boolean;
    FRefreshCallback: TBDERefreshCallback;
    function RefreshCallBack(CBInfo: Pointer): CBRType;
    procedure SetSelfRefreshEnabled(Value: Boolean);
  protected
    procedure DoAfterOpen; override;
    procedure DoBeforeClose; override;
    procedure DoBeforeRefresh; virtual;
    procedure DoAfterRefresh; virtual;
    procedure EnableRefresh; virtual;
    procedure DisableRefresh; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DoRefresh(var Done: Boolean);
  published
    property SelfRefreshEnabled: Boolean
      read FSelfRefreshEnabled write SetSelfRefreshEnabled default true;
    property BeforeRefresh: TDataSetNotifyEvent
      read FBeforeRefresh write FBeforeRefresh;
    property AfterRefresh: TDataSetNotifyEvent
      read FAfterRefresh write FAfterRefresh;
  end;


{ call this if you don't want to wait for Application.Idle
  to make sure every Query is refreshed: }
procedure ForceQueryRefreshNow;

{ check for changed Querys every ... ms (default 1000): }
procedure SetRefreshCheckPeriod(Value: Integer);


implementation


procedure DoRefreshSelfRefreshQuery(
  RefreshDataset: TDataset; var Done: Boolean); far;
begin
  TSelfRefreshQuery(RefreshDataset).DoRefresh(Done);
end;


{ TSelfRefreshQuery --------------------------------------------------------- }

constructor TSelfRefreshQuery.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSelfRefreshEnabled := true;
  FBeforeRefresh := nil;
  FAfterRefresh := nil;
  FRefreshCallback := nil;
end;

destructor TSelfRefreshQuery.Destroy;
begin
  DisableRefresh;
  inherited Destroy;
end;

function TSelfRefreshQuery.RefreshCallBack(CBInfo: Pointer): CBRType;
begin
  Result := cbrUSEDEF;
  Refresher.DatasetNeedsRefresh(self, @DoRefreshSelfRefreshQuery);
end;

procedure TSelfRefreshQuery.DoAfterOpen;
begin
  EnableRefresh;
  inherited DoAfterOpen;
end;

procedure TSelfRefreshQuery.DoBeforeClose;
begin
  DisableRefresh;
  inherited DoBeforeClose;
end;

procedure TSelfRefreshQuery.DoBeforeRefresh;
begin
  if Assigned(FBeforeRefresh) then
    FBeforeRefresh(self);
end;

procedure TSelfRefreshQuery.DoAfterRefresh;
begin
  if Assigned(FAfterRefresh) then
    FAfterRefresh(self);
end;

procedure TSelfRefreshQuery.SetSelfRefreshEnabled(Value: Boolean);
begin
  if (FSelfRefreshEnabled = Value) then
    exit;
  FSelfRefreshEnabled := Value;
  if Value then
    EnableRefresh
  else
    DisableRefresh;
end;

procedure TSelfRefreshQuery.EnableRefresh;
begin
  if not (csDesigning in Componentstate) then
  begin
    if (FRefreshCallBack = nil) and FSelfRefreshEnabled and RequestLive then
    begin
      if not Assigned(Refresher) then
        Refresher := TRefresher.Create;
      try
        FRefreshCallback := TBDERefreshCallback.Create(self, RefreshCallBack);
      except
        FRefreshCallback := nil;
        exit;
      end;
      Refresher.AddDataset(self);
    end;
  end;
end;

procedure TSelfRefreshQuery.DisableRefresh;
begin
  if Assigned(Refresher) and (FRefreshCallBack <> nil) then
  begin
    FRefreshCallBack.Free;
    FRefreshCallBack := nil;
    Refresher.RemoveDataset(self);
  end;
end;

procedure TSelfRefreshQuery.DoRefresh(var Done: Boolean);
begin
  if (State = dsBrowse) then
  begin
    Done := true;
    DisableControls;
    try
      DoBeforeRefresh;
    { Refresh uses DbiForceReread to refresh all buffers for the Query
      associated with the cursor in case remote updates took place: }
      Refresh;
      DoAfterRefresh;
    finally
      EnableControls;
    end;
  end
  else
    Done := false;
end;



{ ---------------------------------------------------------------------------- }

procedure ForceQueryRefreshNow;
begin
  Refreshr.ForceTableRefreshNow;
end;

procedure SetRefreshCheckPeriod(Value: Integer);
begin
  Refreshr.SetRefreshCheckPeriod(Value);
end;



end.
