unit Refresh;

{ Version 2.1  Apr-20-1997
  (C) 1997 Christoph R. Kirchner

  TSelfRefreshTable is a TTable-descendant that uses BDE-Callbacks
  to refresh itself if the table got modified by another program
  or by another TDataset-component in the same application.

  It uses different methods for Delphi 1 and 2, but this is the
  source for both versions.

  Delphi 1: Paradox-tables only.
  Delphi 2: Paradox- and dBase-tables, others not tested yet.

  TSelfRefreshTable uses the Application.OnIdle-event to do the
  refresh if it is save.

  The author assumes no liability for damages and disclaims all 
  warranties. See disclaimer in Readme.txt.
  Please forward any comments or suggestions to Christoph Kirchner at:
  ckirchner@geocities.com
}

{$IFNDEF WIN32}

{ TSelfRefreshTable = class(TFilterTable), requires unit Filt_tbl: }
  { $DEFINE FilterTable}

{$ENDIF NDEF WIN32}

interface

uses
  WinTypes, WinProcs, SysUtils, Messages, Classes, Controls,
  Forms, DB, DBTables, DBITypes, Refreshr
{$IFDEF FilterTable}
  , Filt_tbl
{$ENDIF DEF FilterTable}
  ;

type

{$IFDEF FilterTable}
  TSelfRefreshTable = class(TFilterTable)
{$ELSE DEF FilterTable}
  TSelfRefreshTable = class(TTable)
{$ENDIF DEF FilterTable}
  private
    FBeforeRefresh: TDataSetNotifyEvent;
    FAfterRefresh: TDataSetNotifyEvent;
    FSelfRefreshEnabled: Boolean;
    FRefreshCallback: TBDERefreshCallback;
    function RefreshCallBack(CBInfo: Pointer): CBRType;
    procedure SetSelfRefreshEnabled(Value: Boolean);
  protected
    procedure DoAfterOpen; override;
    procedure DoBeforeClose; override;
    procedure DoBeforeRefresh; virtual;
    procedure DoAfterRefresh; virtual;
    procedure EnableRefresh; virtual;
    procedure DisableRefresh; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DoRefresh(var Done: Boolean);
  published
    property SelfRefreshEnabled: Boolean
      read FSelfRefreshEnabled write SetSelfRefreshEnabled default true;
    property BeforeRefresh: TDataSetNotifyEvent
      read FBeforeRefresh write FBeforeRefresh;
    property AfterRefresh: TDataSetNotifyEvent
      read FAfterRefresh write FAfterRefresh;
  end;


{ call this if you don't want to wait for Application.Idle
  to make sure every table is refreshed: }
procedure ForceTableRefreshNow;

{ check for changed tables every ... ms (default 1000): }
procedure SetRefreshCheckPeriod(Value: Integer);


implementation


procedure DoRefreshSelfRefreshTable(
  RefreshDataset: TDataset; var Done: Boolean); far;
begin
  TSelfRefreshTable(RefreshDataset).DoRefresh(Done);
end;


{ TSelfRefreshTable --------------------------------------------------------- }

constructor TSelfRefreshTable.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSelfRefreshEnabled := true;
  FBeforeRefresh := nil;
  FAfterRefresh := nil;
  FRefreshCallback := nil;
end;

destructor TSelfRefreshTable.Destroy;
begin
  DisableRefresh;
  inherited Destroy;
end;

function TSelfRefreshTable.RefreshCallBack(CBInfo: Pointer): CBRType;
begin
  Result := cbrUSEDEF;
  Refresher.DatasetNeedsRefresh(self, @DoRefreshSelfRefreshTable);
end;

procedure TSelfRefreshTable.DoAfterOpen;
begin
  EnableRefresh;
  inherited DoAfterOpen;
end;

procedure TSelfRefreshTable.DoBeforeClose;
begin
  DisableRefresh;
  inherited DoBeforeClose;
end;

procedure TSelfRefreshTable.DoBeforeRefresh;
begin
  if Assigned(FBeforeRefresh) then
    FBeforeRefresh(self);
end;

procedure TSelfRefreshTable.DoAfterRefresh;
begin
  if Assigned(FAfterRefresh) then
    FAfterRefresh(self);
end;

procedure TSelfRefreshTable.SetSelfRefreshEnabled(Value: Boolean);
begin
  if (FSelfRefreshEnabled = Value) then
    exit;
  FSelfRefreshEnabled := Value;
  if Value then
    EnableRefresh
  else
    DisableRefresh;
end;

procedure TSelfRefreshTable.EnableRefresh;
begin
  if not (csDesigning in Componentstate) then
  begin
    if (FRefreshCallBack = nil) and FSelfRefreshEnabled then
    begin
      if not Assigned(Refresher) then
        Refresher := TRefresher.Create;
      try
        FRefreshCallback := TBDERefreshCallback.Create(self, RefreshCallBack);
      except
        FRefreshCallback := nil;
        exit;
      end;
      Refresher.AddDataset(self);
    end;
  end;
end;

procedure TSelfRefreshTable.DisableRefresh;
begin
  if Assigned(Refresher) and (FRefreshCallBack <> nil) then
  begin
    FRefreshCallBack.Free;
    FRefreshCallBack := nil;
    Refresher.RemoveDataset(self);
  end;
end;

procedure TSelfRefreshTable.DoRefresh(var Done: Boolean);
begin
  if (State = dsBrowse) then
  begin
    Done := true;
    DisableControls;
    try
      DoBeforeRefresh;
    { Refresh uses DbiForceReread to refresh all buffers for the table
      associated with the cursor in case remote updates took place: }
      Refresh;
      DoAfterRefresh;
    finally
      EnableControls;
    end;
  end
  else
    Done := false;
end;



{ ---------------------------------------------------------------------------- }

procedure ForceTableRefreshNow;
begin
  Refreshr.ForceTableRefreshNow;
end;

procedure SetRefreshCheckPeriod(Value: Integer);
begin
  Refreshr.SetRefreshCheckPeriod(Value);
end;

end.
