unit Rebar;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls,CommCtrl;

type
  TGripperStyle=(hbLeft,hbRight,hbNone);
  TRebarGripper=class(TCustomControl)
  private
    { Private declarations }
    FBorderWidth:integer;
    FOnEnter:TNotifyEvent;
    FOnExit:TNotifyEvent;
    procedure SetBorderWidth(Value:integer);
    procedure CMMouseEnter(var msg: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var msg: TMessage); message CM_MOUSELEAVE;
  protected
    { Protected declarations }
    procedure Paint;override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer);override;
  public
    { Public declarations }
    constructor Create(AOwner:TComponent);override;
    destructor Destroy; override;
  published
    { Published declarations }
    property BorderWidth:integer read FBorderWidth write SetBorderWidth;
    property OnEnter:TNotifyEvent read FOnEnter write FOnEnter;
    property OnExit :TNotifyEvent read FOnExit write FOnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

TCollapseEvent=procedure(Sender: TObject; var Collapse:boolean;Button: TMouseButton; Shift: TShiftState; X, Y: Integer) of object;
TBevelBorder=(bfBottom,bfLeft,bfRight,bfTop);
TBevelBorders = set of TBevelBorder;
TBevelEdge=(beBump,beEtched,beRaised,beSunken);
TRebar = class(TCustomPanel)
  private
    { Private declarations }
    FHandCursor:TCursor;
    Resizing:integer;
    FImage:TBitmap;
    FCaption:string;
    FCollapsable:boolean;
    FTxtRect:TRect;
    FGripperStyle:TGripperStyle;
    FGripper:TRebarGripper;
    FBevelBorder:TBevelBorders;
    FBevelEdge:TBevelEdge;
    FOnGripperEnter:TNotifyEvent;
    FOnGripperExit:TNotifyEvent;
    FOnGripperDown:TMouseEvent;
    FOnGripperUp:TMouseEvent;
    FOnCollapse:TCollapseEvent;
    FOnGripperMove:TMouseMoveEvent;
    FOnResize:TNotifyEvent;
    procedure SetCaption(Value:string);
    procedure SetImage(Value:TBitmap);
    procedure SetGripperStyle(Value:TGripperStyle);
    procedure SetBevelBorder(Value:TBevelBorders);
    procedure SetBevelEdge(Value:TBevelEdge);
    procedure DoText;
    procedure DoHandleBar;
    procedure DoBevels;
  protected
    { Protected declarations }
    procedure Paint;override;
    procedure Resize;override;
    procedure OnEnterGripper(Sender:TObject);
    procedure OnExitGripper(Sender:TObject);
    procedure OnGripperDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure OnGripperUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure OnGripperMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer);override;
  public
    { Public declarations }
    constructor Create(AOwner:TComponent);override;
    destructor Destroy; override;
  published
    { Published declarations }
    property Background:TBitmap read FImage write SetImage nodefault ;
    property Align;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderStyle;
    property BorderWidth;
//    property BevelBorder:TBevelBorders read FBevelBorder write SetBevelBorder;
//    property BevelEdge:TBevelEdge read FBevelEdge write SetBevelEdge;
    property Caption:string read FCaption write SetCaption;
    property Color;
    property Cursor;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property Gripper:TGripperStyle read FGripperStyle write SetGripperStyle;
    property Height;
    property HelpContext;
    property Hint;
    property Left;
    property Name;
    property ParentColor;
    property ParentShowHint;
    property ShowHint;
    property Tag;
    property Top;
    property Visible;
    property Width;
    property OnGripperEnter : TNotifyEvent read FOnGripperEnter write FOnGripperEnter;
    property OnGripperExit :TNotifyEvent read FOnGripperExit write FOnGripperExit;
    property OnGripperMouseDown:TMouseEvent read FOnGripperDown write FOnGripperDown;
    property OnGripperMouseUp:TMouseEvent read FOnGripperUp write FOnGripperUp;
    property OnGripperMouseMove:TMouseMoveEvent read FOnGripperMove write FOnGripperMove;
    property OnCollapse:TCollapseEvent read FOnCollapse write FOnCollapse;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize:TNotifyEvent read FOnResize write FOnResize;
    property OnStartDrag;
  end;

procedure Register;

implementation

{$R Gripper.res}

procedure DrawLine(Canvas:TCanvas;Color:TColor;Width,X1,Y1,X2,Y2:integer);
begin
  with Canvas do
  begin
    Pen.Color := Color;
    Pen.Width := Width;
    MoveTo(X1,Y1);
    LineTo(X2,Y2);
  end;
end;

{ TRebarGripper }

constructor TRebarGripper.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  Top := 0;
  Left := 0;
  Width := 6;
  Height := 60;
  FBorderWidth := 1;
  Cursor := crSizeWE;
end;

destructor TRebarGripper.Destroy;
begin
  inherited Destroy;
end;

procedure TRebarGripper.Paint;
var Rec:TRect;Mid:integer;
begin
   if Width mod 2 <> 0 then
     Width := Width + 1;
   Mid := Width div 2 - 1;
   Rec := GetClientRect;
   Canvas.Brush.Color := clBtnFace;
   Canvas.Pen.Color := clBtnFace;
   Canvas.Rectangle(Left,Top,Width,Height);

   Frame3d(Canvas,Rec,clBtnHighLight,clBtnShadow,FBorderWidth);
   DrawLine(Canvas,clBtnShadow,FBorderWidth,Mid,0,Mid,Height);
   Mid := Mid + FBorderWidth;
   DrawLine(Canvas,clBtnHighlight,FborderWidth,Mid,Height-(FBorderWidth + FBorderWidth),Mid,0);
end;

procedure TRebarGripper.SetBorderWidth(Value:integer);
begin
  if FBorderWidth <> Value then
  begin
    FBorderWidth := Value;
    Invalidate;
  end;
end;

procedure TRebarGripper.CMMouseEnter(var msg: TMessage);
begin
  if Assigned(FOnEnter) then FOnEnter(self);
end;

procedure TRebarGripper.CMMouseLeave(var msg: TMessage);
begin
  if Assigned(FOnExit) then FOnExit(self);
end;

procedure TRebarGripper.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(OnMouseDown) then OnMouseDown(self,Button,Shift,X,Y);
end;

procedure TRebarGripper.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(OnMouseUp) then OnMouseUp(self,Button,Shift,X,Y);
end;
procedure TRebarGripper.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(OnMouseMove) then OnMouseMove(self,Shift,X,Y);
end;

{ TRebar }

constructor TRebar.Create(Aowner:TComponent);
begin
 inherited Create(AOwner);
 FHandCursor := 1;
 Screen.Cursors[FHandCursor] := LoadCursor(hInstance,PChar('HANDCURSOR'));
 FImage := TBitmap.Create;
 Width  := 200;
 Height := 100;
 Align  := alNone;
 Top    := 100;
 Left   := 100;
 Width  := 150;
 Height := 80;
 Resizing := 0;
 BevelInner := bvRaised;
 BevelOuter := bvLowered;
 FCollapsable := False;
 FGripperStyle := hbNone;
 FGripper := TRebarGripper.Create(self);
 with FGripper do
 begin
   Parent  := self;
   OnEnter := OnEnterGripper;
   OnExit  := OnExitGripper;
   OnMouseDown := OnGripperDown;
   OnMouseUp := OnGripperUp;
   OnMouseMove := OnGripperMove;
 end;
 FTxtRect:= Rect(0,0,0,Height);
 FCaption := '';
end;

destructor TRebar.Destroy;
begin
 FImage.Free;
 FGripper.Free;
 inherited Destroy;
end;

procedure TRebar.Paint;
var
  X, Y, W, H: LongInt;
  Dest,Source:TRect;
  TopColor, BottomColor: TColor;
var Tmp:TBitmap;

  procedure AdjustColors(Bevel: TPanelBevel);
  begin
    if Bevel = bvLowered then TopColor := clBtnShadow
    else TopColor := clBtnHighlight;

    if Bevel = bvLowered then BottomColor := clBtnHighlight
    else BottomColor := clBtnShadow;
  end;
begin


  if Resizing > 1 then
  begin
    Dec(Resizing);
//    Exit;
  end;



  if FImage.Empty then
  begin
    Canvas.Brush.Color := Color;
    Canvas.Pen.Color := Color;
    Canvas.FillRect(GetClientRect);
  end
  else
  begin
    with FImage do begin
      W := Width;
      H := Height;
    end;

    Tmp := TBitmap.Create;
    Tmp.Width := Width;
    Tmp.Height := Height;


    Y := 0;
    Source := Rect(0,0,W,H);
    while Y < Height do begin
      X := 0;
      while X < Width do
      begin
        Dest := Rect(X,Y,X+W,Y+H);
        Tmp.Canvas.CopyRect(Dest,FImage.Canvas,Source);
//        BitBlt(Tmp.Canvas.Handle,X,Y,W,H,FImage.Canvas.Handle,0,0,SRCCOPY);
//        BitBlt(Canvas.Handle,X,Y,W,H,FImage.Canvas.Handle,0,0,SRCCOPY);
        Inc(X, W);
      end;
      Inc(Y, H);
    end;
    Canvas.Draw(0,0,Tmp);
//    BitBlt(Canvas.Handle,0,0,Width,Height,Tmp.Canvas.Handle,0,0,SRCCOPY);
    Tmp.Free;
  end;

  { draw handlebar }
  DoHandleBar;

  { draw outline }
  Dest := GetClientRect;
  if BevelOuter <> bvNone then
  begin
    AdjustColors(BevelOuter);
    Frame3D(Canvas, Dest, TopColor, BottomColor, BevelWidth);
  end;

  Frame3D(Canvas, Dest, Color, Color, BorderWidth);
  if BevelInner <> bvNone then
  begin
    AdjustColors(BevelInner);
    Frame3D(Canvas, Dest, TopColor, BottomColor, BevelWidth);
  end;
  { draw text }
  DoText;
  DoBevels;
  Resizing := 0;
end;

procedure TRebar.DoText;
var Lft,TTop:integer;
begin
  Canvas.Font := Font;
  if FCaption <> '' then
  begin
    case FGripperStyle of
      hbLeft:   Lft := FGripper.Width + 8;
      hbRight:  Lft := Width - Canvas.TextWidth(FCaption) - 16;
    else
      Lft := FGripper.Width;
    end;
    TTop := (Height - Canvas.TextHeight(FCaption)) div 2;
    FTxtRect := Rect(Lft, 0, Lft + Canvas.TextWidth(FCaption),Height { TTop + Canvas.TextHeight(FCaption)});
    SetBkMode(Canvas.Handle,Windows.Transparent);
    Canvas.TextOut(Lft,TTop,FCaption);
  end;
end;


procedure TRebar.DoHandleBar;
begin
  case FGripperStyle of
    hbNone: FGripper.Left  := -100;
    hbRight: FGripper.Left := Width - FGripper.Width - 4;
    hbLeft: FGripper.Left  := 4;
  end;
  FGripper.Top    := 4;
  FGripper.Height := Height - 8;
end;

procedure TRebar.SetCaption(Value:string);
begin
	if FCaption <> Value then
 begin
   FCaption := Value;
   Invalidate;
 end;
end;

procedure TRebar.SetImage(Value:TBitmap);
begin
    FImage.Assign(Value);
    Invalidate;
end;

procedure TRebar.SetGripperStyle(Value:TGripperStyle);
begin
  if FGripperStyle <> Value then
  begin
    FGripperStyle := Value;
    Invalidate;
  end;
end;

{
TBevelBorders=(bfBottom,bfLeft,bfRight,bfTop,);
TBevelEdge=(beEdge,beBump,beEtched,beSunken);
}

procedure TRebar.DoBevels;
var fFlags,fEdge:integer;Dest:TRect;
begin
     fFlags := 0;
     fEdge := 0;

     if bfBottom in FBevelBorder then
        fFlags := fFlags or BF_BOTTOM;
     if bfLeft in FBevelBorder then
        fFlags := fFlags or BF_LEFT;
     if bfRight in FBevelBorder then
        fFlags := fFlags or BF_RIGHT;
     if bfTop in FBevelBorder then
       fFlags :=  fFlags or BF_TOP;

     case FBevelEdge of
       beBump:     fEdge := EDGE_BUMP;
       beEtched:   fEdge := EDGE_ETCHED;
       beRaised:   fEdge := EDGE_RAISED;
       beSunken:   fEdge := EDGE_SUNKEN;
     end;

     Canvas.Pen.Width := 1;
     Dest := Rect(0,0,Width,Height);
     DrawEdge(Canvas.Handle,Dest,fEdge,fFlags);
end;

procedure TRebar.SetBevelBorder(Value:TBevelBorders);
begin
   if FBevelBorder <> Value then
     FBevelBorder := Value;
   Invalidate;
end;

procedure TRebar.SetBevelEdge(Value:TBevelEdge);
begin
  if FBevelEdge <> Value then
    FBevelEdge := Value;
  Invalidate;
end;

procedure TRebar.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  if PtInRect(FTxtRect,Point(X,Y)) then
  begin
    FCollapsable := True;
    if Cursor <> FHandCursor then
      Cursor := FHandCursor;
  end
  else
  begin
    FCollapsable := False;
    if Cursor <> crDefault then
      Cursor := crDefault;
  end;

  if Assigned(OnMouseMove) then OnMouseMove(self,Shift,X,Y);
end;

procedure TRebar.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var Collapse:boolean;
begin
   Collapse := False;
   if FCollapsable then
   begin
     if Assigned(FOnCollapse) then FOnCollapse(self,Collapse,button,shift,x,y);
     if Collapse then
     begin
       Width := Canvas.TextWidth(FCaption) + FGripper.Width + 14;
       Invalidate;
     end;
   end;
		if Assigned(OnMouseDown) then OnMouseDown(self,Button,shift,x,y);
end;

procedure TRebar.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
		if Assigned(OnMouseUp) then OnMouseUp(self,button,shift,x,y);
end;

procedure TRebar.Resize;
begin
  Inc(Resizing);
  if Assigned(FOnResize) then FOnResize(self);
end;

procedure TRebar.OnEnterGripper(Sender:TObject);
begin
  if Assigned(FOnGripperEnter) then FOnGripperEnter(self);
end;

procedure TRebar.OnExitGripper(Sender:TObject);
begin
  if Assigned(FOnGripperExit) then FOnGripperExit(self);
end;

procedure TRebar.OnGripperDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FOnGripperDown) then FOnGripperDown(self,Button,Shift,X,Y);
end;

procedure TRebar.OnGripperUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FOnGripperUp) then FOnGripperUp(self,Button,Shift,X,Y);
end;

procedure TRebar.OnGripperMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FOnGripperMove) then FOnGripperMove(self,Shift,X,Y);
end;

procedure Register;
begin
  RegisterComponents('Samples', [TRebar]);
end;

end.
