{*****************************************************}
{       TLineList 1.3 for Delphi 1.0/3.0              }
{                                                     }
{       Copyright (c) 1997 Tibor F. Liska             }
{       Tel/Fax:    +36-1-165-2019                    }
{       Office:     +36-1-269-8284                    }
{       E-mail: liska@sztaki.hu                       }
{*****************************************************}
unit LineList;

interface
uses SysUtils, Classes, Strings;

type
  TLineList = class(TStringList)
  private
      FPos,
      FSize: Longint;
      FName: TFileName;
      ReadAhead : Integer;
      From : TFileStream;
      BufferStart,
      BufferEnd,
      LineRest : PChar;
    procedure BufferFree;
    procedure AfterChange(Sender: TObject);
  public
    constructor Create(const Fn: TFileName; Ahead: Integer);
    destructor Destroy; override;
    procedure Reset;
    property FileName : TFileName read FName;
    property FilePos  : Longint   read FPos;
    property FileSize : Longint   read FSize;
  end;

implementation

const
      BufferSize = 4096;

procedure TLineList.BufferFree;
begin
  if From <> nil then From.Free;
  if BufferStart <> nil then FreeMem(BufferStart, BufferSize);
  From := nil;
  BufferStart := nil;
end;

procedure TLineList.AfterChange(Sender: TObject);
  const
       CR  = Chr($0D);
       LF  = Chr($0A);
  var
      n : Integer;
      P, LineFirst : PChar;
      S : string;
      FileOver : Boolean;
begin
  if (From = nil) or (Count >= ReadAhead) then Exit;
  BeginUpdate;
try
  repeat
    n := BufferEnd - LineRest;
    if n <> 0 then
      System.Move(LineRest^, BufferStart^, n);
    LineRest := BufferStart + n;
    LineRest := LineRest + From.Read(LineRest^, BufferEnd - LineRest);
    FileOver := LineRest < BufferEnd;
    if FileOver then
      LineRest^ := #0
    else
    begin
      LineRest := LineStart(BufferStart, LineRest);
      if LineRest = BufferStart then
        raise Exception.Create('Too long line');
    end;
    P := BufferStart;
    while P < LineRest do
    begin
      LineFirst := P;
      while not (P^ in [#0, CR, LF]) do Inc(P);
      SetString(S, LineFirst, P - LineFirst);
      Add(S);
      if P^ = CR then Inc(P);
      if P^ = LF then Inc(P);
    end;
  until (Count > ReadAhead) or FileOver;
  FPos := From.Position;
  if FileOver then BufferFree;
finally
  EndUpdate;
end; end;

constructor TLineList.Create(const Fn: TFileName; Ahead: Integer);
begin
  inherited Create;
  FName := Fn;
  ReadAhead := Ahead;
  OnChange := AfterChange;
  Reset;
end;

destructor TLineList.Destroy;
begin
  BufferFree;
  inherited Destroy;
end;

procedure TLineList.Reset;
begin
  if From = nil then
  begin
    From := TFileStream.Create(FName, fmOpenRead);
    FSize := From.Size;
    GetMem(BufferStart, BufferSize);
  end;
  From.Seek(0, 0);
  BufferEnd := BufferStart + BufferSize;
  LineRest := BufferEnd;
  Clear;
  Changed;
end;

end.
