{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      System Utilities Unit                          *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

unit ProUtils;

{$I PRO.INC}

interface

uses {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF} SysUtils,
  Classes;


{ TProFileStream }

const
  { Constants for Origin parameter of the Seek method }
  fsBegin   = 0;
  fsCurrent = 1;
  fsEnd     = 2;

type
  TProFileStream = class(TFileStream)
  public
    function ReadLn : string;
    procedure WriteLn(S : string);
    function Truncate : LongInt;
    function BOF : Boolean;
    function EOF : Boolean;
    procedure SeekToBegin;
    procedure SeekToEnd;
  end;


{ File management routines }

procedure FileCopy(const SourceFileName, TargetFileName : string);
{ FileCopy copies file from SourceFileName to TargetFileName.
  If TargetFileName doesn't exist, it is created. If TargetFileName exists,
  it is rewritten. }

function DiskInDrive(Drive : Char) : Boolean;
{ DiskInDrive checks presence of a disk in the drive }


{ Miscellaneous routines }

function Min(I, J : LongInt) : LongInt;
{ Returns the smaller value of I and J }

function Max(I, J : LongInt) : LongInt;
{ Returns the greater value of I and J }

function MinFloat(I, J : Extended) : Extended;
{ Returns the smaller value of I and J }

function MaxFloat(I, J : Extended) : Extended;
{ Returns the greater value of I and J }

procedure ExchangeBytes(var I, J : Byte);
{ Exchanges bytes I and J }

procedure ExchangeWords(var I, J : Word);
{ Exchanges words I and J }

procedure ExchangeLongInts(var I, J : LongInt);
{ Exchanges long integers I and J }

function SwapNibbles(Value : Byte) : Byte;
{ Swaps the low and high nibbles of Value (SwapNibbles($F0) returns $0F) }

function SwapWords(Value : LongInt) : LongInt;
{ Swaps low- and high-order words of Value }

function HiWord(Value : LongInt) : Word;
{ Returns high-order word of Value }

function LoWord(Value : LongInt) : Word;
{ Returns low-order word of Value }

function MakeWord(High, Low : Byte) : Word;
{ Constructs a word from two bytes }

{$IFNDEF WIN32}
type
  SmallInt = Integer;
{$ENDIF}

function MakeInt(High, Low : Byte) : SmallInt;
{ Constructs a small integer from two bytes }

function MakeLongInt(High, Low : Word) : LongInt;
{ Constructs a long integer from two words }


{ String routines }

type
  TCharSet = set of Char;

{$IFNDEF WIN32}

type
  ShortString = string[255];

procedure SetLength(var S : string; NewLenght : Integer);
{ SetLength sets the length of a string varible }

procedure SetString(var S : string; Buffer : PChar; Len : Integer);
{ SetString sets the contents and length of the given string variable
  to the block of characters given by the Buffer and Len parameters }

function StringOfChar(Ch : Char; Count : Integer) : string;
{ StringOfChar returns a string containing Count characters
  with the character value given by Ch }

function TrimLeft(const S : string) : string;
{ TrimLeft trims leading spaces and control characters
  from the given string S }

function TrimRight(const S : string) : string;
{ TrimRight trims trailing spaces and control characters
  from the given string S }

function Trim(const S : string) : string;
{ Trim trims leading and trailing spaces and control characters
  from the given string S }

{$ENDIF}

function AddCharRight(const S : string; Ch : Char; Len : Integer) : string;
{ AddCharRight returns a string right-padded to length Len with Ch }

function AddSpaceRight(const S : string; Len : Integer) : string;
{ AddSpaceRight returns a string right-padded to length Len with blanks }

function AddCharLeft(const S : string; Ch : Char; Len : Integer) : string;
{ AddCharLeft returns a string left-padded to length Len with Ch }

function AddSpaceLeft(const S : string; Len : Integer) : string;
{ AddSpaceLeft returns a string left-padded to length Len with blanks }

function CenterCharStr(const S : string; Ch : Char; Width : Integer) : string;
{ CenterCharStr returns a string centered in a string of Ch with
  specified width }

function CenterSpaceStr(const S : string; Width : Integer) : string;
{ CenterSpaceStr returns a string centered in a blank string with
  specified width }

function IsCharInString(Ch : Char; const S : string) : Boolean;
{ IsCharInString returns True if S contains Ch }

function CountCharInString(Ch : Char; const S : string) : Integer;
{ CountCharInString returns numbers of Ch characters in S }

function GetWordCount(const S : string; Delimiters : TCharSet) : Integer;
{ GetWordCount returns numbers of words in S using given a set of word
  delimeters }

function GetWordStr(const S : string; Number : Integer;
  Delimiters : TCharSet) : string;
{ GetWordStr returns Number'th word in S using given a set of word
  delimeters }

function GetWordPosition(const S : string; Number : Integer;
  Delimiters : TCharSet) : Integer;
{ GetWordPosition returns position of Number'th word in S using given a set
  of word delimeters and 0 otherwise }


{ Convert routines }

function ConvertAnsiToOem(const S : string) : string;
{ ConvertAnsiToOem translates a string into the OEM-defined character set }

function ConvertOemToAnsi(const S : string) : string;
{ ConvertOemToAnsi translates a string from the OEM-defined
  character set into either an ANSI or a wide-character string }

implementation

{ TProFileStream }

function TProFileStream.ReadLn : string;
{ Reads chars from file from current position to end of line char #10#13
  (or to end of file) and returns their as string variable.
  Size of reading block:
  - for Delphi 16-bit: 255 chars;
  - for Delphi 32-bit: 1024 chars. If end of line char was not found then
    function reads next block until end of line char appears or file ends. }
{$IFDEF WIN32}
const
  BlockSize = 1024;
var
  BufSize : LongInt;
  P : Byte;
  ReadAgain : Boolean;
begin
  SetLength(Result, BlockSize);
  BufSize := Read(Result[1], BlockSize);
  SetLength(Result, BufSize);
  repeat
    P := Pos(#13#10, Result);
    ReadAgain := (P = 0) and not EOF;
    if ReadAgain then
    begin
      SetLength(Result, BufSize + BlockSize);
      BufSize := BufSize + Read(Result[BufSize + 1], BlockSize);
      SetLength(Result, BufSize);
    end;
  until not ReadAgain;
  if P > 0 then
  begin
    SetLength(Result, P - 1);
    Seek(P - BufSize + 1, fsCurrent);
  end;
end; { TProFileStream.ReadLn }
{$ELSE}
const
  BlockSize = 255;
var
  BufSize : LongInt;
  P : Byte;
begin
  BufSize := Read(Result[1], BlockSize);
  Result[0] := Char(Byte(BufSize));
  P := Pos(#13#10, Result);
  if P > 0 then
  begin
    Result[0] := Char(P - 1);
    Seek(P - BufSize + 1, fsCurrent);
  end;
end; { TProFileStream.ReadLn }
{$ENDIF}

procedure TProFileStream.WriteLn(S : string);
const
  EndOfLine : array[0..1] of Char = #13#10;
begin
  Write(S[1], Length(S));
  Write(EndOfLine, SizeOf(EndOfLine));
end; { TProFileStream.WriteLn }

function TProFileStream.Truncate : LongInt;
var
  NullBuffer : Byte;
begin
  Result := FileWrite(Handle, NullBuffer, 0);
end; { TProFileStream.Truncate }

function TProFileStream.BOF : Boolean;
begin
  BOF := Position = 0
end; { TProFileStream.BOF }

function TProFileStream.EOF : Boolean;
begin
  EOF := Position = Size;
end; { TProFileStream.EOF }

procedure TProFileStream.SeekToBegin;
begin
  FileSeek(Handle, 0, fsBegin);
end; { TProFileStream.SeekToBegin }

procedure TProFileStream.SeekToEnd;
begin
  FileSeek(Handle, 0, fsEnd);
end; { TProFileStream.SeekToEnd }


{ File management routines }

procedure FileCopy(const SourceFileName, TargetFileName : string);
{ FileCopy copies file from SourceFileName to TargetFileName.
  If TargetFileName doesn't exist, it is created. If TargetFileName exists,
  it is rewritten. }
var
  Source, Target : TFileStream;
begin
  Source := TFileStream.Create(SourceFileName, fmOpenRead);
  try
    Target := TFileStream.Create(TargetFileName, fmCreate or fmOpenWrite);
    try
      Target.CopyFrom(Source, Source.Size);
      FileSetDate(Target.Handle, FileGetDate(Source.Handle));
    finally
      Target.Free;
    end;
  finally
    Source.Free;
  end;
end; { FileCopy }

function DiskInDrive(Drive : Char) : Boolean;
{ DiskInDrive checks presence of a disk in the drive }
var
{$IFDEF WIN32}
  ErrorMode : Integer;
{$ELSE}
  ErrorMode : Word;
{$ENDIF}
begin
  if Drive in ['a'..'z'] then Dec(Drive, $20);
  if not (Drive in ['A'..'Z']) then
    raise EConvertError.Create('Not a valid drive ID');

  ErrorMode := SetErrorMode(SEM_FailCriticalErrors);
  try
    Result := DiskSize(Ord(Drive) - $40) <> -1;
  finally
    SetErrorMode(ErrorMode);
  end;
end; { DiskInDrive }


{ Miscellaneous routines }

function Min(I, J : LongInt) : LongInt;
{ Returns the smaller value of I and J }
begin
  if I < J then Result := I else Result := J;
end; { Min }

function Max(I, J : LongInt) : LongInt;
{ Returns the greater value of I and J }
begin
  if I > J then Result := I else Result := J;
end; { Max }

function MinFloat(I, J : Extended) : Extended;
{ Returns the smaller value of I and J }
begin
  if I < J then Result := I else Result := J;
end; { MinFloat }

function MaxFloat(I, J : Extended) : Extended;
{ Returns the greater value of I and J }
begin
  if I > J then Result := I else Result := J;
end; { MaxFloat }

procedure ExchangeBytes(var I, J : Byte);
{ Exchanges bytes I and J }
var
  Temp : Byte;
begin
  Temp := I;
  I := J;
  J := Temp;
end; { ExchangeBytes }

procedure ExchangeWords(var I, J : Word);
{ Exchanges words I and J }
var
  Temp : Word;
begin
  Temp := I;
  I := J;
  J := Temp;
end; { ExchangeWords }

procedure ExchangeLongInts(var I, J : LongInt);
{ Exchanges long integers I and J }
var
  Temp : LongInt;
begin
  Temp := I;
  I := J;
  J := Temp;
end; { ExchangeLongInts }

function SwapNibbles(Value : Byte) : Byte;
{ Swaps the low and high nibbles of Value (SwapNibbles($F0) returns $0F) }
begin
  Result := (Value shr 4) or (Value shl 4);
end; { SwapNibbles }

function SwapWords(Value : LongInt) : LongInt;
{ Swaps low- and high-order words of Value }
begin
  LongRec(Result).Lo := Hi(Value);
  LongRec(Result).Hi := Lo(Value);
end; { SwapWords }

function HiWord(Value : LongInt) : Word;
{ Returns high-order word of Value }
begin
  Result := LongRec(Value).Hi;
end; { HiWord }

function LoWord(Value : LongInt) : Word;
{ Returns low-order word of Value }
begin
  Result := LongRec(Value).Lo;
end; { LoWord }

function MakeWord(High, Low : Byte) : Word;
{ Constructs a word from two bytes }
begin
  WordRec(Result).Lo := Low;
  WordRec(Result).Hi := High;
end; { MakeWord }

function MakeInt(High, Low : Byte) : SmallInt;
{ Constructs a small integer from two bytes }
begin
  WordRec(Result).Lo := Low;
  WordRec(Result).Hi := High;
end; { MakeSmallInt }

function MakeLongInt(High, Low : Word) : LongInt;
{ Constructs a long integer from two words }
begin
  LongRec(Result).Lo := Low;
  LongRec(Result).Hi := High;
end; { MakeLongInt }


{ String routines }

{$IFNDEF WIN32}

procedure SetLength(var S : string; NewLenght : Integer);
{ SetLength sets the length of a string varible }
begin
  S[0] := Chr(Byte(NewLenght));
end; { SetLength }

procedure SetString(var S : string; Buffer : PChar; Len : Integer);
{ SetString sets the contents and length of the given string variable
  to the block of characters given by the Buffer and Len parameters }
begin
  S[0] := Chr(Byte(Len));
  Move(Buffer^, S, Len);
end; { SetString }

function StringOfChar(Ch : Char; Count : Integer) : string;
{ StringOfChar returns a string containing Count characters
  with the character value given by Ch }
begin
  Result[0] := Chr(Byte(Count));
  FillChar(Result[1], Count, Ch);
end; { StringOfChar }

function TrimLeft(const S : string) : string;
{ TrimLeft trims leading spaces and control characters
  from the given string S }
begin
  Result := S;
  while (Length(Result) > 0) and (Result[1] <= #32) do
    Delete(Result, 1, 1);
end; { TrimLeft }

function TrimRight(const S : string) : string;
{ TrimRight trims trailing spaces and control characters
  from the given string S }
begin
  Result := S;
  while (Length(Result) > 0) and (Result[Length(Result)] <= #32) do
    Dec(Result[0]);
end; { TrimRight }

function Trim(const S : string) : string;
{ Trim trims leading and trailing spaces and control characters
  from the given string S }
begin
  Result := S;
  while (Length(Result) > 0) and (Result[1] <= #32) do
    Delete(Result, 1, 1);
  while (Length(Result) > 0) and (Result[Length(Result)] <= #32) do
    Dec(Result[0]);
end; { Trim }

{$ENDIF}

function AddCharRight(const S : string; Ch : Char; Len : Integer) : string;
{ AddCharRight returns a string right-padded to length Len with Ch }
begin
  if Length(S) >= Len then
    Result := S
  else
    Result := S + StringOfChar(Ch, Len - Length(S));
end; { AddCharRight }

function AddSpaceRight(const S : string; Len : Integer) : string;
{ AddSpaceRight returns a string right-padded to length Len with blanks }
begin
  Result := AddCharRight(S, ' ', Len);
end; { AddSpaceRight }

function AddCharLeft(const S : string; Ch : Char; Len : Integer) : string;
{ AddCharLeft returns a string left-padded to length Len with Ch }
begin
  if Length(S) >= Len then
    Result := S
  else
    Result := StringOfChar(Ch, Len - Length(S)) + S;
end; { AddCharLeft }

function AddSpaceLeft(const S : string; Len : Integer) : string;
{ AddSpaceLeft returns a string left-padded to length Len with blanks }
begin
  Result := AddCharLeft(S, ' ', Len);
end; { AddSpaceLeft }

function CenterCharStr(const S : string; Ch : Char; Width : Integer) : string;
{ CenterCharStr returns a string centered in a string of Ch with
  specified width }
var
  TmpStr : string;
begin
  if Length(S) > Width then
    Result := S
  else
  begin
    TmpStr := StringOfChar(Ch, Width - Length(S));
    Insert(S, TmpStr, Length(TmpStr) shr 1 + 1);
    Result := TmpStr;
  end;
end; { CenterCharStr }

function CenterSpaceStr(const S : string; Width : Integer) : string;
{ CenterSpaceStr returns a string centered in a blank string with
  specified width }
begin
  Result := CenterCharStr(S, ' ', Width);
end; { CenterSpaceStr }

function IsCharInString(Ch : Char; const S : string) : Boolean;
  { IsCharInString returns True if S contains Ch }
var
  I : Integer;
begin
  I := 1;
  while (I <= Length(S)) and (Ch <> S[I]) do Inc(I);
  Result := I <= Length(S);
end; { IsCharInString }

function CountCharInString(Ch : Char; const S : string) : Integer;
{ CountCharInString returns numbers of Ch characters in S }
var
  I : Integer;
begin
  Result := 0;
  for I := 1 to Length(S) do
    if Ch = S[I] then Inc(Result);
end; { CountCharInString }

function GetWordCount(const S : string; Delimiters : TCharSet) : Integer;
{ GetWordCount returns numbers of words in S using given a set of word
  delimeters }
var
  InWhite : Boolean;
{$IFDEF WIN32}
  WordCount, CurrPos : Integer;
{$ELSE}
  WordCount, CurrPos : Byte;
{$ENDIF}
begin
  InWhite := True;
  WordCount := 0;
  CurrPos := 0;

  repeat
    Inc(CurrPos);
    if CurrPos <= Length(S) then
      { If current char is delimiter then it is first char of delimiter }
      if S[CurrPos] in Delimiters then
        InWhite := True
      { Else if previous char is delimeter then it begins a new word }
      else if InWhite then
      begin
        Inc(WordCount);
        InWhite := False;
      end;
  until CurrPos > Length(S);            { Review all string }

  Result := WordCount;
end; { GetWordCount }

function GetWordStr(const S : string; Number : Integer;
  Delimiters : TCharSet) : string;
{ GetWordStr returns Number'th word in S using given a set of word
  delimeters }
var
  InWhite : Boolean;
{$IFDEF WIN32}
  WordNumber, StartPos, EndPos : Integer;
{$ELSE}
  WordNumber, StartPos, EndPos : Byte;
{$ENDIF}
begin
  InWhite := True;
  WordNumber := 0;
  StartPos := 0;

  { Find word with necessary number }
  repeat
    Inc(StartPos);
    if StartPos <= Length(S) then
      { If current char is delimiter then it is first char of delimiter }
      if S[StartPos] in Delimiters then
        InWhite := True
      { Else if previous char is delimeter then it begins a new word }
      else if InWhite then
      begin
        Inc(WordNumber);
        InWhite := False;
      end;
  until (StartPos > Length(S)) or (WordNumber = Number);

  { If word with necessary number found then cut out and return it }
  if WordNumber = Number then
  begin
    EndPos := StartPos;
    while (EndPos <= Length(S)) and not InWhite do
    begin
      Inc(EndPos);
      if S[EndPos] in Delimiters then
        InWhite := True;
    end;
    Result := Copy(S, StartPos, EndPos - StartPos);
  end
  else
    Result := '';
end; { GetWordStr }

function GetWordPosition(const S : string; Number : Integer;
  Delimiters : TCharSet) : Integer;
{ GetWordPosition returns position of Number'th word in S using given a set
  of word delimeters and 0 otherwise }
var
  InWhite : Boolean;
{$IFDEF WIN32}
  WordNumber, StartPos : Integer;
{$ELSE}
  WordNumber, StartPos : Byte;
{$ENDIF}
begin
  InWhite := True;
  WordNumber := 0;
  StartPos := 0;

  { Find word with necessary number }
  repeat
    Inc(StartPos);
    if StartPos <= Length(S) then
      { If current char is delimiter then it is first char of delimiter }
      if S[StartPos] in Delimiters then
        InWhite := True
      { Else if previous char is delimeter then it begins a new word }
      else if InWhite then
      begin
        Inc(WordNumber);
        InWhite := False;
      end;
  until (StartPos > Length(S)) or (WordNumber = Number);

  { Return position if word with necessary number was found
    and 0 otherwise }
  if WordNumber = Number then
    Result := StartPos
  else
    Result := 0;
end; { GetWordPosition }


{ Convert routines }

function ConvertAnsiToOem(const S : string) : string;
{ ConvertAnsiToOem translates a string into the OEM-defined
  character set }
{$IFNDEF WIN32}
var
  Source, Dest : array[0..255] of Char;
{$ENDIF}
begin
{$IFDEF WIN32}
  SetLength(Result, Length(S));
  if Length(Result) > 0 then
    AnsiToOem(PChar(S), PChar(Result));
{$ELSE}
  if Length(Result) > 0 then
  begin
    AnsiToOem(StrPCopy(Source, S), Dest);
    Result := StrPas(Dest);
  end;
{$ENDIF}
end; { ConvertAnsiToOem }

function ConvertOemToAnsi(const S : string) : string;
{ ConvertOemToAnsi translates a string from the OEM-defined
  character set into either an ANSI or a wide-character string }
{$IFNDEF WIN32}
var
  Source, Dest : array[0..255] of Char;
{$ENDIF}
begin
{$IFDEF WIN32}
  SetLength(Result, Length(S));
  if Length(Result) > 0 then
    OemToAnsi(PChar(S), PChar(Result));
{$ELSE}
  if Length(Result) > 0 then
  begin
    OemToAnsi(StrPCopy(Source, S), Dest);
    Result := StrPas(Dest);
  end;
{$ENDIF}
end; { ConvertOemToAnsi }

end.
