{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      Lists Unit                                     *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

unit ProLists;

{$I PRO.INC}

interface

uses {$IFDEF WIN32} Windows, {$ELSE} WinTypes, {$ENDIF} SysUtils, Classes;

type

{ Exception classes }

  EIntegerListError = class(EListError);
  EFloatListError = class(EListError);

{ TIntegerList }

  PLongArray = ^TLongArray;
  TLongArray = array[0..MaxListSize - 1] of LongInt;

  TIntegerList = class(TPersistent)
  private
    { Variables for properties }
    FList : TList;
    FMin : LongInt;
    FMax : LongInt;
    FDuplicates : TDuplicates;
    FSorted : Boolean;
    { Property access methods }
    procedure SetMin(Value : LongInt);
    procedure SetMax(Value : LongInt);
    procedure SetSorted(Value : Boolean);
    function GetList : PLongArray;
    { Private methods }
    procedure ReadMin(Reader : TReader);
    procedure WriteMin(Writer : TWriter);
    procedure ReadMax(Reader : TReader);
    procedure WriteMax(Writer : TWriter);
    procedure ReadIntegers(Reader : TReader);
    procedure WriteIntegers(Writer : TWriter);
    procedure QuickSort(L, R : Integer);
  protected
    procedure Error; virtual;
    procedure DefineProperties(Filer : TFiler); override;
    { Property access methods }
    function GetCount : Integer; virtual;
    function Get(Index : Integer) : LongInt; virtual;
    procedure Put(Index : Integer; Value : LongInt); virtual;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(Value : LongInt) : Integer; virtual;
    function AddIntegers(IntegerList : TIntegerList) : Integer; virtual;
    procedure Assign(Source : TPersistent); override;
    procedure Clear; virtual;
    procedure Delete(Index : Integer); virtual;
    function Find(Value : LongInt; var Index : Integer) : Boolean; virtual;
    function Equals(IntegerList : TIntegerList) : Boolean;
    procedure Exchange(Index1, Index2 : Integer); virtual;
    function IndexOf(Value : LongInt) : Integer; virtual;
    procedure Insert(Index : Integer; Value : LongInt); virtual;
    procedure Move(CurIndex, NewIndex : Integer); virtual;
    procedure Sort; virtual;
    property Count : Integer read GetCount;
    property Items[Index : Integer] : LongInt read Get write Put; default;
    property Min : LongInt read FMin write SetMin;
    property Max : LongInt read FMax write SetMax;
    property Duplicates: TDuplicates read FDuplicates write FDuplicates;
    property Sorted: Boolean read FSorted write SetSorted;
    property List : PLongArray read GetList;
  end;

{ TFloatList }

  TFloatList = class(TPersistent)
  private
    { Variables for properties }
    FList : TList;
    FMin : Extended;
    FMax : Extended;
    FDuplicates : TDuplicates;
    FSorted : Boolean;
    { Property access methods }
    procedure SetMin(Value : Extended);
    procedure SetMax(Value : Extended);
    procedure SetSorted(Value : Boolean);
    { Private methods }
    procedure ReadMin(Reader : TReader);
    procedure WriteMin(Writer : TWriter);
    procedure ReadMax(Reader : TReader);
    procedure WriteMax(Writer : TWriter);
    procedure ReadFloats(Reader : TReader);
    procedure WriteFloats(Writer : TWriter);
    procedure QuickSort(L, R : Integer);
  protected
    procedure Error; virtual;
    procedure DefineProperties(Filer : TFiler); override;
    { Property access methods }
    function GetCount : Integer; virtual;
    function Get(Index : Integer) : Extended; virtual;
    procedure Put(Index : Integer; Value : Extended); virtual;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(Value : Extended) : Integer; virtual;
    function AddFloats(FloatList : TFloatList) : Integer; virtual;
    procedure Assign(Source : TPersistent); override;
    procedure Clear; virtual;
    procedure Delete(Index : Integer); virtual;
    function Find(Value : Extended; var Index : Integer) : Boolean; virtual;
    function Equals(FloatList : TFloatList) : Boolean;
    procedure Exchange(Index1, Index2 : Integer); virtual;
    function IndexOf(Value : Extended) : Integer; virtual;
    procedure Insert(Index : Integer; Value : Extended); virtual;
    procedure Move(CurIndex, NewIndex : Integer); virtual;
    procedure Sort; virtual;
    property Count : Integer read GetCount;
    property Items[Index : Integer] : Extended read Get write Put; default;
    property Min : Extended read FMin write SetMin;
    property Max : Extended read FMax write SetMax;
    property Duplicates: TDuplicates read FDuplicates write FDuplicates;
    property Sorted: Boolean read FSorted write SetSorted;
  end;

implementation

uses ProConst;

{ Utility routines }

procedure IntegerListError(Ident : Integer);
begin
  raise EIntegerListError.CreateRes(Ident);
end; { IntegerListError }

procedure FloatListError(Ident : Integer);
begin
  raise EFloatListError.CreateRes(Ident);
end; { FloatListError }


{ TIntegerList }

constructor TIntegerList.Create;
{ Overrides the constructor to initialize variables }
begin
  inherited Create;
  FList := TList.Create;
  FMin := -MaxLongInt;
  FMax := MaxLongInt;
  FDuplicates := dupAccept;
end; { TIntegerList.Create }

destructor TIntegerList.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  Clear;
  FList.Free;
  inherited Destroy;
end; { TIntegerList.Destroy }

procedure TIntegerList.Assign(Source : TPersistent);
begin
  if Source is TIntegerList then
  begin
    FMin := TIntegerList(Source).Min;
    FMax := TIntegerList(Source).Max;
    FDuplicates := TIntegerList(Source).Duplicates;
    FSorted := TIntegerList(Source).Sorted;
    Clear;
    AddIntegers(TIntegerList(Source));
  end
  else
    inherited Assign(Source);
end; { TIntegerList.Assign }

procedure TIntegerList.Error;
begin
  IntegerListError(SIntegerListIndexError);
end; { TIntegerList.Error }

procedure TIntegerList.DefineProperties(Filer : TFiler);
begin
  Filer.DefineProperty('Min', ReadMin, WriteMin, FMin <> -MaxLongInt);
  Filer.DefineProperty('Max', ReadMax, WriteMax, FMax <> MaxLongInt);
  Filer.DefineProperty('Integers', ReadIntegers, WriteIntegers, GetCount > 0);
end; { TIntegerList.DefineProperties }

procedure TIntegerList.ReadMin(Reader : TReader);
begin
  FMin := Reader.ReadInteger;
end; { TIntegerList.ReadMin }

procedure TIntegerList.WriteMin(Writer : TWriter);
begin
  Writer.WriteInteger(FMin);
end; { TIntegerList.WriteMin }

procedure TIntegerList.ReadMax(Reader : TReader);
begin
  FMax := Reader.ReadInteger;
end; { TIntegerList.ReadMax }

procedure TIntegerList.WriteMax(Writer : TWriter);
begin
  Writer.WriteInteger(FMax);
end; { TIntegerList.WriteMax }

procedure TIntegerList.ReadIntegers(Reader : TReader);
begin
  Reader.ReadListBegin;
  Clear;
  while not Reader.EndOfList do
    Add(Reader.ReadInteger);
  Reader.ReadListEnd;
end; { TIntegerList.ReadIntegers }

procedure TIntegerList.WriteIntegers(Writer : TWriter);
var
  I : Integer;
begin
  Writer.WriteListBegin;
  for I := 0 to GetCount - 1 do
    Writer.WriteInteger(Get(I));
  Writer.WriteListEnd;
end; { TIntegerList.WriteIntegers }

procedure TIntegerList.SetMin(Value : LongInt);
var
  I : Integer;
begin
  if FMin <> Value then
  begin
    if Value > FMax then IntegerListError(SIntegerMinValueError);
    for I := 0 to GetCount - 1 do
      if Get(I) < Value then IntegerListError(SIntegerMinValueError);
    FMin := Value;
  end;
end; { TIntegerList.SetMin }

procedure TIntegerList.SetMax(Value : LongInt);
var
  I : Integer;
begin
  if FMax <> Value then
  begin
    if Value < FMin then IntegerListError(SIntegerMaxValueError);
    for I := 0 to GetCount - 1 do
      if Get(I) > Value then IntegerListError(SIntegerMaxValueError);
    FMax := Value;
  end;
end; { TIntegerList.SetMax }

procedure TIntegerList.QuickSort(L, R : Integer);
var
  I, J : Integer;
  P : LongInt;
begin
  I := L;
  J := R;
  P := LongInt(FList[(L + R) shr 1]);
  repeat
    while LongInt(FList[I]) < P do Inc(I);
    while LongInt(FList[J]) > P do Dec(J);
    if I <= J then
    begin
      FList.Exchange(I, J);
      Inc(I);
      Dec(J);
    end;
  until I > J;
  if L < J then QuickSort(L, J);
  if I < R then QuickSort(I, R);
end; { TIntegerList.QuickSort }

procedure TIntegerList.SetSorted(Value : Boolean);
begin
  if FSorted <> Value then
  begin
    if Value then Sort;
    FSorted := Value;
  end;
end; { TIntegerList.SetSorted }

function TIntegerList.GetList : PLongArray;
begin
  Result := PLongArray(FList.List);
end; { TIntegerList.GetList }

function TIntegerList.GetCount : Integer;
begin
  Result := FList.Count;
end; { TIntegerList.GetCount }

function TIntegerList.Get(Index : Integer) : LongInt;
begin
  if (Index < 0) or (Index >= GetCount) then Error;
  Result := LongInt(FList[Index]);
end; { TIntegerList.Get }

procedure TIntegerList.Put(Index : Integer; Value : LongInt);
begin
  if (Index < 0) or (Index >= GetCount) then Error;
  if (Value < FMin) or (Value > FMax) then IntegerListError(SIntegerListValueError);
  FList[Index] := Pointer(Value);
  if FSorted then Sort;
end; { TIntegerList.Put }

function TIntegerList.Add(Value : LongInt) : Integer;
begin
  if FDuplicates = dupAccept then
  begin
    Result := GetCount;
    Insert(Result, Value);
  end
  else if Find(Value, Result) then
    case FDuplicates of
      dupIgnore : Result := GetCount;
      dupError  : IntegerListError(SIntegerDuplicatesError);
    end;
end; { TIntegerList.Add }

function TIntegerList.AddIntegers(IntegerList : TIntegerList) : Integer;
begin
  for Result := 0 to IntegerList.Count - 1 do
    Add(IntegerList[Result]);
end; { TIntegerList.AddIntegers }

procedure TIntegerList.Clear;
begin
  FList.Clear;
end; { TIntegerList.Clear }

procedure TIntegerList.Delete(Index : Integer);
begin
  if (Index < 0) or (Index >= GetCount) then Error;
  FList.Delete(Index);
end; { TIntegerList.Delete }

function TIntegerList.Find(Value : LongInt; var Index : Integer) : Boolean;
begin
  Index := 0;
  while (Index < GetCount) and (Get(Index) <> Value) do Inc(Index);
  Result := Index < GetCount;
end; { TIntegerList.Find }

function TIntegerList.Equals(IntegerList : TIntegerList) : Boolean;
var
  I : Integer;
begin
  Result := GetCount = IntegerList.Count;
  if Result then
    for I := 0 to GetCount - 1 do
      if Get(I) <> IntegerList[I] then
      begin
        Result := False;
        Break;
      end;
end; { TIntegerList.Equals }

procedure TIntegerList.Exchange(Index1, Index2 : Integer);
begin
  if (Index1 < 0) or (Index1 >= GetCount) or
    (Index2 < 0) or (Index2 >= GetCount) then Error;
  FList.Exchange(Index1, Index2);
  FSorted := False;
end; { TIntegerList.Exchange }

function TIntegerList.IndexOf(Value : LongInt) : Integer;
begin
  while (Result < GetCount) and (Get(Result) <> Value) do Inc(Result);
  if Result = GetCount then Result := -1;
end; { TIntegerList.IndexOf }

procedure TIntegerList.Insert(Index : Integer; Value : LongInt);
begin
  if (Index < 0) or (Index > GetCount) then Error;
  if (Value < FMin) or (Value > FMax) then IntegerListError(SIntegerListValueError);
  FList.Expand.Insert(Index, Pointer(Value));
  FSorted := False;
end; { TIntegerList.Insert }

procedure TIntegerList.Move(CurIndex, NewIndex : Integer);
begin
  if (CurIndex < 0) or (CurIndex >= GetCount) or
    (NewIndex < 0) or (NewIndex >= GetCount) then Error;
  FList.Move(CurIndex, NewIndex);
  FSorted := False;
end; { TIntegerList.Move }

procedure TIntegerList.Sort;
begin
  if not FSorted and (GetCount > 1) then
    QuickSort(0, GetCount - 1);
end; { TIntegerList.Sort }


{ TFloatList }

constructor TFloatList.Create;
{ Overrides the constructor to initialize variables }
begin
  inherited Create;
  FList := TList.Create;
  FMin := 3.4e-4932;
  FMax := 1.1e4932;
  FDuplicates := dupAccept;
end; { TFloatList.Create }

destructor TFloatList.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  Clear;
  FList.Free;
  inherited Destroy;
end; { TFloatList.Destroy }

procedure TFloatList.Assign(Source : TPersistent);
begin
  if Source is TFloatList then
  begin
    FMin := TFloatList(Source).Min;
    FMax := TFloatList(Source).Max;
    FDuplicates := TFloatList(Source).Duplicates;
    FSorted := TFloatList(Source).Sorted;
    Clear;
    AddFloats(TFloatList(Source));
  end
  else
    inherited Assign(Source);
end; { TFloatList.Assign }

procedure TFloatList.Error;
begin
  FloatListError(SFloatListIndexError);
end; { TFloatList.Error }

procedure TFloatList.DefineProperties(Filer : TFiler);
begin
  Filer.DefineProperty('Min', ReadMin, WriteMin, FMin <> 3.4e-4932);
  Filer.DefineProperty('Max', ReadMax, WriteMax, FMax <> 1.1e4932);
  Filer.DefineProperty('Floats', ReadFloats, WriteFloats, GetCount > 0);
end; { TFloatList.DefineProperties }

procedure TFloatList.ReadMin(Reader : TReader);
begin
  FMin := Reader.ReadFloat;
end; { TFloatList.ReadMin }

procedure TFloatList.WriteMin(Writer : TWriter);
begin
  Writer.WriteFloat(FMin);
end; { TFloatList.WriteMin }

procedure TFloatList.ReadMax(Reader : TReader);
begin
  FMax := Reader.ReadFloat;
end; { TFloatList.ReadMax }

procedure TFloatList.WriteMax(Writer : TWriter);
begin
  Writer.WriteFloat(FMax);
end; { TFloatList.WriteMax }

procedure TFloatList.ReadFloats(Reader : TReader);
begin
  Reader.ReadListBegin;
  Clear;
  while not Reader.EndOfList do
    Add(Reader.ReadFloat);
  Reader.ReadListEnd;
end; { TFloatList.ReadFloats }

procedure TFloatList.WriteFloats(Writer : TWriter);
var
  I : Integer;
begin
  Writer.WriteListBegin;
  for I := 0 to GetCount - 1 do
    Writer.WriteFloat(Get(I));
  Writer.WriteListEnd;
end; { TFloatList.WriteFloats }

procedure TFloatList.SetMin(Value : Extended);
var
  I : Integer;
begin
  if FMin <> Value then
  begin
    if Value > FMax then FloatListError(SFloatMinValueError);
    for I := 0 to GetCount - 1 do
      if Get(I) < Value then FloatListError(SFloatMinValueError);
    FMin := Value;
  end;
end; { TFloatList.SetMin }

procedure TFloatList.SetMax(Value : Extended);
var
  I : Integer;
begin
  if FMax <> Value then
  begin
    if Value < FMin then FloatListError(SFloatMaxValueError);
    for I := 0 to GetCount - 1 do
      if Get(I) > Value then FloatListError(SFloatMaxValueError);
    FMax := Value;
  end;
end; { TFloatList.SetMax }

procedure TFloatList.QuickSort(L, R : Integer);
var
  I, J : Integer;
  P : PExtended;
begin
  I := L;
  J := R;
  P := FList[(L + R) shr 1];
  repeat
    while Extended(FList[I]^) < P^ do Inc(I);
    while Extended(FList[J]^) > P^ do Dec(J);
    if I <= J then
    begin
      FList.Exchange(I, J);
      Inc(I);
      Dec(J);
    end;
  until I > J;
  if L < J then QuickSort(L, J);
  if I < R then QuickSort(I, R);
end; { TFloatList.QuickSort }

procedure TFloatList.SetSorted(Value : Boolean);
begin
  if FSorted <> Value then
  begin
    if Value then Sort;
    FSorted := Value;
  end;
end; { TFloatList.SetSorted }

function TFloatList.GetCount : Integer;
begin
  Result := FList.Count;
end; { TFloatList.GetCount }

function TFloatList.Get(Index : Integer) : Extended;
begin
  if (Index < 0) or (Index >= GetCount) then Error;
  Result := Extended(FList[Index]^);
end; { TFloatList.Get }

procedure TFloatList.Put(Index : Integer; Value : Extended);
begin
  if (Index < 0) or (Index >= GetCount) then Error;
  if (Value < FMin) or (Value > FMax) then FloatListError(SFloatListValueError);
  Extended(FList[Index]^) := Value;
  if FSorted then Sort;
end; { TFloatList.Put }

function TFloatList.Add(Value : Extended) : Integer;
begin
  if FDuplicates = dupAccept then
  begin
    Result := GetCount;
    Insert(Result, Value);
  end
  else if Find(Value, Result) then
    case FDuplicates of
      dupIgnore : Result := GetCount;
      dupError  : FloatListError(SFloatDuplicatesError);
    end;
end; { TFloatList.Add }

function TFloatList.AddFloats(FloatList : TFloatList) : Integer;
begin
  for Result := 0 to FloatList.Count - 1 do
    Add(FloatList[Result]);
end; { TFloatList.AddFloats }

procedure TFloatList.Clear;
var
  I : Integer;
begin
  for I := 0 to FList.Count - 1 do
    FreeMem(FList[I], SizeOf(Extended));
  FList.Clear;
end; { TFloatList.Clear }

procedure TFloatList.Delete(Index : Integer);
begin
  if (Index < 0) or (Index >= GetCount) then Error;
  Dispose(PExtended(FList[Index]));
  FList.Delete(Index);
end; { TFloatList.Delete }

function TFloatList.Find(Value : Extended; var Index : Integer) : Boolean;
begin
  Index := 0;
  while (Index < GetCount) and (Get(Index) <> Value) do Inc(Index);
  Result := Index < GetCount;
end; { TFloatList.Find }

function TFloatList.Equals(FloatList : TFloatList) : Boolean;
var
  I : Integer;
begin
  Result := GetCount = FloatList.Count;
  if Result then
    for I := 0 to GetCount - 1 do
      if Get(I) <> FloatList[I] then
      begin
        Result := False;
        Break;
      end;
end; { TFloatList.Equals }

procedure TFloatList.Exchange(Index1, Index2 : Integer);
begin
  if (Index1 < 0) or (Index1 >= GetCount) or
    (Index2 < 0) or (Index2 >= GetCount) then Error;
  FList.Exchange(Index1, Index2);
  FSorted := False;
end; { TFloatList.Exchange }

function TFloatList.IndexOf(Value : Extended) : Integer;
begin
  while (Result < GetCount) and (Get(Result) <> Value) do Inc(Result);
  if Result = GetCount then Result := -1;
end; { TFloatList.IndexOf }

procedure TFloatList.Insert(Index : Integer; Value : Extended);
var
  P : PExtended;
begin
  if (Index < 0) or (Index > GetCount) then Error;
  if (Value < FMin) or (Value > FMax) then FloatListError(SFloatListValueError);
  New(P);
  P^ := Value;
  FList.Expand.Insert(Index, P);
  FSorted := False;
end; { TFloatList.Insert }

procedure TFloatList.Move(CurIndex, NewIndex : Integer);
begin
  if (CurIndex < 0) or (CurIndex >= GetCount) or
    (NewIndex < 0) or (NewIndex >= GetCount) then Error;
  FList.Move(CurIndex, NewIndex);
  FSorted := False;
end; { TFloatList.Move }

procedure TFloatList.Sort;
begin
  if not FSorted and (GetCount > 1) then
    QuickSort(0, GetCount - 1);
end; { TFloatList.Sort }

end.
